package cn.blankcat.dto.websocket;

import cn.blankcat.dto.audio.AudioAction;
import cn.blankcat.dto.channel.Channel;
import cn.blankcat.dto.guild.Guild;
import cn.blankcat.dto.member.Member;
import cn.blankcat.dto.message.Message;
import com.fasterxml.jackson.annotation.JsonValue;

import java.util.*;

public class WSEvent {

    // WSEventType 类型
    public enum WSEventType{
        EventGuildCreate("GUILD_CREATE"),
        EventGuildUpdate("GUILD_UPDATE"),
        EventGuildDelete("GUILD_DELETE"),
        EventChannelCreate("CHANNEL_CREATE"),
        EventChannelUpdate("CHANNEL_UPDATE"),
        EventChannelDelete("CHANNEL_DELETE"),
        EventGuildMemberAdd("GUILD_MEMBER_ADD"),
        EventGuildMemberUpdate("GUILD_MEMBER_UPDATE"),
        EventGuildMemberRemove("GUILD_MEMBER_REMOVE"),
        EventMessageCreate("MESSAGE_CREATE"),
        EventAtMessageCreate("AT_MESSAGE_CREATE"),
        EventDirectMessageCreate("DIRECT_MESSAGE_CREATE"),
        EventAudioStart("AUDIO_START"),
        EventAudioFinish("AUDIO_FINISH"),
        EventAudioOnMic("AUDIO_ON_MIC"),
        EventAudioOffMic("AUDIO_OFF_MIC"),
        // 这两类是官方放在type字段，但官方sdk没有提供，所以自己提供了
        TypeReady("READY"),
        TypeResumed("RESUMED"),
        // 有三类不是官方定义的type, 这里加上去方便自定义handler处理
        TypeHello("heartbeat_interval"),
        TypeHeartbeatAck("\"op\":11"),
        TypeInvalidSession("\"op\":9");


        private final String value;
        WSEventType(String value){
            this.value = value;
        }
        @JsonValue
        public String getValue(){
            return value;
        }

        // 根据value获取WSEventType
        public static WSEventType ofValue(String value){
            if (value == null || value.isEmpty()) {
                return null;
            }
            return Arrays.stream(WSEventType.values()).filter(wsEventType -> wsEventType.value.equals(value)).findFirst().orElse(null);
        }
    }

    public static Map<Session.Intent, WSEventType[]> intentEventMap;
    public static Map<WSEventType, Session.Intent> eventIntentMap;
    public static Map<WSEventType, Class<?>> eventClassMap;

    static {
        intentEventMap = new HashMap<>();
        eventIntentMap = new HashMap<>();
        eventClassMap = new HashMap<>();
        intentEventMap.put(Session.Intent.IntentGuilds, new WSEventType[]{WSEventType.EventGuildCreate, WSEventType.EventGuildUpdate, WSEventType.EventGuildDelete, WSEventType.EventChannelCreate, WSEventType.EventChannelUpdate, WSEventType.EventChannelDelete});
        intentEventMap.put(Session.Intent.IntentGuildMembers, new WSEventType[]{WSEventType.EventGuildMemberAdd, WSEventType.EventGuildMemberUpdate, WSEventType.EventGuildMemberRemove});
        intentEventMap.put(Session.Intent.IntentGuildMessages, new WSEventType[]{WSEventType.EventMessageCreate});
        intentEventMap.put(Session.Intent.IntentGuildAtMessage, new WSEventType[]{WSEventType.EventAtMessageCreate});
        intentEventMap.put(Session.Intent.IntentDirectMessages, new WSEventType[]{WSEventType.EventDirectMessageCreate});
        intentEventMap.put(Session.Intent.IntentAudio, new WSEventType[]{WSEventType.EventAudioStart, WSEventType.EventAudioFinish, WSEventType.EventAudioOnMic, WSEventType.EventAudioOffMic});
        eventIntentMap = transposeIntentEventMap(intentEventMap);
        eventClassMap.put(WSEventType.EventGuildCreate, Guild.class);
        eventClassMap.put(WSEventType.EventGuildUpdate, Guild.class);
        eventClassMap.put(WSEventType.EventGuildDelete, Guild.class);
        eventClassMap.put(WSEventType.EventChannelCreate, Channel.class);
        eventClassMap.put(WSEventType.EventChannelUpdate, Channel.class);
        eventClassMap.put(WSEventType.EventChannelDelete, Channel.class);
        eventClassMap.put(WSEventType.EventGuildMemberAdd, Member.class);
        eventClassMap.put(WSEventType.EventGuildMemberUpdate, Member.class);
        eventClassMap.put(WSEventType.EventGuildMemberRemove, Member.class);
        eventClassMap.put(WSEventType.EventMessageCreate, Message.class);
        eventClassMap.put(WSEventType.EventAtMessageCreate, Message.class);
        eventClassMap.put(WSEventType.EventDirectMessageCreate, Message.class);
        eventClassMap.put(WSEventType.EventAudioStart, AudioAction.class);
        eventClassMap.put(WSEventType.EventAudioFinish, AudioAction.class);
        eventClassMap.put(WSEventType.EventAudioOnMic, AudioAction.class);
        eventClassMap.put(WSEventType.EventAudioOffMic, AudioAction.class);

        eventClassMap.put(WSEventType.TypeReady, WSReadyData.class);
        eventClassMap.put(WSEventType.TypeResumed, WSPayload.class);
        eventClassMap.put(WSEventType.TypeHello, WSHelloData.class);
        eventClassMap.put(WSEventType.TypeInvalidSession, WSPayload.class);
        eventClassMap.put(WSEventType.TypeHeartbeatAck, WSPayload.class);
    }

    public static void init(){
        eventIntentMap = transposeIntentEventMap(intentEventMap);
    }

    public static Map<WSEventType, Session.Intent> transposeIntentEventMap(Map<Session.Intent, WSEventType[]> intentEventMap){
        Map<WSEventType, Session.Intent> result = new HashMap<>();
        for (Session.Intent intent : intentEventMap.keySet()) {
            for (WSEventType wsEventType : intentEventMap.get(intent)) {
                result.put(wsEventType, intent);
            }
        }
        return result;
    }

    public static int eventToIntent(WSEventType ...wsEventTypes){
        int i = 0;
        for (WSEventType wsEventType : wsEventTypes) {
            i = i | eventIntentMap.get(wsEventType).getValue();
        }
        return i;
    }

    public static int allEventToIntent(){
        return eventToIntent(WSEventType.EventAtMessageCreate
                , WSEventType.EventGuildCreate, WSEventType.EventGuildMemberAdd);
    }
}
