package cn.ac.caict.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.*;
import java.util.function.Function;


/**
 * <pre>
 *  常见转换示例
 *
 *           Client                                                      Server
 *       ======================================================================================
 *
 *          1. OutputStream -->\                                      /--> 2. InputStream -->
 *                               Socket <--> network <--> ServerSocket                       |
 *          4. InputStream  <--/                                      \<--3. OutputStream <--
 *
 * </pre>
 */
public class StreamTransfer {

    public static final int DEFAULT_BUFFER_SIZE = 4096;

    private static final Logger logger = LoggerFactory.getLogger(StreamTransfer.class);

    public static void transfer(@Nonnull InputStream inputStream, @Nonnull OutputStream outputStream) {
        try (BufferedInputStream bufferIn = new BufferedInputStream(inputStream);
             OutputStream out = outputStream) {
            int read;
            byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
            while ((read = bufferIn.read(buffer, 0, buffer.length)) != -1) {
                out.write(buffer, 0, read);
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
    }


    /**
     * @param inputStream 输入
     * @param decoder     转换处理
     * @param <T>         输出结果对象
     * @return 返回 T
     */
    public static <T> T decode(@Nonnull InputStream inputStream, Function<byte[], T> decoder) throws IOException {
        try (ByteArrayOutputStream byteArray = new ByteArrayOutputStream()) {
            transfer(inputStream, byteArray);
            return decoder.apply(byteArray.toByteArray());
        }
    }

}