package cn.ac.caict.util;

import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Base64;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.IntPredicate;

/**
 * 图片工具类
 */
public class Images {


    public enum ImageFormat {
        JPEG, PNG, GIF, BMP, DWG, TIF, TIFF, ICO
    }

    private static final Logger logger = LoggerFactory.getLogger(Images.class);

    private static final String DEFAULT_IMAGE_FORMAT_NAME = "png";

    public static String getBase64(InputStream inputStream) {
        return getBase64(inputStream, DEFAULT_IMAGE_FORMAT_NAME);
    }

    public static String getBase64(InputStream inputStream, String formatName) {
        String image = null;
        try (InputStream input = inputStream;
             ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
            BufferedImage bufferedImage = ImageIO.read(input);
            ImageIO.write(bufferedImage, formatName, outputStream);
            byte[] bytes = outputStream.toByteArray();
            image = new String(Base64.getMimeEncoder().encode(bytes));

        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return image;
    }


    public static int length(String base64) {
        return Strings.length(base64) / 3 * 4;
    }


    /**
     * 校验照片大小
     *
     * @param image     : 照片内容 (base64)
     * @param predicate ：判断照片大小
     * @return 是否符合照片大小
     */
    public static boolean checkLength(String image, IntPredicate predicate) {
        Objects.requireNonNull(image);
        return predicate.test(length(image));
    }

    /**
     * 校验照片格式
     *
     * @param image  : 照片内容
     * @param format : 需要的照片格式
     * @return 返回结果
     */
    public static boolean checkFormat(String image, ImageFormat format) {
        Objects.requireNonNull(image);
        return Objects.isNull(format) || ImageType.getType(image) == format;
    }


    public static class ImageType {
        private static final Map<String, ImageFormat> IMG_TYPE = new LinkedHashMap<>(); //[0]文件标识  [1]文件类型

        static {

            IMG_TYPE.put("FFD8FF", ImageFormat.JPEG);
            IMG_TYPE.put("47494638", ImageFormat.GIF);
            IMG_TYPE.put("89504E47", ImageFormat.PNG);
            IMG_TYPE.put("424D", ImageFormat.BMP);
            IMG_TYPE.put("41433130", ImageFormat.DWG);
            IMG_TYPE.put("4D4D", ImageFormat.TIF);
            IMG_TYPE.put("49492A00", ImageFormat.TIFF);
            IMG_TYPE.put("000001000", ImageFormat.ICO);
        }

        public static ImageFormat getType(String base64Image) {
            Objects.requireNonNull(base64Image);
            if (Strings.length(base64Image) > 24) {
                String headerBase64 = base64Image.substring(0, 24);
                byte[] header = Base64.getDecoder().decode(headerBase64);
                String fileHeaderKey = getFileHeader(header);

                for (Map.Entry<String, ImageFormat> image : IMG_TYPE.entrySet()) {
                    if (fileHeaderKey.toUpperCase().startsWith(image.getKey())) {
                        return image.getValue();
                    }
                }
            } else {
                throw new IllegalArgumentException("image length too short.");
            }
            return null;
        }



        private static String getFileHeader(byte[] bt) {
            byte[] byteHeader = new byte[32];
            if (bt != null) {
                System.arraycopy(bt, 0, byteHeader, 0, Math.min(bt.length, 32));
            }
            return Hex.toHexString(byteHeader);
        }
    }
}
