package cn.ac.caict.net.http.httpcomponents;


import cn.ac.caict.net.http.HttpClient;
import cn.ac.caict.net.http.HttpResponseEntity;
import cn.ac.caict.util.DateUtils;
import org.apache.hc.client5.http.fluent.Request;
import org.apache.hc.core5.http.Method;
import org.apache.hc.core5.http.NameValuePair;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.http.message.BasicNameValuePair;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;

public class DefaultHttpClient implements HttpClient {


    private static final Logger logger = LoggerFactory.getLogger(DefaultHttpClient.class);

    @Override
    public HttpResponseEntity get(String url, Map<String, String> headers, Charset charset, int connectTimeout, int socketTimeout) {
        return execute(
                Method.GET,
                Request::get,
                null,
                url, headers, charset,
                connectTimeout, socketTimeout
        );
    }

    @Override
    public HttpResponseEntity post(String url, String content, Map<String, String> headers, Charset charset, int connectTimeout, int socketTimeout) {
        return execute(
                Method.POST,
                Request::post,
                (request) -> {
                    if (content != null) {
                        StringEntity stringEntity = new StringEntity(content, charset);
                        request.body(stringEntity);
                    }
                },
                url, headers, charset,
                connectTimeout, socketTimeout
        );
    }

    @Override
    public HttpResponseEntity postForm(String url, Map<String, String> params,
                                   Map<String, String> headers, Charset charset,
                                   int connectTimeout, int socketTimeout) {
        return execute(
                Method.POST,
                Request::post,
                (request) -> {
                    if (params != null) {
                        List<NameValuePair> formParams = new ArrayList<>();
                        params.forEach((k, v) -> formParams.add(new BasicNameValuePair(k, v)));
                        request.bodyForm(formParams, charset);
                    }
                },
                url, headers, charset,
                connectTimeout, socketTimeout
        );
    }

    private HttpResponseEntity execute(
            Method md,
            Function<String, Request> method,
            Consumer<Request> setParams,
            String uri, Map<String, String> headers, Charset charset,
            int connectTimeout, int socketTimeout) {

        String uuid = DateUtils.now(DateUtils.TIME_MILLIS) ;

        logger.info(" ** caict client.invoke ** Method = {} , {} - uri : {} , connect_timeout = {} , socket_timeout = {} " ,md.name() ,uuid ,uri ,connectTimeout,socketTimeout);

        Objects.requireNonNull(charset);
        final String start = DateUtils.now(DateUtils.TIME_MILLIS);
        final Request request = method.apply(uri);
        if (null != headers) {
            headers.forEach(request::addHeader);
        }

        if (setParams != null) {
            setParams.accept(request);
        }
        HttpResponseEntity httpResponseEntity = null;

        try {
            request.connectTimeout(Timeout.of(connectTimeout, TimeUnit.MILLISECONDS)).responseTimeout(Timeout.of(socketTimeout, TimeUnit.MILLISECONDS));
            httpResponseEntity = request
                    .execute(DefaultExecutor.CLIENT)
                    .handleResponse(classicHttpResponse -> {
                        final HttpResponseEntity entity = new HttpResponseEntity();
                        entity.setStatus(classicHttpResponse.getCode());
                        if (classicHttpResponse.getEntity() != null) {

                            String str = EntityUtils.toString(classicHttpResponse.getEntity(), charset);
                            if (logger.isDebugEnabled()) {
                                logger.debug(" >>> response {}", str);
                            }
                            entity.setEntity(str);
                        }
                        return entity;
                    });
        } catch (Throwable e) {
            logger.error(e.getMessage(), e);
        }

        if (httpResponseEntity == null) {
            httpResponseEntity = new HttpResponseEntity();
        }


        final String end = DateUtils.now(DateUtils.TIME_MILLIS);
        httpResponseEntity.setRequestTime(start);
        httpResponseEntity.setResponseTime(end);
        httpResponseEntity.setRrMillis(DateUtils.betweenMillis(start, end));

        logger.info(" ** caict client.invoke ** Method = {} , {} - uri : {} , status = {} , millis = {} ",md.name() ,uuid ,uri ,httpResponseEntity.getStatus(),httpResponseEntity.getRrMillis());

        return httpResponseEntity;
    }


  /* public static void main(String[] args) {
        DefaultHttpClient dhc = new DefaultHttpClient();
        HttpResponseEntity hre = dhc.get("https://www.baidu.com", null,
                StandardCharsets.UTF_8, 1000, 2000
        );
        System.err.println(hre.getEntity());
    }*/
}
