package cn.ac.caict.entity;

import cn.ac.caict.codec.text.JsonCodec;
import com.google.common.base.Joiner;

import java.util.Objects;

import static cn.ac.caict.constants.SecureConstants.DataCryptoAlg.AES;
import static cn.ac.caict.constants.SecureConstants.DataCryptoAlg.SM4;
import static cn.ac.caict.constants.SecureConstants.KeyCryptoAlg.RSA;
import static cn.ac.caict.constants.SecureConstants.KeyCryptoAlg.SM2;


/**
 * 请求消息封装
 * 1、加解密配置
 * 2、请求消息体
 *
 * @param <T>
 */
public class ServicesRequestEntity<T> implements ServicesRequest<T> {

    /**
     * 每一个请求ID
     */
    protected String clientTraceId;

    /**
     * 请求服务接口账号
     * Access Key           : key
     * Access Key Secret    : 秘钥
     */
    protected String accessKey;


    /**
     * 非对称秘钥版本号
     * rsa : 用于cipher - key加密
     */
    protected String keyAlg;
    protected String keyVersion;

    /**
     * 加密用的key
     * 1、随机生成（sm4/ Aes-128）
     */
    protected String cipherKey;
    /**
     * 数据加密算法
     * AES / SM4
     */
    protected String dataAlg;

    /**
     * 精确到毫秒 [时间戳 2分钟内有效]
     * 1970+
     */
    protected long timestamp;

    /**
     * RsaWithSha256（access_key + clientTraceId +  timestamp + jsonData + access_secret）
     * SHA256withRSA
     * | - 私钥签名
     * | - 公钥签名
     */
    protected String sign;


    /**
     * Json -> (body);
     * 加密传输
     */
    protected String cipherData;

    protected T data;

    protected ServicesRequestEntity() {
    }


    /**
     * @param keyAlg        : 非对称秘钥加密算法 （rsa / sm2）
     * @param keyVersion    : 秘钥版本号
     * @param alg           : 数据加密算法 (Aes/sm4)
     * @param cipherKey     : 经过 rsa/sm2 加密后的数据
     * @param sign          ：签名 （access_key + clientTraceId +  timestamp + jsonData + access_secret）
     * @param clientTraceId ：流水号
     * @param accessKey     : 访问秘钥
     * @param timestamp     : 毫秒时间戳
     * @param cipherData    : 加密 -> json(obj)
     * @param dataObj
     */
    public ServicesRequestEntity(String keyAlg, String keyVersion, String alg, String cipherKey,
                                 String clientTraceId, String accessKey, long timestamp,
                                 String sign, String cipherData,
                                 T dataObj) {
        this.keyAlg = keyAlg;
        this.clientTraceId = clientTraceId;
        this.accessKey = accessKey;
        this.cipherKey = cipherKey;
        this.dataAlg = alg;
        this.keyVersion = keyVersion;
        this.timestamp = timestamp;
        this.sign = sign;
        this.cipherData = cipherData;
        this.data = dataObj;
    }


    public static DefaultBuilder secure() {
        return new DefaultBuilder();
    }


    /**
     * 只传输加密数据，原始数据不传递
     */
    public ServicesRequestEntity<T> nullPlainData() {
        this.data = null;
        return this;
    }


    /**
     * 安全算法相关
     */
    public interface CryptoBuilder<B extends CryptoBuilder<B>> {

        B cipherCodec(CaictEntityCipherCodec codec);

        B rsa();

        B sm2();

        B keyVersion(String version);

        B sm4();

        B aes();

        B publicKey(String otherPublicKeyPKCS8);

        B privateKey(String selfPrivateKeyPKCS8);


        /**
         * 无意义 - 表示接下来配置参数
         */
        B params();
    }


    /**
     * 构造参数
     */
    public interface ParamsBuilder extends CryptoBuilder<ParamsBuilder> {

        ParamsBuilder accessKey(String accessKey, String accessSecret);


        ParamsBuilder clientTraceId(String clientTraceId);

        /**
         * 当前毫秒数
         */
        ParamsBuilder timestamp(long timestamp);

        /**
         * 当前参数明文结构
         */

        <T> ServicesRequestEntity<T> data(T data);


        /**
         * 明文
         */
        <T> ServicesRequestEntity<T> plain();

    }


    public static class DefaultBuilder implements ParamsBuilder {

        //需要发送到服务端
        private String accessKey;

        private String clientTraceId;
        private String keyAlg;
        private String cipherKey;
        private String dataAlg;
        private String keyVersion;
        private long timestamp;
        private String sign;
        private String cipherData;

        // 数据加解密 - 不需要传递到服务端
        private String accessSecret;

        private String jsonData;


        //优先级高与单独配置
        private CaictEntityCipherCodec codec;

        private CaictEntityCipherCodec.CaictEntityCipherCodecBuilder codecBuilder;


        public DefaultBuilder() {
            codecBuilder = new CaictEntityCipherCodec.CaictEntityCipherCodecBuilder(RSA);
        }


        @Override
        public ParamsBuilder cipherCodec(CaictEntityCipherCodec codec) {
            this.codec = codec;
            this.dataAlg = codec.getDataAlg();
            this.keyAlg = codec.getKeyAlg();
            return this;
        }

        @Override
        public ParamsBuilder rsa() {
            this.keyAlg = RSA;
            codecBuilder.rsa();
            return this;
        }

        @Override
        public ParamsBuilder sm2() {
            this.keyAlg = SM2;
            codecBuilder.sm2();
            return this;
        }

        @Override
        public ParamsBuilder keyVersion(String version) {
            this.keyVersion = version;
            return this;
        }

        @Override
        public ParamsBuilder sm4() {
            this.dataAlg = SM4;
            codecBuilder.sm4();
            return this;
        }

        @Override
        public ParamsBuilder aes() {
            this.dataAlg = AES;
            codecBuilder.aes();
            return this;
        }

        @Override
        public ParamsBuilder publicKey(String otherPublicKeyPKCS8) {
            codecBuilder.publicKey(otherPublicKeyPKCS8);
            return this;
        }

        @Override
        public ParamsBuilder privateKey(String selfPrivateKeyPKCS8) {
            codecBuilder.privateKey(selfPrivateKeyPKCS8);
            return this;
        }


        @Override
        public ParamsBuilder params() {

            if (codec == null && codecBuilder != null) {
                codec = codecBuilder.build();
            }
            return this;
        }


        @Override
        public ParamsBuilder accessKey(String accessKey, String accessSecret) {
            this.accessKey = accessKey;
            this.accessSecret = accessSecret;
            return this;
        }

        @Override
        public ParamsBuilder clientTraceId(String clientTraceId) {
            this.clientTraceId = clientTraceId;
            return this;
        }

        @Override
        public ParamsBuilder timestamp(long timestamp) {
            this.timestamp = timestamp;
            return this;
        }


        @Override
        public <T> ServicesRequestEntity<T> data(T data) {
            return this.build(data);
        }


        @Override
        public <T> ServicesRequestEntity<T> plain() {
            return null;
        }

        public <T> ServicesRequestEntity<T> build(T data) {
            T dataObj = data;
            if (codec != null) {
                // 需要加密，进行加密处理
                if(Objects.nonNull(data)){
                    if(data instanceof String){
                        this.jsonData = (String) data;
                    }else{
                        this.jsonData = JsonCodec.toJson(data);
                    }
                }
                 signAndEncrypt();
            }


            return new ServicesRequestEntity(
                    this.keyAlg, this.keyVersion, this.dataAlg, this.cipherKey,
                    this.clientTraceId, this.accessKey, this.timestamp,
                    this.sign, this.cipherData,
                    dataObj
            );
        }

        /**
         * 对称算法 ： 对数据加密
         *   算法 [SM4,AES]
         * 非对称算法 ：对密钥(对称密钥产生的随机密钥)加解密
         *   算法 [SM2,RSA]
         * 步骤:
         *  0、要求: byte[] 字节序: Big-Endian  (Java不需要处理)
         *
         *  1、【随机密钥生成..】 随机生成对称密钥
         *  2、【签名........】 自己私钥 对数据签名  Hex(byte[])
         *  3、【数据加密....】  使用随机密钥,对数据进行加密 Base64(byte[])
         *  4、【随机密钥加密.】 使用（服务方）公钥对对称密钥加密
         */
        private void signAndEncrypt() {
            // 步骤-1
            // 随机生成默认位数的Key (AES/SM4)
            // 密钥长度: 128
            byte[] key = codec.randomKey();

            // 步骤-2                             - **自己的私钥，进行签名处理**
            // 拼装签名数据
            String signData = Joiner.on("").useForNull("").join(
                    this.accessKey, this.clientTraceId, this.timestamp, this.jsonData, this.accessSecret
            ).toLowerCase();
            // 签名处理 - 使用私钥进行签名 签名算法返回 byte[]
            // RSA : SHA256WithRSA
            // SM2 : SM3WITHSM2
            // 签名信息16进制处理 hex（byte[]）
            this.sign = codec.sign(signData);

            // 步骤-3                            - **随机密钥对数据加密**
            // 对数据进行加密 处理 AES/SM4
            // 密钥长度 128位 - 每次一个密钥
            this.cipherData = codec.encrypt(this.jsonData, key);

            // 步骤-4
            // 使用公钥算法对随机密钥加密 (RSA/SM2)  - **服务端公钥**
            // RSA : RSA/ECB/PKCS1PADDING
            // SM2 : sm2p256v1
            this.cipherKey = codec.encryptRandomKey(key);
        }


    }

    @Override
    public String getClientTraceId() {
        return clientTraceId;
    }

    @Override
    public String getAccessKey() {
        return accessKey;
    }

    @Override
    public String getKeyAlg() {
        return keyAlg;
    }

    @Override
    public String getKeyVersion() {
        return keyVersion;
    }

    @Override
    public String getCipherKey() {
        return cipherKey;
    }

    @Override
    public String getDataAlg() {
        return dataAlg;
    }

    @Override
    public long getTimestamp() {
        return timestamp;
    }

    @Override
    public String getSign() {
        return sign;
    }

    @Override
    public String getCipherData() {
        return cipherData;
    }

    @Override
    public T getData() {
        return data;
    }
}
