package cn.ac.caict.codec.crypto.symmetric;


import cn.ac.caict.codec.crypto.CipherFactory;
import cn.ac.caict.exception.CaictCryptoException;

import javax.crypto.*;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.*;

/**
 * 对称秘钥加解密处理
 *
 * @author pony
 */
public interface SymmetricCodec {

    /**
     * 加密算法
     */
    String alg();

    /**
     * key : 算法
     */
    String keyAlg();

    /**
     * cipher provider
     */
    String provider();


    byte[] randomKey();

    /**
     * 生成一个随机 key
     *
     * @param length 长度
     * @return byte[]
     * @throws NoSuchProviderException  异常
     * @throws NoSuchAlgorithmException 异常
     */
    default byte[] randomKey(int length) throws NoSuchProviderException, NoSuchAlgorithmException {
        KeyGenerator keyGenerator = KeyGenerator.getInstance(keyAlg(), provider());
        keyGenerator.init(length);
        SecretKey secretKey = keyGenerator.generateKey();
        return secretKey.getEncoded();
    }


    /**
     * byte[] 转换为 Key
     */
    default Key toKey(byte[] key) {
        return new SecretKeySpec(key, keyAlg());
    }


    /**
     * 加密&解密处理
     *
     * @param data        数据
     * @param key         key
     * @param iv          向量(cbc模式必须添加)
     * @param encryptMode 加密 - 解密
     * @return 加解密结果
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    default byte[] encrypt(byte[] data, Key key, byte[] iv, int encryptMode)
            throws BadPaddingException, IllegalBlockSizeException{
        Cipher cipher = CipherFactory.getInstance(provider(), alg(), encryptMode, key, iv);
        return cipher.doFinal(data);
    }


    /**
     * 加密
     */
    default byte[] encode(byte[] data, byte[] keyByte, byte[] iv) throws BadPaddingException, IllegalBlockSizeException {
        Key key = toKey(keyByte);
        return encrypt(data, key, iv, Cipher.ENCRYPT_MODE);
    }

    /**
     * 解密
     */
    default byte[] decode(byte[] data, byte[] keyByte, byte[] iv) throws BadPaddingException, IllegalBlockSizeException {
        Key key = toKey(keyByte);
        return encrypt(data, key, iv, Cipher.DECRYPT_MODE);
    }


}
