package cn.ac.caict.codec.crypto.asymmetric.sm2;

import cn.ac.caict.codec.crypto.DefaultProviderFactory;
import cn.ac.caict.codec.crypto.asymmetric.AsymmetricCodec;
import cn.ac.caict.codec.crypto.asymmetric.AsymmetricKeyCodec;
import cn.ac.caict.codec.crypto.asymmetric.DefaultAsymmetricCodec;
import cn.ac.caict.codec.crypto.asymmetric.SignatureCodec;

import javax.crypto.Cipher;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECGenParameterSpec;


/**
 * SM2
 * @author pony
 */
public class SM2Codec extends DefaultAsymmetricCodec implements AsymmetricCodec, AsymmetricKeyCodec, SignatureCodec {


    private static final String KEY_ALG = "EC";
    private static final String ALG = "SM2";
    private static final String SIGN_ALG = "SM3WITHSM2";

    private static int DEFAULT_KEY_SIZE = 256;

    private static final String sm2p256v1 = "sm2p256v1";
    private static final String wapip192v1 = "wapip192v1";


    private AlgorithmParameterSpec algorithmParameterSpec;

    public static SM2Codec noAlgorithmParameterSpec() {
        SM2Codec sm2Codec = new SM2Codec(ALG, KEY_ALG, SIGN_ALG);
        return sm2Codec;
    }


    public static SM2Codec wapip192v1() {
        //GMNamedCurves.getByName()
        SM2Codec sm2Codec = new SM2Codec(ALG, KEY_ALG, SIGN_ALG);
        sm2Codec.algorithmParameterSpec = new ECGenParameterSpec(wapip192v1);
        return sm2Codec;
    }

    public static SM2Codec sm2p256v1() {
        //GMNamedCurves.getByName()
        SM2Codec sm2Codec = new SM2Codec(ALG, KEY_ALG, SIGN_ALG);
        sm2Codec.algorithmParameterSpec = new ECGenParameterSpec(sm2p256v1);
        return sm2Codec;
    }

    public SM2Codec() {
        this(ALG, KEY_ALG, SIGN_ALG);
        this.algorithmParameterSpec = new ECGenParameterSpec(sm2p256v1);
    }

    public SM2Codec(String alg, String keyAlg, String signAlg) {
        this(alg, keyAlg, signAlg, DefaultProviderFactory.getDefaultProvider());
    }

    public SM2Codec(String alg, String keyAlg, String signAlg, String provider) {
        super(alg, keyAlg, signAlg, provider);
    }

    @Override
    public byte[] encrypt(byte[] data, PublicKey publicKey) throws Exception {
        Cipher cipher = Cipher.getInstance(alg(), provider());
        if (algorithmParameterSpec != null) {
            cipher.init(Cipher.ENCRYPT_MODE, publicKey, algorithmParameterSpec);
        } else {
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
        }
        return cipher.doFinal(data);
    }

    @Override
    public byte[] decrypt(byte[] data, PrivateKey privateKey) throws Exception {
        Cipher cipher = Cipher.getInstance(alg(), provider());
        if (algorithmParameterSpec != null) {
            cipher.init(Cipher.DECRYPT_MODE, privateKey, algorithmParameterSpec);
        } else {
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
        }
        return cipher.doFinal(data);
    }

    @Override
    public AsymmetricKeyCodec keyCodec() {
        return this;
    }

    @Override
    public SignatureCodec signatureCodec() {
        return this;
    }


    @Override
    public int defaultKeySize() {
        return DEFAULT_KEY_SIZE;
    }
}
