package cn.ac.caict.codec.crypto.asymmetric.rsa;

import cn.ac.caict.codec.crypto.DefaultProviderFactory;
import cn.ac.caict.codec.crypto.asymmetric.AsymmetricCodec;
import cn.ac.caict.codec.crypto.asymmetric.AsymmetricKeyCodec;
import cn.ac.caict.codec.crypto.asymmetric.DefaultAsymmetricCodec;
import cn.ac.caict.codec.crypto.asymmetric.SignatureCodec;

import javax.crypto.Cipher;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.security.PrivateKey;
import java.security.PublicKey;


/**
 * @author pony
 * @see <a href="https://tools.ietf.org/html/rfc3447"/>
 * key - block
 * <p>
 * <p>
 * The padding margin is as follows:
 * <p>
 * RSA/ECB/PKCS1Padding, 11
 * RSA/ECB/NoPadding, 0
 * RSA/ECB/OAEPPadding, 42 // Actually it's OAEPWithSHA1AndMGF1Padding
 * RSA/ECB/OAEPWithMD5AndMGF1Padding, 34
 * RSA/ECB/OAEPWithSHA1AndMGF1Padding, 42
 * RSA/ECB/OAEPWithSHA224AndMGF1Padding, 58
 * RSA/ECB/OAEPWithSHA256AndMGF1Padding, 66
 * RSA/ECB/OAEPWithSHA384AndMGF1Padding, 98
 * RSA/ECB/OAEPWithSHA512AndMGF1Padding, 130
 * RSA/ECB/OAEPWithSHA3-224AndMGF1Padding, 58
 * RSA/ECB/OAEPWithSHA3-256AndMGF1Padding, 66
 * RSA/ECB/OAEPWithSHA3-384AndMGF1Padding, 98
 * RSA/ECB/OAEPWithSHA3-512AndMGF1Padding, 130
 */
public class RSACodec extends DefaultAsymmetricCodec implements AsymmetricCodec, AsymmetricKeyCodec, SignatureCodec {

    private static final String KEY_ALG = "RSA";
    private static final String ALG = "RSA/ECB/PKCS1PADDING";
    private static final String SIGN_ALG = "SHA256WithRSA";

    private static int DEFAULT_KEY_SIZE = 2048;

    public RSACodec() {
        this(ALG, KEY_ALG, SIGN_ALG);
    }

    public RSACodec(String alg, String keyAlg, String signAlg) {
        this(alg, keyAlg, signAlg, DefaultProviderFactory.getDefaultProvider());
    }

    public RSACodec(String alg, String keyAlg, String signAlg, String provider) {
        super(alg, keyAlg, signAlg, provider);
    }


    @Override
    public byte[] encrypt(byte[] data, PublicKey publicKey) throws Exception {
        Cipher cipher = Cipher.getInstance(alg(), provider());
        cipher.init(Cipher.ENCRYPT_MODE, publicKey);

        int blockSize = cipher.getBlockSize();
        int inputLen = data.length;
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();) {
            int offSet = 0;
            byte[] buf;
            int i = 0;
            while (inputLen - offSet > 0) {
                if (inputLen - offSet > blockSize) {
                    buf = cipher.doFinal(data, offSet, blockSize);
                } else {
                    buf = cipher.doFinal(data, offSet, inputLen - offSet);
                }
                outputStream.write(buf, 0, buf.length);
                i++;
                offSet = i * blockSize;
            }
            return outputStream.toByteArray();
        }
    }

    @Override
    public byte[] decrypt(byte[] data, PrivateKey privateKey) throws Exception {
        Cipher cipher = Cipher.getInstance(alg(), provider());
        cipher.init(Cipher.DECRYPT_MODE, privateKey);
        int blockSize = cipher.getBlockSize();
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(data);
             ByteArrayOutputStream outputStream = new ByteArrayOutputStream();) {
            byte[] buf = new byte[blockSize];           //解密数据
            int len;
            while ((len = inputStream.read(buf)) > 0) {
                outputStream.write(cipher.doFinal(buf, 0, len));
            }

            return outputStream.toByteArray();
        }

    }


    @Override
    public AsymmetricKeyCodec keyCodec() {
        return this;
    }

    @Override
    public SignatureCodec signatureCodec() {
        return this;
    }

    @Override
    public int defaultKeySize() {
        return DEFAULT_KEY_SIZE;
    }
}
