package cn.ac.caict.codec.crypto.asymmetric;

import javax.crypto.Cipher;
import javax.crypto.EncryptedPrivateKeyInfo;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import java.io.IOException;
import java.security.*;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;


/**
 * java
 * 生成的私钥是pkcs8格式
 * 公钥是x.509格式
 * @author pony
 */
public interface AsymmetricKeyCodec extends AsymmetricCodec {

    int defaultKeySize();

    default AlgorithmParameterSpec getAlgorithmParameterSpec(){
        return null;
    }


    default KeyPair keyPair() throws NoSuchProviderException, NoSuchAlgorithmException {

        return keyPair(defaultKeySize());
    }

    default KeyPair keyPair(int keySize) throws NoSuchProviderException, NoSuchAlgorithmException {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(keyAlg(), provider());
        keyPairGenerator.initialize(keySize);
        if(getAlgorithmParameterSpec()!=null){
            try {
                keyPairGenerator.initialize(getAlgorithmParameterSpec());
            } catch (InvalidAlgorithmParameterException e) {
                e.printStackTrace();
            }
        }

        return keyPairGenerator.generateKeyPair();
    }


    /**
     * keyPair : 获取公钥
     *
     * @param keyPair
     */
    default byte[] getPublicKey(KeyPair keyPair) {
        PublicKey publicKey = keyPair.getPublic();
        return publicKey.getEncoded();
    }


    /**
     * keyPair : 获取私钥
     *
     * @param keyPair
     */
    default byte[] getPrivateKey(KeyPair keyPair) {
        PrivateKey privateKey = keyPair.getPrivate();
        return privateKey.getEncoded();
    }

    /**
     * 获取x509格式公钥
     */
    default PublicKey getX509PublicKey(byte[] publicKey) throws NoSuchProviderException, NoSuchAlgorithmException, InvalidKeySpecException {
        X509EncodedKeySpec x509spec = new X509EncodedKeySpec(publicKey);
        return KeyFactory
                .getInstance(keyAlg(), provider())
                .generatePublic(x509spec);
    }


    /**
     * 获取pkcs#8格式 私钥 - 私钥没有密码
     */
    default PrivateKey getPrivateKeyFromPKCS8(byte[] privateKey) throws NoSuchProviderException, NoSuchAlgorithmException, InvalidKeySpecException {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(privateKey);
        return KeyFactory
                .getInstance(keyAlg(), provider())
                .generatePrivate(pkcs8KeySpec);
    }


    /**
     * 获取pkcs#8格式 私钥 - 私钥有密码
     */
    default PrivateKey getPrivateKeyFromPKCS8(byte[] privateKey, String keyPass) throws Exception {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(privateKey);
        return getPrivateKeyFromPKCS8(privateKey, keyPass.toCharArray());
    }


    /**
     * 根据带有密码的私钥进行解密
     *
     * @param privateKey  ： 带有密码的私钥
     * @param keyPassword : 私钥密码
     * @return ： 生成pkcs8格式的私钥
     */
    default PrivateKey getPrivateKeyFromPKCS8(byte[] privateKey, char[] keyPassword) throws Exception {
        EncryptedPrivateKeyInfo epkInfo = new EncryptedPrivateKeyInfo(privateKey);
        SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(epkInfo.getAlgName(), provider());
        PBEKeySpec pbeKeySpec = new PBEKeySpec(keyPassword);
        SecretKey pbeKey = keyFactory.generateSecret(pbeKeySpec);
        Cipher cipher = Cipher.getInstance(epkInfo.getAlgName(), provider());
        cipher.init(Cipher.DECRYPT_MODE, pbeKey, epkInfo.getAlgParameters());
        PKCS8EncodedKeySpec pkcs8EncodedKeySpec = epkInfo.getKeySpec(cipher);
        return KeyFactory.getInstance(keyAlg(), provider())
                .generatePrivate(pkcs8EncodedKeySpec);
    }



    default byte[] getX509FromPublicKey(PublicKey x509Key) throws NoSuchProviderException, NoSuchAlgorithmException, InvalidKeySpecException {
        return x509Key.getEncoded();
    }
    default byte[] getPKCS8FromPrivateKey(PrivateKey privateKey) throws IOException {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(privateKey.getEncoded());
        return pkcs8KeySpec.getEncoded();
    }

}
