package cn.ac.caict;

import cn.ac.caict.codec.text.JsonCodec;
import cn.ac.caict.entity.CaictEntityCipherCodec;
import cn.ac.caict.entity.ServicesRequestEntity;
import cn.ac.caict.entity.ServicesResponseEntity;
import cn.ac.caict.net.http.HttpClient;
import cn.ac.caict.net.http.HttpClientFactory;
import cn.ac.caict.net.http.HttpResponseEntity;
import com.google.common.base.Joiner;
import org.apache.hc.core5.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 调用服务客户端
 * sm2 + sm4描述加解密国产
 * <pre>
 *  encrypt:
 *     1、请求信息，Json.toJson(obj) > data
 *     2、sm4随机生成秘钥      randomKey
 *     3、数据加密处理   base64(sm4(data))
 *     4、SM3withSM2对数据签名处理
 *     6、sm2对sm4随机秘钥进行加密处理
 *
 * </pre>
 *
 *
 *
 * @param <T> 请求数据 Object
 */
public class CaictClient<T> {


    private static final Logger logger = LoggerFactory.getLogger(CaictClient.class);


    private final String keyVersion;

    private final String accessKey;                     // 接口账号
    private final String accessSecret;                  // 签名key

    private final CaictEntityCipherCodec codec;

    private final HttpClient httpClient;

    private static final String CONTENT_TYPE_KEY = "Content-type";
    private static final String CONTENT_TYPE_VALUE = "application/json; charset=utf-8";

    /**
     * 生产 rsa + aes 组合的请求客户端
     *
     * @param accessKey    调用接口账号
     * @param accessSecret 调用接口账号签名key
     * @param selfPriKey   私钥 - 生成的私钥
     * @param serverPubKey 公钥 - 服务端公钥
     * @return 返回rsa加密国产客户端
     */
    public static <T> CaictClient<T> client0(String accessKey, String accessSecret, String keyVersion, String selfPriKey, String serverPubKey) {
        return new CaictClient<>(KeyAlg.RSA, DataAlg.AES, accessKey, accessSecret, keyVersion, selfPriKey, serverPubKey);
    }

    /**
     * 产生SM2 + SM4 组合的公钥
     */
    public static <T> CaictClient<T> client1(String accessKey, String accessSecret, String keyVersion, String selfPriKey, String serverPubKey) {
        return new CaictClient<>(KeyAlg.SM2, DataAlg.SM4, accessKey, accessSecret, keyVersion, selfPriKey, serverPubKey);
    }

    public CaictClient(KeyAlg keyAlg, DataAlg dataAlg, String accessKey, String accessSecret, String keyVersion, String selfPriKey, String serverPubKey) {
        // key加密算法 rsa / sm2

        this.accessKey = accessKey;
        this.accessSecret = accessSecret;
        // pkcs8 格式私钥  - 自身的key
        // x509  格式公钥  - 服务端的key
        this.keyVersion = keyVersion;
        this.codec = CaictEntityCipherCodec
                .keyAlg(keyAlg.getName())
                    .privateKey(selfPriKey)
                    .publicKey(serverPubKey)
                .dataAlg(dataAlg.getName())
                .build();
        this.httpClient = HttpClientFactory.createDefault();

    }

    /**
     * 请求服务
     *
     * @param url            url地址
     * @param connectTimeout 链接超时时间
     * @param socketTime     ReadTime
     * @param request        请求信息
     * @return 返回消息
     */
    public CaictResponse invoke(String url, int connectTimeout, int socketTime, String clientTraceId, T request) {
         ServicesRequestEntity<T> servicesRequest = ServicesRequestEntity
                        .secure()
                            .cipherCodec(this.codec)    //加解密处理
                            .keyVersion(this.keyVersion)
                        .params()
                            .accessKey(accessKey, accessSecret)
                            .clientTraceId(clientTraceId)
                            .timestamp(System.currentTimeMillis())
                        .data(request)
                        .nullPlainData();
        final String json = JsonCodec.toJson(servicesRequest);
        final Map<String, String> header = new HashMap<>();
        header.put(CONTENT_TYPE_KEY, CONTENT_TYPE_VALUE);
        // http invoke ...
        final HttpResponseEntity entity = httpClient.post(url, json, header, StandardCharsets.UTF_8, connectTimeout, socketTime);

        final CaictResponse response = new CaictResponse(entity);
        if(entity.getStatus() == HttpStatus.SC_OK && Objects.nonNull(entity.getEntity())){
            doCipherAndVerifySign(response);
        }

        return response;
    }


    protected void doCipherAndVerifySign(CaictResponse response){
        HttpResponseEntity httpEntity = response.getResponseEntity();

        if(Objects.nonNull(httpEntity.getEntity())){
            ServicesResponseEntity result = JsonCodec.readValue(httpEntity.getEntity(),ServicesResponseEntity.class);

            if(Objects.nonNull(result.getCipherData())){
                byte[] key = codec.decryptRandomKey(result.getCipherKey());
                String plainText = codec.decrypt(result.getCipherData(),key);

                String signData = Joiner.on("").useForNull("").join(
                        result.getServerTraceId(), result.getClientTraceId(), plainText
                ).toLowerCase();

                boolean verify = codec.verifySign(result.getSign(),
                        signData,
                        StandardCharsets.UTF_8
                );

                logger.info(" ** xty.client ** 解析响应结果 验签数据 >> {} " , signData);
                logger.info(" ** xty.client ** 解析响应结果 验签结果 >> {} " , verify);
                logger.info(" ** xty.client ** 解析响应结果 服务结果 >> {} " , plainText);

                //result.setCipherData(null);

                // 设置解密后的明文信息
                result.setPlainData(plainText);
            }
            response.setResult(result);
        }

    }


}
