package cn.bestwu.apidoc.starter

import cn.bestwu.generator.dsl.Generators
import cn.bestwu.logging.RequestLogging
import cn.bestwu.logging.RequestLoggingHandler
import com.beust.klaxon.JsonArray
import com.beust.klaxon.JsonObject
import com.beust.klaxon.Parser
import com.fasterxml.jackson.databind.DeserializationFeature
import com.fasterxml.jackson.databind.ObjectMapper
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.method.HandlerMethod
import java.io.File

/**
 * 请求日志过滤器
 *
 * @author Peter Wu
 * @since 1.2.24
 */
class ApiDocHandler(private var generatorProperties: GeneratorProperties, private var apidocProperties: ApidocProperties) : RequestLoggingHandler {

    private val parser = Parser()
    private var objectMapper: ObjectMapper = ObjectMapper()

    init {
        objectMapper.enable(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS)
        objectMapper.enable(DeserializationFeature.USE_BIG_INTEGER_FOR_INTS)
    }

    @Synchronized
    override fun handle(logging: RequestLogging, handler: HandlerMethod?) {
        if (ApiDoc.enable || ApiDoc.tableNames.isNotEmpty()) {
            try {
                println("生成文档相关数据")

                var httpMethod: String? = ""
                if (handler != null) {
                    val methodAnnotation = handler.getMethodAnnotation(RequestMapping::class.java)
                    val requestMethods = methodAnnotation?.method ?: arrayOf()
                    requestMethods.forEachIndexed { i, requestMethod ->
                        httpMethod += requestMethod
                        if (i < requestMethods.size - 1) {
                            httpMethod += ","
                        }
                    }
                }
                if (httpMethod == "")
                    httpMethod = logging.requestMethod

                var resource = logging.resourceType
                var name = logging.resourceName
                name = if (ApiDoc.name.isNotBlank()) ApiDoc.name else name
                name = name?.replace("[(\\[\\])]".toRegex(), "")

                if (resource.isNullOrBlank() && name != null) {
                    resource = name
                }
                if (ApiDoc.resource.isNotBlank()) {
                    resource = ApiDoc.resource
                }
                if (resource != null) {
                    resource = resource.replace("[(\\[\\])]".toRegex(), "")
                }

                val url = logging.resourceUri ?: logging.servletPath
                val uriVariables = logging.uriVariables

                //生成相应数据

                val baseDir = apidocProperties.sourceFile
                baseDir.mkdirs()
                fixFile(baseDir, "code.json")
                fixFile(baseDir, "datastructure.json")
                val commonField = fixFile(baseDir, "field.json")

                val subDir = File(baseDir, ApiDoc.docModule)
                subDir.mkdirs()
                val trees: JsonArray<JsonObject>
                val apis: JsonArray<JsonObject>

                //api
                val apifile = File(subDir, "api.json")
                @Suppress("UNCHECKED_CAST")
                apis = if (apifile.exists())
                    parser.parse(apifile.inputStream()) as JsonArray<JsonObject>
                else
                    JsonArray()

                val api = JsonObject()
                api["method"] = httpMethod
                api["name"] = name
                val headers = mutableMapOf<String, Collection<String>>()
                val requestHeaders = logging.requestHeaders
                if (requestHeaders != null)
                    (mutableListOf<String>() + ApiDoc.headers + ApiDoc.headerRequires).forEach {
                        val key = if (ApiDoc.headerRequires.contains(it)) "$it&" else "$it^"
                        headers[key] = requestHeaders.getOrDefault(it, requestHeaders.getOrDefault(it.toLowerCase(), listOf("")))
                    }
                api["headers"] = headers

                val requires = ApiDoc.requires
                val requiresMap = mutableMapOf<String, Array<String>>()
                requires.forEach {
                    requiresMap[it] = arrayOf("")
                }
                val parameterMap = logging.parameters
                val params = JsonObject()
                (requiresMap + (parameterMap
                        ?: mutableMapOf())).forEach { (k, v) ->
                    val key = if (requires.contains(k)) "$k&" else "$k^"
                    params[key] = if (ApiDoc.fileNames.contains(k)) null else JsonArray(*v)
                }
                api["params"] = params
                api["resource"] = resource
                api["url"] = url
                api["desc"] = ApiDoc.desc
                api["version"] = apidocProperties.version.toList()
                api["uriVariables"] = uriVariables

                val responseObj = logging.responseBody

                api["results"] = responseObj

                apis.remove(apis.find {
                    it["name"] == api["name"] && it["resource"] == api["resource"]
                })
                apis.add(api)

                println("${if (apifile.exists()) "更新" else "创建"}$apifile")
                apifile.writeText(apis.toJsonString(true))

                //tree
                val treefile = File(subDir, "tree.json")
                trees = if (treefile.exists()) {
                    @Suppress("UNCHECKED_CAST")
                    parser.parse(treefile.inputStream()) as JsonArray<JsonObject>
                } else {
                    JsonArray()
                }
                var tree = trees.find { it["text"] == resource }
                if (tree == null) {
                    tree = JsonObject()
                    tree["text"] = resource
                    tree["expanded"] = true
                    tree["children"] = JsonArray<JsonObject>()
                    trees.add(tree)
                }
                @Suppress("UNCHECKED_CAST")
                val children = tree["children"] as JsonArray<JsonObject>
                var child = children.find { it["text"] == name }
                if (child == null) {
                    child = JsonObject()
                    child["leaf"] = true
                    child["text"] = name
                    children.add(child)
                }

                println("${if (treefile.exists()) "更新" else "创建"}$treefile")
                treefile.writeText(trees.toJsonString(true))

                //field
                val fieldFile = File(subDir, "field/${if (resource.isNullOrBlank()) "${if (ApiDoc.tableNames.isNotEmpty()) ApiDoc.tableNames[0] else "field"}.json" else "$resource.json"}")
                if (ApiDoc.tableNames.isNotEmpty()) {
                    generatorProperties.replaceAll = true
                    generatorProperties.basePath = subDir
                    generatorProperties.dir = "field"
                    generatorProperties.generators = arrayOf(Generators.field.apply {
                        if (!resource.isNullOrBlank())
                            fileName = "$resource.json"
                    })
                    generatorProperties.tableNames = ApiDoc.tableNames
                    generatorProperties.datasource.schema = ApiDoc.schema

                    Generators.call(generatorProperties)
                }
                //field
                val fields = if (fieldFile.exists()) {
                    @Suppress("UNCHECKED_CAST")
                    parser.parse(fieldFile.inputStream()) as JsonArray<JsonObject>
                } else {
                    JsonArray()
                }
                @Suppress("UNCHECKED_CAST")
                val commonFields = parser.parse(commonField.inputStream()) as JsonArray<JsonObject>

                if (uriVariables != null)
                    fillField(uriVariables, fields, commonFields)
                fillField(headers, fields, commonFields)
                fillField(params.map, fields, commonFields)
                @Suppress("UNCHECKED_CAST")
                if (responseObj is Map<*, *>)
                    fillField(responseObj as Map<String, *>, fields, commonFields)

                fieldFile.parentFile.mkdirs()
                println("${if (fieldFile.exists()) "更新" else "创建"}$fieldFile")
                fieldFile.writeText(fields.toJsonString(true))
            } finally {
                ApiDoc.enable = false
                ApiDoc.tableNames = arrayOf()
            }
        }
    }

    private fun fillField(params: Map<String, *>, fields: JsonArray<JsonObject>, commonFields: JsonArray<JsonObject>) {
        params.forEach { k, v ->
            val key = if (k.endsWith("&") || k.endsWith("^")) k.substring(0, k.length - 1) else k
            fillField(commonFields, key, fields)
            if (v is Map<*, *>) {
                @Suppress("UNCHECKED_CAST")
                fillField(v as Map<String, *>, fields, commonFields)
            } else if (v is List<*> && v.size > 0 && v[0] is Map<*, *>) {
                @Suppress("UNCHECKED_CAST")
                fillField(v[0] as Map<String, *>, fields, commonFields)
            }
        }
    }

    private fun fillField(commonFields: JsonArray<JsonObject>, key: String, fields: JsonArray<JsonObject>) {
        if (!commonFields.any { f -> f["name"] == key } && !fields.any { f -> f["name"] == key }) {
            if (key.contains(".")) {
                val find = fields.find { it["name"] == key.substring(key.lastIndexOf(".") + 1) }
                if (find != null) {
                    val field = mutableMapOf(
                            "desc" to find["desc"],
                            "name" to key,
                            "nullable" to find["nullable"],
                            "type" to find["type"],
                            "value" to find["value"]
                    )
                    fields.add(JsonObject(field))
                    return
                }
            }
            val field = mutableMapOf(
                    "desc" to "",
                    "name" to key,
                    "nullable" to true,
                    "type" to "String",
                    "value" to null
            )
            fields.add(JsonObject(field))
        }
    }

    private fun fixFile(dir: File, fileName: String): File {
        val file = File(dir, fileName)
        if (!file.exists()) {
            file.writeText(ApiDocHandler::class.java.getResource("/$fileName").readText())
            println("创建$file")
        }
        return file
    }
}
