package cn.bestwu.simpleframework.support.excel;

import cn.bestwu.simpleframework.support.excel.ExcelImportException.CellError;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.groups.Default;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

/**
 * 导入Excel文件（支持“XLS”和“XLSX”格式）
 */
public class ExcelImport extends AbstractExcelUtil {

  private static final Logger log = LoggerFactory.getLogger(ExcelImport.class);
  private static final Validator validator = Validation.buildDefaultValidatorFactory()
      .getValidator();
  /**
   * 工作表对象
   */
  private Workbook workbook;
  /**
   * 验证 groups
   */
  private Class<?>[] validateGroups = new Class[]{Default.class};

  /**
   * 构造函数
   *
   * @param fileName 导入文件
   * @throws IOException IOException
   */
  public ExcelImport(String fileName)
      throws IOException {
    this(new File(fileName));
  }

  /**
   * 构造函数
   *
   * @param file 导入文件对象
   * @throws IOException IOException
   */
  public ExcelImport(File file)
      throws IOException {
    this(file.getName(), new FileInputStream(file));
  }

  /**
   * 构造函数
   *
   * @param multipartFile 导入文件对象
   * @throws IOException IOException
   */
  public ExcelImport(MultipartFile multipartFile)
      throws IOException {
    this(multipartFile.getOriginalFilename(), multipartFile.getInputStream());
  }

  /**
   * 构造函数
   *
   * @param is is
   * @param fileName 导入文件对象
   * @throws IOException IOException
   */
  public ExcelImport(String fileName, InputStream is)
      throws IOException {
    if (!StringUtils.hasText(fileName)) {
      throw new RuntimeException("导入文档为空!");
    } else if (fileName.toLowerCase().endsWith("xls")) {
      workbook = new HSSFWorkbook(is);
    } else if (fileName.toLowerCase().endsWith("xlsx")) {
      workbook = new XSSFWorkbook(is);
    } else {
      throw new RuntimeException("文档格式不正确!");
    }
    log.debug("Initialize success.");
  }

  public Workbook getWorkbook() {
    return workbook;
  }

  /**
   * 获取导入数据列表
   *
   * @param cls 导入对象类型
   * @param groups 导入分组
   * @param <E> E
   * @return List
   * @throws InstantiationException InstantiationException
   * @throws IllegalAccessException IllegalAccessException
   * @throws NoSuchMethodException NoSuchMethodException
   * @throws ExcelImportException ExcelImportException
   */
  public <E> List<E> getDataList(Class<? extends E> cls, int... groups)
      throws InstantiationException, IllegalAccessException, NoSuchMethodException, ExcelImportException {
    return getDataList(0, 0, cls, groups);
  }

  /**
   * 获取导入数据列表
   *
   * @param cls 导入对象类型
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @param sheetIndex 工作表编号
   * @param groups 导入分组
   * @param <E> E
   * @return List
   * @throws InstantiationException InstantiationException
   * @throws IllegalAccessException IllegalAccessException
   * @throws NoSuchMethodException NoSuchMethodException
   * @throws ExcelImportException ExcelImportException
   */
  public <E> List<E> getDataList(int sheetIndex, int headerNum, Class<? extends E> cls,
      int... groups)
      throws InstantiationException, IllegalAccessException, NoSuchMethodException, ExcelImportException {
    if (workbook.getNumberOfSheets() < sheetIndex) {
      throw new RuntimeException("文档中没有工作表!");
    }
    Sheet sheet = workbook.getSheetAt(sheetIndex);

    List<ExcelFieldDescription> fieldDescriptions = getExcelFieldDescriptions(cls, FieldType.IMPORT,
        groups);
    //log.debug("Import column count:"+excelFieldDescriptions.size());
    // Get excel data
    List<E> dataList = new ArrayList<>();
    for (int i = headerNum + 1; i < sheet.getLastRowNum() + headerNum + 1; i++) {
      Row row = sheet.getRow(i);
      if (row == null) {
        continue;
      }
      E e = readRow(fieldDescriptions, cls, row);
      if (e != null) {
        dataList.add(e);
      }
    }
    return dataList;
  }

  public <E> E readRow(List<ExcelFieldDescription> fieldDescriptions, Class<E> cls, Row row)
      throws InstantiationException, IllegalAccessException, NoSuchMethodException, ExcelImportException {
    boolean notAllBlank = false;
    int column = 0;
    E e = cls.newInstance();
    StringBuilder sb = new StringBuilder();
    List<CellError> rowErrors = new ArrayList<>();
    int rowNum = row.getRowNum() + 1;
    for (ExcelFieldDescription fieldDescription : fieldDescriptions) {
      Object val = getCellValue(row, column++);
      if (val != null) {
        notAllBlank = notAllBlank || StringUtils.hasText(val.toString());
        ExcelField ef = fieldDescription.getExcelField();
        // Get param type and type cast
        Class<?> valType = Class.class;
        AccessibleObject accessibleObject = fieldDescription.getAccessibleObject();
        if (accessibleObject instanceof Field) {
          valType = ((Field) accessibleObject).getType();
        } else if (accessibleObject instanceof Method) {
          Method method = ((Method) accessibleObject);
          if ("get".equals(method.getName().substring(0, 3))) {
            valType = method.getReturnType();
          } else if ("set".equals(method.getName().substring(0, 3))) {
            valType = ((Method) accessibleObject).getParameterTypes()[0];
          }
        }
        //log.debug("Import value type: ["+i+","+column+"] " + valType);
        String valStr = String.valueOf(val).trim();
        try {
          Class<? extends CellValueConverter> converter = ef.converter();
          if (converter != CellValueConverter.class) {
            CellValueConverter newInstance = getCellValueConverter(converter);
            val = newInstance.fromCell(valStr, fieldDescription, e, row);
          } else if (valType == String.class) {
            if (val instanceof Number) {
              String pattern = ef.pattern();
              if (!StringUtils.hasText(pattern)) {
                pattern = "0";
              }
              DecimalFormat df = new DecimalFormat(pattern);
              val = df.format(new BigDecimal(valStr));
            } else {
              val = valStr;
            }
          } else if (StringUtils.hasText(valStr)) {
            if (valType == Integer.class) {
              val = Double.valueOf(valStr).intValue();
            } else if (valType == Long.class) {
              val = Double.valueOf(valStr).longValue();
            } else if (valType == BigDecimal.class) {
              val = new BigDecimal(valStr);
            } else if (valType == Double.class) {
              val = Double.valueOf(valStr);
            } else if (valType == Float.class) {
              val = Float.valueOf(valStr);
            } else if (valType == Date.class) {
              val = DateUtil.getJavaDate(Double.valueOf(valStr));
            } else {
              log.warn("Get cell value [" + rowNum + "," + column + "]");
              val = null;
            }
          } else {
            val = null;
          }
        } catch (Exception ex) {
          log.warn(
              "Get cell value [" + rowNum + "," + column + "] error: " + ex.toString());
          String message = ex.getMessage();
          rowErrors.add(new CellError(rowNum, column - 1, ef.title(), valStr,
              new IllegalArgumentException(
                  StringUtils.hasText(message) ? message : "typeMismatch",
                  ex)));
          val = null;
        }
        // set entity value
        String propertyName = null;
        if (accessibleObject instanceof Field) {
          propertyName = ((Field) accessibleObject).getName();
          ReflectionUtils.invokeMethod(
              BeanUtils.getPropertyDescriptor(e.getClass(), propertyName)
                  .getWriteMethod(), e, val);
        } else if (accessibleObject instanceof Method) {
          String mthodName = ((Method) accessibleObject).getName();
          if ("get".equals(mthodName.substring(0, 3))) {
            propertyName = mthodName.substring(3);
            mthodName = "set" + propertyName;
            propertyName = StringUtils.uncapitalize(propertyName);
          }

          ReflectionUtils.invokeMethod(cls.getMethod(mthodName, valType), e, val);
        }
        ConstraintViolationException exception = validateProperty(e, propertyName, validateGroups);
        if (exception != null) {
          rowErrors.add(new CellError(rowNum, column - 1, ef.title(), valStr, exception));
        }
      }
      sb.append(val).append(", ");
    }
    log.debug("Read success: [" + rowNum + "] " + sb.toString());
    if (notAllBlank) {
      if (!rowErrors.isEmpty()) {
        throw new ExcelImportException(rowErrors.get(0).getException().getMessage(), rowErrors);
      }
      return e;
    } else {
      return null;
    }
  }


  /**
   * 校验对象
   *
   * @param object 待校验对象
   * @param groups 待校验的组
   */
  private ConstraintViolationException validateProperty(Object object, String propertyName,
      Class<?>... groups) {
    Set<ConstraintViolation<Object>> constraintViolations = validator
        .validateProperty(object, propertyName, groups);
    if (constraintViolations.isEmpty()) {
      return null;
    } else {
      return new ConstraintViolationException(constraintViolations);
    }
  }

  /**
   * @param validateGroups 验证 groups
   */
  public void validateGroups(Class<?>... validateGroups) {
    this.validateGroups = validateGroups;
  }

  /**
   * 获取单元格值
   *
   * @param row 获取的行
   * @param column 获取单元格列号
   * @return 单元格值
   */
  public static Object getCellValue(Row row, int column) {
    Object val = "";
    try {
      Cell cell = row.getCell(column);
      if (cell != null) {
        if (cell.getCellType() == CellType.NUMERIC) {
          val = cell.getNumericCellValue();
        } else if (cell.getCellType() == CellType.STRING) {
          val = cell.getStringCellValue();
        } else if (cell.getCellType() == CellType.FORMULA) {
          val = cell.getCellFormula();
        } else if (cell.getCellType() == CellType.BOOLEAN) {
          val = cell.getBooleanCellValue();
        } else if (cell.getCellType() == CellType.ERROR) {
          val = cell.getErrorCellValue();
        }
      }
    } catch (Exception e) {
      log.warn(row + "，" + column + "，读取数据错误", e);
      return val;
    }
    return val;
  }
}
