package cn.bestwu.simpleframework.support.excel;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.mail.internet.MimeUtility;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.poi.hssf.usermodel.DVConstraint;
import org.apache.poi.hssf.usermodel.HSSFDataValidation;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.ss.SpreadsheetVersion;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Comment;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.ss.usermodel.DataValidation;
import org.apache.poi.ss.usermodel.DataValidationHelper;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFClientAnchor;
import org.apache.poi.xssf.usermodel.XSSFDataValidation;
import org.apache.poi.xssf.usermodel.XSSFDataValidationConstraint;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

/**
 * 导出Excel文件（导出“XLSX”格式，支持大数据量导出 ）
 *
 * @see SpreadsheetVersion
 */
public class ExcelExport extends AbstractExcelUtil {

  private static final Logger log = LoggerFactory.getLogger(ExcelExport.class);
  /**
   * 注解列表（ExcelFieldDescription）
   */
  private List<ExcelFieldDescription> excelFieldDescriptions;
  /**
   * 工作薄对象
   */
  private Workbook workbook;
  /**
   * 工作表对象
   */
  private Sheet sheet;
  /**
   * 样式列表
   */
  private Map<String, CellStyle> styles;
  /**
   * 当前行号+1
   */
  private int rownum;
  /**
   * 当前行
   */
  private Row row;
  /**
   * 默认单元格样式
   */
  private CellStyle defaultCellStyle;
  /**
   * 当前单元格号+1
   */
  private int cellnum = 0;
  /**
   * 是否自动换行
   */
  private boolean wrapText = false;
  /**
   * 是否包含批注
   */
  private boolean includeComment = true;
  private DataFormat wbDataFormat;

  /**
   * 构造函数
   *
   * @param cls 实体对象，通过annotation.ExportField获取标题
   */
  public ExcelExport(Class<?> cls) {
    this("工作表1", cls);
  }

  /**
   * 构造函数
   *
   * @param rowAccessWindowSize 行访问窗口大小
   * @param cls 实体对象，通过annotation.ExportField获取标题
   */
  public ExcelExport(int rowAccessWindowSize, Class<?> cls) {
    this(rowAccessWindowSize, "工作表1", cls);
  }

  /**
   * 构造函数
   *
   * @param sheetname sheetname
   * @param cls 实体对象，通过annotation.ExportField获取标题
   * @param groups 导入分组
   */
  public ExcelExport(String sheetname, Class<?> cls, int... groups) {
    this(500, sheetname, cls, groups);
  }

  /**
   * @param workbook 工作表
   * @param sheetIndex of the sheet number (0-based physical and logical)
   * @param cls 实体对象，通过annotation.ExportField获取标题
   * @param groups 导入分组
   */
  public ExcelExport(Workbook workbook, Integer sheetIndex, Class<?> cls, int... groups) {
    this.workbook = workbook;
    wbDataFormat = workbook.createDataFormat();
    this.sheet = workbook.getSheetAt(sheetIndex);
    rownum = 0;
    createStyles();
    initialize(cls, groups);
  }

  /**
   * 构造函数
   *
   * @param rowAccessWindowSize 行访问窗口大小
   * @param sheetname sheetname
   * @param cls 实体对象，通过annotation.ExportField获取标题
   * @param groups 导入分组
   */
  public ExcelExport(int rowAccessWindowSize, String sheetname, Class<?> cls, int... groups) {
    this.workbook = new SXSSFWorkbook(rowAccessWindowSize);
    wbDataFormat = workbook.createDataFormat();
    initSheet(sheetname);
    initialize(cls, groups);
  }

  /**
   * @param sheetname sheetname
   * @return this
   */
  private ExcelExport initSheet(String sheetname) {
    this.sheet = workbook.createSheet(sheetname);
    rownum = 0;
    createStyles();
    return this;
  }

  public void includeComment() {
    this.includeComment = true;
  }

  public void excludeComment() {
    this.includeComment = false;
  }

  public Workbook getWorkbook() {
    return workbook;
  }

  public Sheet getSheet() {
    return sheet;
  }

  public ExcelExport wrapText(boolean wrapText) {
    this.wrapText = wrapText;
    return this;
  }

  public void setDefaultCellStyle(CellStyle defaultCellStyle) {
    this.defaultCellStyle = defaultCellStyle;
  }

  /**
   * @param cls 实体对象，通过annotation.ExportField获取标题
   * @param groups 导入分组
   * @return this
   */
  public ExcelExport initialize(Class<?> cls, int... groups) {
    excelFieldDescriptions = getExcelFieldDescriptions(cls, FieldType.EXPORT, groups);
    return this;
  }

  /**
   * @param title 标题
   */
  public void createTitle(String title) {
    createTitle(title, getCellStyle(CellStyleType.TITLE));
  }

  /**
   * @param title 标题
   * @param titleStyle 标题样式
   */
  public void createTitle(String title, CellStyle titleStyle) {
    Assert.hasText(title, "title不能为空");
    Row titleRow = sheet.createRow(rownum++);
    titleRow.setHeightInPoints(30);
    Cell titleCell = titleRow.createCell(0);
    titleCell.setCellStyle(titleStyle);
    titleCell.setCellValue(title);
    sheet.addMergedRegion(new CellRangeAddress(titleRow.getRowNum(),
        titleRow.getRowNum(), 0, excelFieldDescriptions.size() - 1));
  }

  private void createHeader() {
    // Create header
    Row headerRow = sheet.createRow(rownum++);
    headerRow.setHeightInPoints(16);
    for (int i = 0; i < excelFieldDescriptions.size(); i++) {
      ExcelFieldDescription excelFieldDescription = excelFieldDescriptions.get(i);
      ExcelField excelField = excelFieldDescription.getExcelField();
      Cell cell = headerRow.createCell(i);
      String t = excelField.title();
      cell.setCellValue(t);
      cell.setCellStyle(getCellStyle(CellStyleType.HEADER));
      String commentStr = excelField.comment();
      if (includeComment && StringUtils.hasText(commentStr)) {
        Comment comment = this.sheet.createDrawingPatriarch().createCellComment(
            new XSSFClientAnchor(0, 0, 0, 0, (short) i, rownum - 1, (short) i, rownum - 1));
        comment.setString(new XSSFRichTextString(commentStr));
        cell.setCellComment(comment);
      }
      if (sheet instanceof SXSSFSheet) {
        ((SXSSFSheet) sheet).trackAllColumnsForAutoSizing();
      }
      sheet.autoSizeColumn(i);
      int colWidth = sheet.getColumnWidth(i) * 2;
      sheet.setColumnWidth(i, colWidth < 3000 ? 3000 : colWidth);
    }
  }

  /**
   * @param styleType 样式类型
   * @return 单元格样式
   */
  public CellStyle getCellStyle(CellStyleType styleType) {
    HorizontalAlignment defaultAlignment = getDefaultAlign(styleType);
    return getCellStyle(styleType, defaultAlignment, true);
  }

  /**
   * @param styleType 样式类型
   * @param alignment 对齐方式
   * @return 单元格样式
   */
  public CellStyle getCellStyle(CellStyleType styleType, HorizontalAlignment alignment) {
    return getCellStyle(styleType, alignment, true);
  }

  /**
   * @param styleType 样式类型
   * @param alignment 对齐方式
   * @param border 是否有边框
   * @return 单元格样式
   */
  public CellStyle getCellStyle(CellStyleType styleType, HorizontalAlignment alignment,
      boolean border) {
    String styleName = styleName(styleType, alignment, border);
    CellStyle cellStyle = styles.get(styleName);
    if (cellStyle == null) {
      HorizontalAlignment defaultAlignment = getDefaultAlign(styleType);
      CellStyle defaultCellStyle = styles.get(styleName(styleType, defaultAlignment, border));
      cellStyle = workbook.createCellStyle();
      cellStyle.cloneStyleFrom(defaultCellStyle);
      cellStyle.setAlignment(alignment);
      styles.put(styleName, cellStyle);
    }

    return cellStyle;
  }

  @NotNull
  private HorizontalAlignment getDefaultAlign(CellStyleType styleType) {
    HorizontalAlignment defaultAlignment = null;
    switch (styleType) {
      case TITLE:
        defaultAlignment = HorizontalAlignment.CENTER;
        break;
      case HEADER:
        defaultAlignment = HorizontalAlignment.CENTER;
        break;
      case DATA:
        defaultAlignment = HorizontalAlignment.GENERAL;
        break;
    }
    return defaultAlignment;
  }

  private String styleName(CellStyleType styleType, HorizontalAlignment alignment, boolean border) {
    return styleType.name() + "-" + alignment.name() + "-" + (border ? "BORDER" : "NOBORDER");
  }

  /**
   * 创建表格样式
   */
  private void createStyles() {
    this.styles = new HashMap<>();

    CellStyle style = workbook.createCellStyle();
    style.setAlignment(HorizontalAlignment.CENTER);
    style.setVerticalAlignment(VerticalAlignment.CENTER);
    Font titleFont = workbook.createFont();
    titleFont.setFontName("Arial");
    titleFont.setFontHeightInPoints((short) 16);
    titleFont.setBold(true);
    style.setFont(titleFont);
    String styleName = styleName(CellStyleType.TITLE, HorizontalAlignment.CENTER, false);
    styles.put(styleName, style);

    style = createBorderCellStyle(styleName);
    styles.put(styleName(CellStyleType.TITLE, HorizontalAlignment.CENTER, true), style);

    style = workbook.createCellStyle();
    style.setVerticalAlignment(VerticalAlignment.CENTER);
    style.setWrapText(wrapText);
    Font dataFont = workbook.createFont();
    dataFont.setFontName("Arial");
    dataFont.setFontHeightInPoints((short) 10);
    style.setFont(dataFont);
    styleName = styleName(CellStyleType.DATA, HorizontalAlignment.GENERAL, false);
    styles.put(styleName, style);

    style = createBorderCellStyle(styleName);
    styleName = styleName(CellStyleType.DATA, HorizontalAlignment.GENERAL, true);
    styles.put(styleName, style);

    style = workbook.createCellStyle();
    style.cloneStyleFrom(styles.get(styleName));
    style.setAlignment(HorizontalAlignment.LEFT);
    styles.put(styleName(CellStyleType.DATA, HorizontalAlignment.LEFT, true), style);

    style = workbook.createCellStyle();
    style.cloneStyleFrom(styles.get(styleName));
    style.setAlignment(HorizontalAlignment.CENTER);
    styles.put(styleName(CellStyleType.DATA, HorizontalAlignment.CENTER, true), style);
    defaultCellStyle = style;

    style = workbook.createCellStyle();
    style.cloneStyleFrom(styles.get(styleName));
    style.setAlignment(HorizontalAlignment.RIGHT);
    styles.put(styleName(CellStyleType.DATA, HorizontalAlignment.RIGHT, true), style);

    style = workbook.createCellStyle();
    style.cloneStyleFrom(
        styles.get(styleName(CellStyleType.DATA, HorizontalAlignment.GENERAL, false)));
    style.setAlignment(HorizontalAlignment.CENTER);
    style.setFillForegroundColor(IndexedColors.GREY_50_PERCENT.getIndex());
    style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
    Font headerFont = workbook.createFont();
    headerFont.setFontName("Arial");
    headerFont.setFontHeightInPoints((short) 10);
    headerFont.setBold(true);
    headerFont.setColor(IndexedColors.WHITE.getIndex());
    style.setFont(headerFont);
    styleName = styleName(CellStyleType.HEADER, HorizontalAlignment.CENTER, false);
    styles.put(styleName, style);

    style = createBorderCellStyle(styleName);
    styles.put(styleName(CellStyleType.HEADER, HorizontalAlignment.CENTER, true), style);
  }

  @NotNull
  private CellStyle createBorderCellStyle(String styleName) {
    CellStyle style = workbook.createCellStyle();
    style.cloneStyleFrom(styles.get(styleName));
    style.setBorderRight(BorderStyle.THIN);
    style.setRightBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
    style.setBorderLeft(BorderStyle.THIN);
    style.setLeftBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
    style.setBorderTop(BorderStyle.THIN);
    style.setTopBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
    style.setBorderBottom(BorderStyle.THIN);
    style.setBottomBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
    return style;
  }

  /**
   * @param firstCol 开始列
   * @param lastCol 结束列
   */
  public void addMergedRegion(int firstCol, int lastCol) {
    int row = rownum - 1;
    addMergedRegion(row, row, firstCol, lastCol);
  }

  /**
   * @param firstRow 开始行
   * @param lastRow 结束行
   * @param firstCol 开始列
   * @param lastCol 结束列
   */
  public void addMergedRegion(int firstRow, int lastRow, int firstCol, int lastCol) {
    sheet.addMergedRegion(new CellRangeAddress(firstRow, lastRow, firstCol, lastCol));
  }

  /**
   * 添加数据有效性检查.
   *
   * @param firstCol 开始列
   * @param lastCol 结束列
   * @param explicitListValues 有效性检查的下拉列表
   */
  public void addValidationData(int firstCol, int lastCol, String[] explicitListValues) {
    int num = rownum - 1;
    addValidationData(num, num, firstCol, lastCol, explicitListValues);
  }

  /**
   * 添加数据有效性检查.
   *
   * @param firstRow 开始行
   * @param lastRow 结束行
   * @param firstCol 开始列
   * @param lastCol 结束列
   * @param explicitListValues 有效性检查的下拉列表
   */
  public void addValidationData(int firstRow, int lastRow, int firstCol, int lastCol,
      String[] explicitListValues) {
    if (sheet instanceof XSSFSheet || sheet instanceof SXSSFSheet) {
      DataValidationHelper dvHelper = sheet.getDataValidationHelper();
      XSSFDataValidationConstraint dvConstraint = (XSSFDataValidationConstraint) dvHelper
          .createExplicitListConstraint(explicitListValues);
      CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol,
          lastCol);
      XSSFDataValidation validation = (XSSFDataValidation) dvHelper
          .createValidation(dvConstraint, addressList);
      validation.setSuppressDropDownArrow(true);
      validation.setShowErrorBox(true);
      sheet.addValidationData(validation);
    } else if (sheet instanceof HSSFSheet) {
      CellRangeAddressList addressList = new CellRangeAddressList(firstRow, lastRow, firstCol,
          lastCol);
      DVConstraint dvConstraint = DVConstraint.createExplicitListConstraint(explicitListValues);
      DataValidation validation = new HSSFDataValidation(addressList, dvConstraint);
      validation.setSuppressDropDownArrow(true);
      validation.setShowErrorBox(true);
      sheet.addValidationData(validation);
    }

  }

  /**
   * 添加一个单元格
   *
   * @return 单元格对象
   */
  public Cell addCell() {
    return addCell(cellnum);
  }

  /**
   * 添加一个单元格
   *
   * @param cellStyle 样式
   * @return 单元格对象
   */
  public Cell addCell(CellStyle cellStyle) {
    return addCell(cellnum, cellStyle);
  }

  /**
   * 添加一个单元格
   *
   * @param column 列号
   * @return 单元格对象
   */
  public Cell addCell(int column) {
    return addCell(column, defaultCellStyle);
  }

  /**
   * 添加一个单元格
   *
   * @param column 列号
   * @param cellStyle 样式
   * @return 单元格对象
   */
  public Cell addCell(int column, CellStyle cellStyle) {
    cellnum = column + 1;
    Cell cell = row.createCell(column);
    CellStyle style = workbook.createCellStyle();
    style.cloneStyleFrom(cellStyle);
    cell.setCellStyle(style);
    return cell;
  }

  /**
   * 添加一行
   *
   * @return 行对象
   */
  public Row addRow() {
    row = sheet.getRow(rownum);
    if (row == null) {
      this.row = sheet.createRow(rownum++);
    }
    cellnum = 0;
    return this.row;
  }

  /**
   * 添加一个单元格
   *
   * @param row 添加的行
   * @param column 添加列号
   * @param val 添加值
   * @param ef ExcelField
   * @param e E
   * @return 单元格对象
   */
  private Cell addCell(Row row, int column, Object val, ExcelField ef,
      ExcelFieldDescription excelFieldDescription, Object e) {
    HorizontalAlignment align = HorizontalAlignment.forInt(ef.align().ordinal());
    Class<? extends CellValueConverter> converter = ef.converter();
    Cell cell = row.createCell(column);
    String cellFormatString = ef.pattern();
    try {
      if (val == null) {
        cell.setCellValue("");
      } else if (converter != CellValueConverter.class) {
        CellValueConverter newInstance = getCellValueConverter(converter);
        cell.setCellValue(newInstance.toCell(val, excelFieldDescription, e));
      } else {
        if (val instanceof String) {
          cell.setCellValue((String) val);
        } else if (val instanceof Integer) {
          cell.setCellValue((Integer) val);
          cellFormatString = setCellFormatString(cellFormatString, "0");
        } else if (val instanceof Long) {
          cell.setCellValue((Long) val);
          cellFormatString = setCellFormatString(cellFormatString, "0");
        } else if (val instanceof BigDecimal) {
          cell.setCellValue(((BigDecimal) val).toPlainString());
          cellFormatString = setCellFormatString(cellFormatString, "0.00");
        } else if (val instanceof Double) {
          cell.setCellValue((Double) val);
          cellFormatString = setCellFormatString(cellFormatString, "0.00");
        } else if (val instanceof Float) {
          cell.setCellValue((Float) val);
          cellFormatString = setCellFormatString(cellFormatString, "0.00");
        } else if (val instanceof Date) {
          cell.setCellValue((Date) val);
          cellFormatString = setCellFormatString(cellFormatString, "yyyy-MM-dd HH:mm");
        } else {
          log.warn("Can not set cell value [" + val + "]");
          cell.setCellValue("");
        }
      }
    } catch (Exception ex) {
      log.warn("Set cell value [" + row.getRowNum() + "," + column + "] error: " + ex.toString());
      cell.setCellValue(val.toString());
    }
    CellStyle style = styles.get("data_column_" + column);
    if (style == null) {
      style = workbook.createCellStyle();
      style.cloneStyleFrom(getCellStyle(CellStyleType.DATA, align));
      cellFormatString = setCellFormatString(cellFormatString, "@");
      style.setDataFormat(getFormat(cellFormatString));
      styles.put("data_column_" + column, style);
    }
    cell.setCellStyle(style);
    return cell;
  }

  public short getFormat(String cellFormatString) {
    return wbDataFormat.getFormat(cellFormatString);
  }

  @NotNull
  private String setCellFormatString(String cellFormatString, String format) {
    if (!StringUtils.hasText(cellFormatString)) {
      return format;
    } else {
      return cellFormatString;
    }
  }

  /**
   * 调用Getter方法. 支持多级，如：对象名.对象名.方法
   */
  private Object invokeGetter(Object obj, String propertyName) throws NoSuchMethodException {
    Object object = obj;
    for (String name : StringUtils.split(propertyName, ".")) {
      String getterMethodName = "get" + StringUtils.capitalize(name);
      object = ReflectionUtils.invokeMethod(object.getClass().getMethod(getterMethodName), object);
    }
    return object;
  }

  /**
   * 添加数据（通过annotation.ExportField添加数据）
   *
   * @param <E> E
   * @param list list
   * @return list 数据列表
   */
  public <E> ExcelExport setDataList(List<E> list) {
    return setDataList(0, list);
  }

  /**
   * 添加数据（通过annotation.ExportField添加数据）
   *
   * @param <E> E
   * @param list list
   * @param startColunm 起始列
   * @return list 数据列表
   */
  public <E> ExcelExport setDataList(int startColunm, List<E> list) {
    createHeader();
    for (E e : list) {
      int colunm = startColunm;
      Row row = this.addRow();
      StringBuilder sb = new StringBuilder();
      for (ExcelFieldDescription fieldDescription : excelFieldDescriptions) {
        ExcelField excelField = fieldDescription.getExcelField();
        Object val = null;
        // Get entity value
        try {
          if (StringUtils.hasText(excelField.value())) {
            val = invokeGetter(e, excelField.value());
          } else {
            AccessibleObject accessibleObject = fieldDescription.getAccessibleObject();
            if (accessibleObject instanceof Field) {
              String propertyName = ((Field) accessibleObject).getName();
              val = ReflectionUtils
                  .invokeMethod(BeanUtils.getPropertyDescriptor(e.getClass(), propertyName)
                      .getReadMethod(), e);
            } else if (accessibleObject instanceof Method) {
              String methodName = ((Method) accessibleObject).getName();
              val = ReflectionUtils.invokeMethod(e.getClass().getMethod(methodName), e);
            }
          }
        } catch (Exception ex) {
          // Failure to ignore
          log.info(ex.toString());
          val = "";
        }
        this.addCell(row, colunm++, val, excelField, fieldDescription, e);
        sb.append(val).append(", ");
      }
      log.debug("Write success: [" + row.getRowNum() + "] " + sb.toString());
    }
    return this;
  }

  /**
   * 输出数据流
   *
   * @param os 输出数据流
   * @return this
   * @throws IOException IOException
   */
  public ExcelExport write(OutputStream os) throws IOException {
    workbook.write(os);
    dispose();
    return this;
  }

  /**
   * 输出到客户端
   *
   * @param request request
   * @param response response
   * @param fileName 输出文件名
   * @return this
   * @throws IOException IOException
   */
  public ExcelExport write(HttpServletRequest request, HttpServletResponse response,
      String fileName) throws IOException {
    response.reset();
    String agent = request.getHeader("USER-AGENT");

    String newFileName;
    if (null != agent && (agent.contains("Trident") || agent.contains("Edge"))) {
      newFileName = URLEncoder.encode(fileName, "UTF-8");
    } else {
      newFileName = MimeUtility.encodeText(fileName, "UTF8", "B");
    }
    response.setHeader("Content-Disposition",
        "attachment;filename=" + newFileName + getExtension() + ";filename*=UTF-8''" + URLEncoder
            .encode(fileName, "UTF-8") + getExtension());
    response.setContentType("application/vnd.ms-excel; charset=utf-8");
    response.setHeader("Pragma", "No-cache");
    response.setHeader("Cache-Control", "no-cache");
    response.setDateHeader("Expires", 0);
    write(response.getOutputStream());
    return this;
  }

  @NotNull
  public String getExtension() {
    if (workbook instanceof XSSFWorkbook || workbook instanceof SXSSFWorkbook) {
      return ".xlsx";
    } else {
      return ".xls";
    }
  }

  /**
   * 输出到文件
   *
   * @param fileName 输出文件名
   * @return this
   * @throws IOException IOException
   */
  public ExcelExport writeFile(String fileName) throws IOException {
    FileOutputStream os = new FileOutputStream(fileName);
    this.write(os);
    return this;
  }

  /**
   * 清理临时文件
   *
   * @return this
   * @throws IOException IOException
   */
  public ExcelExport dispose() throws IOException {
    workbook.close();
    if (workbook instanceof SXSSFWorkbook) {
      ((SXSSFWorkbook) workbook).dispose();
    }
    return this;
  }

}
