package cn.bestwu.simpleframework.support;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Calendar;
import java.util.Date;

/**
 * @author Peter Wu
 */
public class LocalDateTimeHelper {

  public final static ZoneOffset DEFAULT_ZONE_OFFSET = ZoneOffset.of("+8");

  private final LocalDateTime localDateTime;
  private ZoneOffset zoneOffset;

  private LocalDateTimeHelper(LocalDateTime localDateTime, ZoneOffset zoneOffset) {
    this.localDateTime = localDateTime;
    this.zoneOffset = zoneOffset;
  }

  public static LocalDateTimeHelper now() {
    return new LocalDateTimeHelper(LocalDateTime.now(), DEFAULT_ZONE_OFFSET);
  }

  public static LocalDateTimeHelper parse(CharSequence text) {
    return new LocalDateTimeHelper(LocalDateTime.parse(text), DEFAULT_ZONE_OFFSET);
  }

  public static LocalDateTimeHelper parse(CharSequence text, DateTimeFormatter formatter) {
    return new LocalDateTimeHelper(LocalDateTime.parse(text, formatter), DEFAULT_ZONE_OFFSET);
  }

  public static LocalDateTimeHelper of(int year, int month, int dayOfMonth) {
    return new LocalDateTimeHelper(LocalDate.of(year, month, dayOfMonth).atStartOfDay(),
        DEFAULT_ZONE_OFFSET);
  }

  public static LocalDateTimeHelper of(LocalDateTime localDateTime) {
    return new LocalDateTimeHelper(localDateTime, DEFAULT_ZONE_OFFSET);
  }

  public static LocalDateTimeHelper of(LocalDate localDate) {
    return new LocalDateTimeHelper(localDate.atStartOfDay(), DEFAULT_ZONE_OFFSET);
  }

  public static LocalDateTimeHelper of(Date date) {
    Calendar calendar = Calendar.getInstance();
    calendar.setTime(date);
    return of(calendar);
  }

  public static LocalDateTimeHelper of(long millis) {
    Calendar calendar = Calendar.getInstance();
    calendar.setTimeInMillis(millis);
    return of(calendar);
  }

  public static LocalDateTimeHelper of(Calendar calendar) {
    return new LocalDateTimeHelper(
        LocalDateTime.of(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1,
            calendar.get(Calendar.DAY_OF_MONTH), calendar.get(Calendar.HOUR_OF_DAY),
            calendar.get(Calendar.MINUTE), calendar.get(Calendar.SECOND),
            calendar.get(Calendar.MILLISECOND) * 1000000),
        DEFAULT_ZONE_OFFSET);
  }

  public static Date toDate(LocalDate localDate) {
    return Date.from(localDate.atStartOfDay(DEFAULT_ZONE_OFFSET).toInstant());
  }

  //--------------------------------------------
  public long toMillis() {
    return toInstant().toEpochMilli();
  }

  public Date toDate() {
    return Date.from(toInstant());
  }

  public Instant toInstant() {
    return toLocalDateTime().toInstant(zoneOffset);
  }

  public LocalDateTime toLocalDateTime() {
    return localDateTime;
  }

  public LocalDate toLocalDate() {
    return localDateTime.toLocalDate();
  }

  public String format() {
    return toLocalDateTime().format(DateTimeFormatter.ISO_DATE_TIME);
  }

  public String format(DateTimeFormatter dateTimeFormatter) {
    return localDateTime.format(dateTimeFormatter);
  }

  public LocalDateTimeHelper zoneOffset(ZoneOffset zoneOffset) {
    this.zoneOffset = zoneOffset;
    return this;
  }

  /**
   * 取得当月第一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getFirstDayOfMonth() {
    return LocalDateTimeHelper.of(localDateTime.with(TemporalAdjusters.firstDayOfMonth()))
        .zoneOffset(zoneOffset);
  }

  /**
   * 取得下月第一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getFirstDayOfNextMonth() {
    return LocalDateTimeHelper.of(localDateTime.with(TemporalAdjusters.firstDayOfNextMonth()))
        .zoneOffset(zoneOffset);
  }

  /**
   * 取得当月最后一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getLastDayOfMonth() {
    return LocalDateTimeHelper.of(localDateTime.with(TemporalAdjusters.lastDayOfMonth()))
        .zoneOffset(zoneOffset);
  }

  /**
   * 取得当季度第一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getFirstDayOfQuarter() {
    return LocalDateTimeHelper
        .of(localDateTime.withMonth(localDateTime.getMonth().firstMonthOfQuarter().getValue())
            .with(TemporalAdjusters.firstDayOfMonth())).zoneOffset(zoneOffset);
  }

  /**
   * 取得下季度第一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getFirstDayOfNextQuarter() {
    return LocalDateTimeHelper
        .of(localDateTime
            .withMonth(localDateTime.getMonth().firstMonthOfQuarter().plus(3).getValue())
            .with(TemporalAdjusters.firstDayOfMonth())).zoneOffset(zoneOffset);
  }

  /**
   * 取得当季度最后一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getLastDayOfQuarter() {
    return LocalDateTimeHelper
        .of(localDateTime
            .withMonth(localDateTime.getMonth().firstMonthOfQuarter().plus(2).getValue())
            .with(TemporalAdjusters.lastDayOfMonth())).zoneOffset(zoneOffset);
  }

  /**
   * 获取当年的第一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getFirstDayOfYear() {
    return LocalDateTimeHelper.of(localDateTime.with(TemporalAdjusters.firstDayOfYear()))
        .zoneOffset(zoneOffset);
  }

  /**
   * 获取下年的第一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getFirstDayOfNextYear() {
    return LocalDateTimeHelper.of(localDateTime.with(TemporalAdjusters.firstDayOfNextYear()))
        .zoneOffset(zoneOffset);
  }

  /**
   * 获取当年的最后一天
   *
   * @return LocalDateHelper
   */
  public LocalDateTimeHelper getLastDayOfYear() {
    return LocalDateTimeHelper.of(localDateTime.with(TemporalAdjusters.lastDayOfYear()))
        .zoneOffset(zoneOffset);
  }

}
