package cn.bestwu.simpleframework.security.config;

import cn.bestwu.simpleframework.security.URLFilterInvocationSecurityMetadataSource;
import cn.bestwu.simpleframework.security.exception.CustomWebResponseExceptionTranslator;
import cn.bestwu.simpleframework.security.exception.SecurityOAuth2ErrorHandler;
import java.util.Collection;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.access.AccessDecisionManager;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.config.annotation.ObjectPostProcessor;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.common.exceptions.OAuth2Exception;
import org.springframework.security.oauth2.common.exceptions.UnauthorizedUserException;
import org.springframework.security.oauth2.config.annotation.web.configuration.ResourceServerConfigurerAdapter;
import org.springframework.security.oauth2.config.annotation.web.configurers.ResourceServerSecurityConfigurer;
import org.springframework.security.oauth2.provider.endpoint.TokenEndpoint;
import org.springframework.security.oauth2.provider.error.DefaultOAuth2ExceptionRenderer;
import org.springframework.security.oauth2.provider.error.OAuth2AccessDeniedHandler;
import org.springframework.security.oauth2.provider.error.OAuth2AuthenticationEntryPoint;
import org.springframework.security.oauth2.provider.error.WebResponseExceptionTranslator;
import org.springframework.security.web.access.intercept.FilterSecurityInterceptor;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerAdapter;

@ConditionalOnClass(OAuth2Exception.class)
@Configuration
@ConditionalOnWebApplication
public class SecurityResourceServerConfiguration {

  @Value("${app.web.ok.enable:false}")
  private Boolean okEnable;

  public final MessageSource messageSource;

  public SecurityResourceServerConfiguration(MessageSource messageSource) {
    this.messageSource = messageSource;
  }

  @Bean
  public SecurityOAuth2ErrorHandler securityErrorHandler() {
    return new SecurityOAuth2ErrorHandler();
  }

  @Bean
  public WebResponseExceptionTranslator<OAuth2Exception> webResponseExceptionTranslator() {
    return new CustomWebResponseExceptionTranslator(okEnable, messageSource);
  }

  @Configuration
  @ConditionalOnWebApplication
  public class ResourceServerConfiguration extends ResourceServerConfigurerAdapter {

    private final WebResponseExceptionTranslator<OAuth2Exception> webResponseExceptionTranslator;
    @Value("${server.client-cache:true}")
    private boolean supportClientCache;
    @Value("${security.cors.enable:false}")
    private boolean enableCors;
    @Value("${security.http.session-creation-policy:STATELESS}")
    private SessionCreationPolicy sessionCreationPolicy;
    @Value("${security.http.frame-options-disable:true}")
    private boolean frameOptionsDisable;
    private final URLFilterInvocationSecurityMetadataSource securityMetadataSource;
    private final RequestMappingHandlerAdapter requestMappingHandlerAdapter;


    public ResourceServerConfiguration(TokenEndpoint tokenEndpoint,
        WebResponseExceptionTranslator<OAuth2Exception> webResponseExceptionTranslator,
        URLFilterInvocationSecurityMetadataSource securityMetadataSource,
        RequestMappingHandlerAdapter requestMappingHandlerAdapter) {
      this.webResponseExceptionTranslator = webResponseExceptionTranslator;
      this.securityMetadataSource = securityMetadataSource;
      this.requestMappingHandlerAdapter = requestMappingHandlerAdapter;
      tokenEndpoint.setProviderExceptionHandler(webResponseExceptionTranslator);
    }

    @Bean
    public AccessDecisionManager accessDecisionManager() {
      return new AccessDecisionManager() {
        @Override
        public void decide(Authentication authentication, Object object,
            Collection<ConfigAttribute> configAttributes) {
          if (configAttributes.isEmpty()) {
            return;
          }
          Collection<? extends GrantedAuthority> authorities = authentication.getAuthorities();
          for (ConfigAttribute configAttribute : configAttributes) {
            String attribute = configAttribute.getAttribute();
            for (GrantedAuthority grantedAuthority : authorities) {
              if (attribute.trim().equals(grantedAuthority.getAuthority().trim())) {
                return;
              }
            }
          }
          if (authentication instanceof AnonymousAuthenticationToken) {
            throw new UnauthorizedUserException("请重新登录");
          } else {
            throw new AccessDeniedException("无权访问");
          }
        }

        @Override
        public boolean supports(ConfigAttribute attribute) {
          return true;
        }

        @Override
        public boolean supports(Class<?> clazz) {
          return true;
        }

      };
    }


    @Override
    public void configure(ResourceServerSecurityConfigurer resources) {
      resources.stateless(false);
      DefaultOAuth2ExceptionRenderer exceptionRenderer = new DefaultOAuth2ExceptionRenderer();
      exceptionRenderer.setMessageConverters(requestMappingHandlerAdapter.getMessageConverters());
      OAuth2AuthenticationEntryPoint oAuth2AuthenticationEntryPoint = new OAuth2AuthenticationEntryPoint();
      oAuth2AuthenticationEntryPoint.setExceptionTranslator(webResponseExceptionTranslator);
      oAuth2AuthenticationEntryPoint.setExceptionRenderer(exceptionRenderer);
      resources.authenticationEntryPoint(oAuth2AuthenticationEntryPoint);
      OAuth2AccessDeniedHandler accessDeniedHandler = new OAuth2AccessDeniedHandler();
      accessDeniedHandler.setExceptionTranslator(webResponseExceptionTranslator);
      accessDeniedHandler.setExceptionRenderer(exceptionRenderer);
      resources.accessDeniedHandler(accessDeniedHandler);
    }

    @Override
    public void configure(HttpSecurity http) throws Exception {
      if (supportClientCache) {
        http.headers().cacheControl().disable();
      }
      if (enableCors) {
        http.cors();
      }
      if (frameOptionsDisable) {
        http.headers().frameOptions().disable();
      }
      http
          .sessionManagement().sessionCreationPolicy(sessionCreationPolicy)
          .and()
          .authorizeRequests()
          .withObjectPostProcessor(new ObjectPostProcessor<FilterSecurityInterceptor>() {
            public <O extends FilterSecurityInterceptor> O postProcess(
                O fsi) {
              fsi.setSecurityMetadataSource(securityMetadataSource);
              fsi.setAccessDecisionManager(accessDecisionManager());
              return fsi;
            }
          })
          .anyRequest().authenticated()
      ;
    }

  }

}