package cn.bestwu.simpleframework.security;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.access.SecurityConfig;
import org.springframework.security.web.FilterInvocation;
import org.springframework.security.web.access.intercept.FilterInvocationSecurityMetadataSource;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.util.AntPathMatcher;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

/**
 * 自定义权限过滤
 *
 * @author Peter Wu
 */
public class URLFilterInvocationSecurityMetadataSource implements
    FilterInvocationSecurityMetadataSource {

  private final Map<AntPathRequestMatcher, Collection<ConfigAttribute>> allDefaultRequestMap = new HashMap<>();
  private Map<AntPathRequestMatcher, Collection<ConfigAttribute>> requestMap;
  private final ISecurityService securityService;
  private final String[] ignored;

  // ~ Constructors
  // ===================================================================================================
  public URLFilterInvocationSecurityMetadataSource(
      ISecurityService securityService,
      RequestMappingHandlerMapping handlerMapping, String[] ignored) {
    this.securityService = securityService;
    this.ignored = ignored;

    handlerMapping.getHandlerMethods().forEach((mappingInfo, handlerMethod) -> {
      //非匿名权限
      if (!handlerMethod.hasMethodAnnotation(Anonymous.class) && !handlerMethod
          .hasMethodAnnotation(ClientAuthorize.class)) {
        for (String pattern : mappingInfo.getPatternsCondition().getPatterns()) {
          if (!ignored(pattern)) {
            Set<RequestMethod> methods = mappingInfo.getMethodsCondition().getMethods();
            if (methods.isEmpty()) {
              this.allDefaultRequestMap.put(new AntPathRequestMatcher(pattern),
                  SecurityConfig.createList("authenticated"));
            } else {
              for (RequestMethod requestMethod : methods) {
                this.allDefaultRequestMap
                    .put(new AntPathRequestMatcher(pattern, requestMethod.name()),
                        SecurityConfig.createList("authenticated"));
              }
            }
          }
        }
      }
    });

    bindRequestMap();
  }

  private boolean ignored(String path) {
    if (ignored == null) {
      return false;
    }
    AntPathMatcher antPathMatcher = new AntPathMatcher();
    for (String pattern : ignored) {
      if (antPathMatcher.match(pattern, path)) {
        return true;
      }
    }
    return false;
  }

  // ~ Methods
  // ========================================================================================================
  @Override
  public Collection<ConfigAttribute> getAllConfigAttributes() {
    Set<ConfigAttribute> allAttributes = new HashSet<>();

    for (Map.Entry<AntPathRequestMatcher, Collection<ConfigAttribute>> entry : requestMap
        .entrySet()) {
      allAttributes.addAll(entry.getValue());
    }
    return allAttributes;
  }

  @Override
  public Collection<ConfigAttribute> getAttributes(Object object) {
    final HttpServletRequest request = ((FilterInvocation) object).getRequest();
    List<Match> matches = new ArrayList<>();
    Comparator<String> comparator = new AntPathMatcher()
        .getPatternComparator(getRequestPath(request));
    for (Map.Entry<AntPathRequestMatcher, Collection<ConfigAttribute>> entry : requestMap
        .entrySet()) {
      if (entry.getKey().matches(request)) {
        matches.add(new Match(comparator, entry.getKey().getPattern(), entry.getValue()));
      }
    }
    if (!matches.isEmpty()) {
      Collections.sort(matches);
      Match bestMatch = matches.get(0);
      if (matches.size() > 1) {
        Match secondBestMatch = matches.get(1);
        if (comparator.compare(bestMatch.path, secondBestMatch.path) == 0) {
          String m1 = bestMatch.path;
          String m2 = secondBestMatch.path;
          throw new IllegalStateException("Ambiguous handler methods mapped for HTTP path '" +
              request.getRequestURL() + "': {" + m1 + ", " + m2 + "}");
        }
      }
      return bestMatch.configAttributes;
    }
    return Collections.emptyList();
  }

  private String getRequestPath(HttpServletRequest request) {
    String url = request.getServletPath();

    if (request.getPathInfo() != null) {
      url += request.getPathInfo();
    }

    return url;
  }

  private class Match implements Comparable<Match> {

    private final Comparator<String> comparator;
    private final String path;

    private final Collection<ConfigAttribute> configAttributes;

    private Match(Comparator<String> comparator, String path,
        Collection<ConfigAttribute> configAttributes) {
      this.comparator = comparator;
      this.path = path;
      this.configAttributes = configAttributes;
    }

    @Override
    public int compareTo(Match o) {
      return comparator.compare(path, o.path);
    }
  }

  @Override
  public boolean supports(Class<?> clazz) {
    return FilterInvocation.class.isAssignableFrom(clazz);
  }

  protected void bindRequestMap() {
    this.requestMap = new LinkedHashMap<>(allDefaultRequestMap);
    List<? extends IMenu> list = securityService.findAllMenus();
    for (IMenu menu : list) {
      String url = menu.getUrl();
      String method = menu.getMethod();
      String[] authorities = menu.getAuthorities();
      Collection<ConfigAttribute> atts = SecurityConfig
          .createList(authorities);
      for (String s : url.split(",")) {
        for (String m : method.split(",")) {
          this.requestMap.put(new AntPathRequestMatcher(s, m), atts);
        }
      }
    }
  }

  public void refreshResuorceMap() {
    this.bindRequestMap();
  }

}