package cn.bestwu.simpleframework.security.exception;

import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.springframework.context.MessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.oauth2.common.exceptions.OAuth2Exception;
import org.springframework.security.oauth2.provider.error.DefaultOAuth2ExceptionRenderer;
import org.springframework.security.oauth2.provider.error.DefaultWebResponseExceptionTranslator;
import org.springframework.security.oauth2.provider.error.OAuth2ExceptionRenderer;
import org.springframework.security.oauth2.provider.error.WebResponseExceptionTranslator;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.context.request.ServletWebRequest;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.mvc.support.DefaultHandlerExceptionResolver;

/**
 * 自定义处理OAuth2SecurityException
 *
 * @author Peter Wu
 */
public class AbstractOAuth2SecurityExceptionHandler {

  private WebResponseExceptionTranslator<OAuth2Exception> exceptionTranslator = new DefaultWebResponseExceptionTranslator();

  private OAuth2ExceptionRenderer exceptionRenderer = new DefaultOAuth2ExceptionRenderer();

  // This is from Spring MVC.
  private final HandlerExceptionResolver handlerExceptionResolver = new DefaultHandlerExceptionResolver();
  public static MessageSource messageSource;

  public AbstractOAuth2SecurityExceptionHandler(
      MessageSource messageSource) {
    AbstractOAuth2SecurityExceptionHandler.messageSource = messageSource;
  }

  public static String getText(Object code, Object... args) {
    ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder
        .getRequestAttributes();
    HttpServletRequest request =
        requestAttributes == null ? null : requestAttributes.getRequest();
    String codeString = String.valueOf(code);
    return messageSource.getMessage(codeString, args, codeString,
        request == null ? Locale.CHINA : request.getLocale());
  }

  public void setExceptionTranslator(
      WebResponseExceptionTranslator<OAuth2Exception> exceptionTranslator) {
    this.exceptionTranslator = exceptionTranslator;
  }

  public void setExceptionRenderer(OAuth2ExceptionRenderer exceptionRenderer) {
    this.exceptionRenderer = exceptionRenderer;
  }

  protected void doHandle(HttpServletRequest request, HttpServletResponse response,
      RuntimeException authException)
      throws IOException, ServletException {
    try {
      ResponseEntity<OAuth2Exception> result = exceptionTranslator.translate(authException);
      result = enhanceResponse(result, authException);
      Map<String, Object> body = new HashMap<>();
      HttpStatus statusCode = result.getStatusCode();
      body.put("status", String.valueOf(statusCode.value()));
      body.put("message", getText(result.getBody().getMessage()));

      exceptionRenderer.handleHttpEntityResponse(
          ResponseEntity.status(statusCode).headers(result.getHeaders()).body(
              body), new ServletWebRequest(request, response));
      response.flushBuffer();
    } catch (ServletException e) {
      // Re-use some of the default Spring dispatcher behaviour - the exception came from the filter chain and
      // not from an MVC handler so it won't be caught by the dispatcher (even if there is one)
      if (handlerExceptionResolver.resolveException(request, response, this, e) == null) {
        throw e;
      }
    } catch (IOException | RuntimeException e) {
      throw e;
    } catch (Exception e) {
      // Wrap other Exceptions. These are not expected to happen
      throw new RuntimeException(e);
    }
  }

  protected ResponseEntity<OAuth2Exception> enhanceResponse(ResponseEntity<OAuth2Exception> result,
      Exception authException) {
    return result;
  }

}