package cn.bestwu.simpleframework.config;

import cn.bestwu.logging.annotation.NoRequestLogging;
import cn.bestwu.simpleframework.data.Repositories;
import cn.bestwu.simpleframework.data.binding.WrapperBinderProperties;
import cn.bestwu.simpleframework.data.resolver.EntityPathWrapperArgumentResolver;
import cn.bestwu.simpleframework.data.resolver.PagHandlerMethodArgumentResolver;
import cn.bestwu.simpleframework.support.packagescan.PackageScanClassResolver;
import cn.bestwu.simpleframework.web.CustomErrorController;
import cn.bestwu.simpleframework.web.ErrorAttributes;
import cn.bestwu.simpleframework.web.filter.OrderedHiddenHttpMethodFilter;
import cn.bestwu.simpleframework.web.filter.OrderedHttpPutFormContentFilter;
import cn.bestwu.simpleframework.web.kaptcha.KaptchaProperties;
import cn.bestwu.simpleframework.web.resolver.ModifyModelMethodArgumentResolver;
import cn.bestwu.simpleframework.web.resolver.StringToEnumConverterFactory;
import cn.bestwu.simpleframework.web.serializer.BigDecimalSerializer;
import cn.bestwu.simpleframework.web.serializer.MixIn;
import cn.bestwu.simpleframework.web.xss.XssFilter;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.google.code.kaptcha.Constants;
import com.google.code.kaptcha.Producer;
import com.google.code.kaptcha.impl.DefaultKaptcha;
import com.google.code.kaptcha.util.Config;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.lang.reflect.ParameterizedType;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.boot.autoconfigure.web.servlet.error.ErrorMvcAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.error.ErrorController;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.convert.converter.Converter;
import org.springframework.format.FormatterRegistry;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

/**
 * Rest MVC 配置
 *
 * @author Peter Wu
 */
@Configuration
@ConditionalOnWebApplication
@AutoConfigureBefore(ErrorMvcAutoConfiguration.class)
public class MvcConfiguration {

  private Logger log = LoggerFactory.getLogger(MvcConfiguration.class);

  /**
   * specifying the packages to scan for mixIn annotation.
   */
  @Value("${app.jackson.mix-in-annotation.basePackages:}")
  private String[] basePackages;

  @Bean(name = "error")
  @ConditionalOnMissingBean(name = "error")
  public View error() {
    return new View() {
      private ObjectMapper objectMapper = new ObjectMapper();

      @Override
      public String getContentType() {
        return "text/html;charset=utf-8";
      }

      @Override
      public void render(Map<String, ?> model, HttpServletRequest request,
          HttpServletResponse response)
          throws Exception {
        if (response.getContentType() == null) {
          response.setContentType(getContentType());
        }
        String result = objectMapper.writeValueAsString(model);
        response.getWriter().append(result);
      }
    };
  }

  @Bean
  public Module module(ApplicationContext applicationContext) {
    SimpleModule module = new SimpleModule();
    Set<String> packages = new HashSet<>();
    if (basePackages.length == 0) {
      for (Object o : applicationContext.getBeansWithAnnotation(ComponentScan.class).values()) {
        ComponentScan annotation = AnnotatedElementUtils
            .findMergedAnnotation(o.getClass(), ComponentScan.class);
        for (Class<?> aClass : annotation.basePackageClasses()) {
          packages.add(aClass.getPackage().getName());
        }
        packages.addAll(Arrays.asList(annotation.basePackages()));
        if (packages.isEmpty()) {
          packages.add(o.getClass().getPackage().getName());
        }
      }
    } else {
      packages.addAll(Arrays.asList(basePackages));
    }

    packages.add("cn.bestwu.simpleframework");
    basePackages = packages.toArray(new String[0]);

    Set<Class<?>> allSubClasses = new PackageScanClassResolver()
        .findImplementations(MixIn.class, basePackages);
    for (Class<?> aClass : allSubClasses) {
      ParameterizedType object = (ParameterizedType) aClass.getGenericInterfaces()[0];
      Class targetType = (Class) object.getActualTypeArguments()[0];
      if (log.isDebugEnabled()) {
        log.debug("setMixInAnnotation:{}=>{}", targetType, aClass);
      }
      module.setMixInAnnotation(targetType, aClass);
    }
    return module;
  }

  /*
   * 隐藏方法，网页支持
   */
  @Bean
  public OrderedHiddenHttpMethodFilter hiddenHttpMethodFilter() {
    return new OrderedHiddenHttpMethodFilter();
  }

  /*
   * Put方法，网页支持
   */
  @Bean
  public OrderedHttpPutFormContentFilter putFormContentFilter() {
    return new OrderedHttpPutFormContentFilter();
  }

  @Bean
  public XssFilter xssFilter() {
    return new XssFilter();
  }

  @ConditionalOnMissingBean(ErrorAttributes.class)
  @Bean
  public ErrorAttributes errorAttributes() {
    return new ErrorAttributes();
  }

  @ConditionalOnMissingBean(ErrorController.class)
  @Bean
  public CustomErrorController customErrorController(ErrorAttributes errorAttributes,
      ServerProperties serverProperties) {
    return new CustomErrorController(errorAttributes, serverProperties.getError());
  }

  @Configuration
  @ConditionalOnWebApplication
  protected static class ObjectMapperBuilderCustomizer implements
      Jackson2ObjectMapperBuilderCustomizer {

    @Override
    public void customize(Jackson2ObjectMapperBuilder jacksonObjectMapperBuilder) {
      jacksonObjectMapperBuilder.serializerByType(BigDecimal.class, new BigDecimalSerializer());
    }
  }

  @Configuration
  @ConditionalOnClass(DefaultKaptcha.class)
  @ConditionalOnWebApplication
  @EnableConfigurationProperties(KaptchaProperties.class)
  protected static class KaptchaConfiguration {

    @Bean
    @ConditionalOnMissingBean(Producer.class)
    public DefaultKaptcha kaptcha(KaptchaProperties kaptchaProperties) {
      Properties properties = new Properties();
      properties.put("kaptcha.border", kaptchaProperties.getBorder());
      properties
          .put("kaptcha.textproducer.font.color", kaptchaProperties.getTextproducerFontColor());
      properties
          .put("kaptcha.textproducer.char.space", kaptchaProperties.getTextproducerCharSpace());
      Config config = new Config(properties);
      DefaultKaptcha defaultKaptcha = new DefaultKaptcha();
      defaultKaptcha.setConfig(config);
      return defaultKaptcha;
    }

    @Controller
    protected static class CaptchaController {

      private Producer producer;

      public CaptchaController(Producer producer) {
        this.producer = producer;
      }

      @NoRequestLogging
      @GetMapping("/captcha.jpg")
      public void captcha(HttpServletRequest request, HttpServletResponse response)
          throws IOException {
        response.setHeader("Cache-Control", "no-store, no-cache");
        response.setContentType("image/jpeg");

        //生成文字验证码
        String text = producer.createText();
        //生成图片验证码
        BufferedImage image = producer.createImage(text);
        //保存到 session
        request.getSession().setAttribute(Constants.KAPTCHA_SESSION_KEY, text);

        ServletOutputStream out = response.getOutputStream();
        ImageIO.write(image, "jpg", out);
      }
    }
  }

  @Configuration
  @ConditionalOnWebApplication
  @EnableConfigurationProperties(WrapperBinderProperties.class)
  protected static class WebMvcConfiguration implements WebMvcConfigurer {

    private final Repositories repositories;
    private final WrapperBinderProperties properties;

    @Autowired
    public WebMvcConfiguration(Repositories repositories,
        WrapperBinderProperties properties) {
      this.repositories = repositories;
      this.properties = properties;
    }

    @Override
    public void addArgumentResolvers(List<HandlerMethodArgumentResolver> argumentResolvers) {
      argumentResolvers.add(new PagHandlerMethodArgumentResolver());
      argumentResolvers.add(new ModifyModelMethodArgumentResolver(repositories));
      argumentResolvers.add(new EntityPathWrapperArgumentResolver(repositories, properties));
    }


    /**
     * @param registry 注册转换类
     */
    @Override
    public void addFormatters(FormatterRegistry registry) {
      registry.addConverterFactory(new StringToEnumConverterFactory());
      registry.addConverter(new Converter<String, Date>() {
        @Override
        public Date convert(String source) {
          return new Date(Long.parseLong(source));
        }
      });
      registry.addConverter(new Converter<String, java.sql.Date>() {
        @Override
        public java.sql.Date convert(String source) {
          return new java.sql.Date(Long.parseLong(source));
        }
      });
    }
  }

}
