package cn.bestwu.simpleframework.web;

import cn.bestwu.simpleframework.exception.BusinessException;
import cn.bestwu.simpleframework.exception.ResourceNotFoundException;
import cn.bestwu.simpleframework.support.excel.ExcelImportException;
import cn.bestwu.simpleframework.support.excel.ExcelImportException.CellError;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.web.DefaultErrorAttributes;
import org.springframework.context.MessageSource;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.Order;
import org.springframework.core.convert.ConversionFailedException;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.transaction.TransactionSystemException;
import org.springframework.util.StringUtils;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.multipart.MultipartException;
import org.springframework.web.util.WebUtils;

/**
 * ErrorAttributes 错误属性
 *
 * @author Peter Wu
 */
@Order(Ordered.HIGHEST_PRECEDENCE)
public class ErrorAttributes extends DefaultErrorAttributes {

  protected final String KEY_STATUS = "status";
  protected final String KEY_MESSAGE = "message";
  private final Logger log = LoggerFactory.getLogger(ErrorAttributes.class);
  @Autowired(required = false)
  protected HttpServletRequest request;
  @Autowired
  private MessageSource messageSource;

  @Override
  public Map<String, Object> getErrorAttributes(RequestAttributes requestAttributes,
      boolean includeStackTrace) {
    String statusCode = null;
    Integer httpStatusCode = null;
    String message;
    Map<String, String> errors = new LinkedHashMap<>();
    Throwable e = getError(requestAttributes);
    Map<String, Object> errorAttributes = new LinkedHashMap<>();

    if (e != null) {
      message = e.getMessage();

      if (includeStackTrace) {
        addStackTrace(errorAttributes, e);
      }

      if (e instanceof ResourceNotFoundException || e instanceof EmptyResultDataAccessException) {
        httpStatusCode = HttpStatus.NOT_FOUND.value();
        if (!StringUtils.hasText(message)) {
          message = "resource.not.found";
        }
      } else if (e instanceof HttpRequestMethodNotSupportedException) {
        httpStatusCode = HttpStatus.METHOD_NOT_ALLOWED.value();
        if (!StringUtils.hasText(message)) {
          message = "method.not.allowed";
        }
      } else if (e instanceof BindException) {//参数错误
        httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
        BindException er = (BindException) e;
        List<FieldError> fieldErrors = er.getFieldErrors();
        for (FieldError fieldError : fieldErrors) {
          String defaultMessage = fieldError.getDefaultMessage();
          if (defaultMessage.contains("required type")) {
            defaultMessage = getText(fieldError.getCode());
          }
          errors.put(fieldError.getField(), getText(fieldError.getField()) + defaultMessage);

        }
        message = errors.values().iterator().next();

        if (!StringUtils.hasText(message)) {
          message = "data.valid.failed";
        }
      } else if (e instanceof IllegalArgumentException) {
        httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
        if (!StringUtils.hasText(message)) {
          message = "data.valid.failed";
        }
      } else if (e instanceof MissingServletRequestParameterException) {
        httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
      } else if (e instanceof ConversionFailedException) {
        httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
        message = getText("typeMismatch", ((ConversionFailedException) e).getValue(),
            ((ConversionFailedException) e).getTargetType());
        if (!StringUtils.hasText(message)) {
          message = "data.valid.failed";
        }
      } else if (e instanceof ConstraintViolationException
          || e instanceof org.springframework.transaction.TransactionSystemException) {//数据验证
        if (e instanceof org.springframework.transaction.TransactionSystemException) {
          e = ((TransactionSystemException) e).getRootCause();
        }

        if (e instanceof ConstraintViolationException) {
          httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();

          ConstraintViolationException er = (ConstraintViolationException) e;
          Set<ConstraintViolation<?>> constraintViolations = er.getConstraintViolations();
          for (ConstraintViolation<?> constraintViolation : constraintViolations) {
            errors
                .put(constraintViolation.getPropertyPath().toString(),
                    getText(constraintViolation.getPropertyPath()) + constraintViolation
                        .getMessage());
          }
          message = errors.values().iterator().next();

          if (!StringUtils.hasText(message)) {
            message = "data.valid.failed";
          }
        }
      } else if (e instanceof DataIntegrityViolationException) {
        String specificCauseMessage = ((DataIntegrityViolationException) e).getMostSpecificCause()
            .getMessage();
        String duplicateRegex = "^Duplicate entry '(.*?)'.*";
        String constraintSubfix = "Cannot delete or update a parent row";
        if (specificCauseMessage.matches(duplicateRegex)) {
          httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
          message = getText("duplicate.entry",
              specificCauseMessage.replaceAll(duplicateRegex, "$1"));
          if (!StringUtils.hasText(message)) {
            message = "data.valid.failed";
          }
        } else if (specificCauseMessage.startsWith(constraintSubfix)) {
          httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
          message = "cannot.delete.update.parent";
          if (!StringUtils.hasText(message)) {
            message = "data.valid.failed";
          }
        } else {
          message = ((DataIntegrityViolationException) e).getRootCause().getMessage();
        }
      } else if (e instanceof HttpMediaTypeNotAcceptableException) {
        httpStatusCode = HttpStatus.NOT_ACCEPTABLE.value();
        message =
            "MediaType not Acceptable!Must ACCEPT:" + ((HttpMediaTypeNotAcceptableException) e)
                .getSupportedMediaTypes();
      } else if (e instanceof IllegalStateException) {
        httpStatusCode = HttpStatus.CONFLICT.value();
      } else if (e instanceof MultipartException) {
        httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
        message = "upload.fail";
      } else if (e instanceof HttpMessageNotWritableException) {
        if (message.contains("Session is closed")) {
          httpStatusCode = HttpStatus.REQUEST_TIMEOUT.value();
          message = "request.timeout";
        }
      } else if (e instanceof ExcelImportException) {
        httpStatusCode = HttpStatus.UNPROCESSABLE_ENTITY.value();
        List<CellError> cellErrors = ((ExcelImportException) e).getErrors();

        for (CellError cellError : cellErrors) {
          String key =
              "第" + cellError.getRow() + "行第" + ((char) ('A' + cellError.getColumn())) + "列";
          ConstraintViolationException value = cellError.getException();
          for (ConstraintViolation<?> constraintViolation : value.getConstraintViolations()) {
            errors.put(key, getText(constraintViolation.getPropertyPath()) + constraintViolation
                .getMessage());
          }
        }
        Entry<String, String> firstError = errors.entrySet().iterator().next();
        message = firstError.getKey() + ": " + firstError.getValue();
      } else if (e instanceof BusinessException) {
        statusCode = ((BusinessException) e).getCode();
      }
    } else {
      message = getAttribute(requestAttributes, WebUtils.ERROR_MESSAGE_ATTRIBUTE);
    }

    if (httpStatusCode == null) {
      if (e != null) {
        ResponseStatus responseStatus = AnnotatedElementUtils
            .findMergedAnnotation(e.getClass(), ResponseStatus.class);
        if (responseStatus != null) {
          httpStatusCode = responseStatus.code().value();
          String reason = responseStatus.reason();
          if (StringUtils.hasText(reason)) {
            message = reason;
          }
        }
      }
      if (httpStatusCode == null) {
        httpStatusCode = getStatus(requestAttributes).value();
        if (httpStatusCode != 406 && httpStatusCode != 401 && httpStatusCode != 403
            && httpStatusCode != 404 && httpStatusCode != 405) {
          if (e != null) {
            log.error(httpStatusCode + ":" + message, e);
          } else {
            log.warn(httpStatusCode + ":" + message);
          }
        }
      }
    }
    setStatus(requestAttributes, httpStatusCode);

    statusCode = statusCode == null ? String.valueOf(httpStatusCode) : statusCode;
    message = StringUtils.hasText(message) ? message : "No message available";
    message = getText(message);

    errorAttributes.put(KEY_STATUS, statusCode);
    errorAttributes.put(KEY_MESSAGE, message);
    if (!errors.isEmpty()) {
      errorAttributes.put("errors", errors);
    }

    return errorAttributes;
  }

  /**
   * 增加StackTrace
   *
   * @param errorAttributes errorAttributes
   * @param error error
   */
  private void addStackTrace(Map<String, Object> errorAttributes, Throwable error) {
    StringWriter stackTrace = new StringWriter();
    error.printStackTrace(new PrintWriter(stackTrace));
    stackTrace.flush();
    errorAttributes.put("trace", stackTrace.toString());
  }

  private void setStatus(RequestAttributes requestAttributes, Object statusCode) {
    requestAttributes.setAttribute(WebUtils.ERROR_STATUS_CODE_ATTRIBUTE, statusCode,
        RequestAttributes.SCOPE_REQUEST);
  }

  private HttpStatus getStatus(RequestAttributes requestAttributes) {
    Integer statusCode = getAttribute(requestAttributes, WebUtils.ERROR_STATUS_CODE_ATTRIBUTE);
    if (statusCode != null) {
      try {
        return HttpStatus.valueOf(statusCode);
      } catch (Exception ignored) {
      }
    }
    return HttpStatus.INTERNAL_SERVER_ERROR;
  }

  @SuppressWarnings("unchecked")
  private <T> T getAttribute(RequestAttributes requestAttributes, String name) {
    return (T) requestAttributes.getAttribute(name, RequestAttributes.SCOPE_REQUEST);
  }

  /**
   * 得到国际化信息 未找到时返回代码 code
   *
   * @param code 模板
   * @param args 参数
   * @return 信息
   */
  private String getText(Object code, Object... args) {
    String codeString = String.valueOf(code);
    return messageSource.getMessage(codeString, args, codeString,
        request == null ? Locale.CHINA : request.getLocale());
  }
}
