package cn.bestwu.simpleframework.support.excel;

import cn.bestwu.simpleframework.support.excel.ExcelImportException.CellError;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Set;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.groups.Default;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DateUtil;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

/**
 * 导入Excel文件（支持“XLS”和“XLSX”格式）
 */
public class ExcelImport {

  private static Logger log = LoggerFactory.getLogger(ExcelImport.class);
  private static Validator validator = Validation.buildDefaultValidatorFactory().getValidator();
  /**
   * 工作表对象
   */
  private Sheet sheet;
  /**
   * 验证 groups
   */
  private Class<?>[] validateGroups = new Class[]{Default.class};
  /**
   * 标题行号
   */
  private int headerNum;

  /**
   * 构造函数
   *
   * @param fileName 导入文件，读取第一个工作表
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @throws IOException IOException
   */
  public ExcelImport(String fileName, int headerNum)
      throws IOException {
    this(new File(fileName), headerNum);
  }

  /**
   * 构造函数
   *
   * @param file 导入文件对象，读取第一个工作表
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @throws IOException IOException
   */
  public ExcelImport(File file, int headerNum)
      throws IOException {
    this(file, headerNum, 0);
  }

  /**
   * 构造函数
   *
   * @param fileName 导入文件
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @param sheetIndex 工作表编号
   * @throws IOException IOException
   */
  public ExcelImport(String fileName, int headerNum, int sheetIndex)
      throws IOException {
    this(new File(fileName), headerNum, sheetIndex);
  }

  /**
   * 构造函数
   *
   * @param file 导入文件对象
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @param sheetIndex 工作表编号
   * @throws IOException IOException
   */
  public ExcelImport(File file, int headerNum, int sheetIndex)
      throws IOException {
    this(file.getName(), new FileInputStream(file), headerNum, sheetIndex);
  }

  /**
   * 构造函数
   *
   * @param multipartFile 导入文件对象
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @param sheetIndex 工作表编号
   * @throws IOException IOException
   */
  public ExcelImport(MultipartFile multipartFile, int headerNum, int sheetIndex)
      throws IOException {
    this(multipartFile.getOriginalFilename(), multipartFile.getInputStream(), headerNum,
        sheetIndex);
  }

  /**
   * 构造函数
   *
   * @param is is
   * @param fileName 导入文件对象
   * @param headerNum 标题行号，数据行号=标题行号+1
   * @param sheetIndex 工作表编号
   * @throws IOException IOException
   */
  public ExcelImport(String fileName, InputStream is, int headerNum, int sheetIndex)
      throws IOException {
    /*
    工作薄对象
   */
    Workbook wb;
    if (!StringUtils.hasText(fileName)) {
      throw new RuntimeException("导入文档为空!");
    } else if (fileName.toLowerCase().endsWith("xls")) {
      wb = new HSSFWorkbook(is);
    } else if (fileName.toLowerCase().endsWith("xlsx")) {
      wb = new XSSFWorkbook(is);
    } else {
      throw new RuntimeException("文档格式不正确!");
    }
    if (wb.getNumberOfSheets() < sheetIndex) {
      throw new RuntimeException("文档中没有工作表!");
    }
    this.sheet = wb.getSheetAt(sheetIndex);
    this.headerNum = headerNum;
    log.debug("Initialize success.");
  }

  /**
   * 获取行对象
   *
   * @param rownum rownum
   * @return Row
   */
  public Row getRow(int rownum) {
    return this.sheet.getRow(rownum);
  }

  /**
   * 获取数据行号
   *
   * @return DataRowNum
   */
  public int getDataRowNum() {
    return headerNum + 1;
  }

  /**
   * 获取最后一个数据行号
   *
   * @return RowNum
   */
  public int getLastDataRowNum() {
    return this.sheet.getLastRowNum() + headerNum + 1;
  }

  /**
   * 获取最后一个列号
   *
   * @return cellNum
   */
  public int getLastCellNum() {
    return this.getRow(headerNum).getLastCellNum();
  }

  /**
   * 获取单元格值
   *
   * @param row 获取的行
   * @param column 获取单元格列号
   * @return 单元格值
   */
  public Object getCellValue(Row row, int column) {
    Object val = "";
    try {
      Cell cell = row.getCell(column);
      if (cell != null) {
        if (cell.getCellTypeEnum() == CellType.NUMERIC) {
          val = cell.getNumericCellValue();
        } else if (cell.getCellTypeEnum() == CellType.STRING) {
          val = cell.getStringCellValue();
        } else if (cell.getCellTypeEnum() == CellType.FORMULA) {
          val = cell.getCellFormula();
        } else if (cell.getCellTypeEnum() == CellType.BOOLEAN) {
          val = cell.getBooleanCellValue();
        } else if (cell.getCellTypeEnum() == CellType.ERROR) {
          val = cell.getErrorCellValue();
        }
      }
    } catch (Exception e) {
      log.warn(row + "，" + column + "，读取数据错误", e);
      return val;
    }
    return val;
  }

  /**
   * 获取导入数据列表
   *
   * @param cls 导入对象类型
   * @param groups 导入分组
   * @param <E> E
   * @return List
   * @throws InstantiationException InstantiationException
   * @throws IllegalAccessException IllegalAccessException
   * @throws NoSuchMethodException NoSuchMethodException
   * @throws ExcelImportException ExcelImportException
   */
  public <E> List<E> getDataList(Class<E> cls, int... groups)
      throws InstantiationException, IllegalAccessException, NoSuchMethodException, ExcelImportException {
    List<Object[]> annotationList = new ArrayList<>();
    // Get annotation field
    Field[] fs = cls.getDeclaredFields();
    for (Field f : fs) {
      ExcelField ef = f.getAnnotation(ExcelField.class);
      if (ef != null && (ef.type() == 0 || ef.type() == 2)) {
        if (groups != null && groups.length > 0) {
          boolean inGroup = false;
          for (int g : groups) {
            if (inGroup) {
              break;
            }
            for (int efg : ef.groups()) {
              if (g == efg) {
                inGroup = true;
                annotationList.add(new Object[]{ef, f});
                break;
              }
            }
          }
        } else {
          annotationList.add(new Object[]{ef, f});
        }
      }
    }
    // Get annotation method
    Method[] ms = cls.getDeclaredMethods();
    for (Method m : ms) {
      ExcelField ef = m.getAnnotation(ExcelField.class);
      if (ef != null && (ef.type() == 0 || ef.type() == 2)) {
        if (groups != null && groups.length > 0) {
          boolean inGroup = false;
          for (int g : groups) {
            if (inGroup) {
              break;
            }
            for (int efg : ef.groups()) {
              if (g == efg) {
                inGroup = true;
                annotationList.add(new Object[]{ef, m});
                break;
              }
            }
          }
        } else {
          annotationList.add(new Object[]{ef, m});
        }
      }
    }
    // Field sorting
    annotationList.sort(Comparator.comparingInt(o -> ((ExcelField) o[0]).sort()));
    //log.debug("Import column count:"+annotationList.size());
    // Get excel data
    List<CellError> errors = new ArrayList<>();

    List<E> dataList = new ArrayList<>();
    for (int i = this.getDataRowNum(); i < this.getLastDataRowNum(); i++) {
      E e = cls.newInstance();
      int column = 0;
      Row row = this.getRow(i);
      StringBuilder sb = new StringBuilder();
      for (Object[] os : annotationList) {
        Object val = this.getCellValue(row, column++);
        if (val != null) {
          ExcelField ef = (ExcelField) os[0];
          // Get param type and type cast
          Class<?> valType = Class.class;
          if (os[1] instanceof Field) {
            valType = ((Field) os[1]).getType();
          } else if (os[1] instanceof Method) {
            Method method = ((Method) os[1]);
            if ("get".equals(method.getName().substring(0, 3))) {
              valType = method.getReturnType();
            } else if ("set".equals(method.getName().substring(0, 3))) {
              valType = ((Method) os[1]).getParameterTypes()[0];
            }
          }
          //log.debug("Import value type: ["+i+","+column+"] " + valType);
          try {
            if (valType == String.class) {
              String s = String.valueOf(val.toString());
              if (StringUtils.endsWithIgnoreCase(s, ".0")) {
                val = s.substring(0, s.lastIndexOf(".0"));
              } else if (StringUtils.endsWithIgnoreCase(s, "E10")) {
                val = s.substring(0, s.lastIndexOf("E10")).replace(".", "");
              } else {
                val = String.valueOf(val.toString());
              }
            } else if (valType == Integer.class) {
              val = Double.valueOf(val.toString()).intValue();
            } else if (valType == Long.class) {
              val = Double.valueOf(val.toString()).longValue();
            } else if (valType == Double.class) {
              val = Double.valueOf(val.toString());
            } else if (valType == Float.class) {
              val = Float.valueOf(val.toString());
            } else if (valType == Date.class) {
              val = DateUtil.getJavaDate((Double) val);
            } else {
              if (ef.fieldType() != Class.class) {
                val = ef.fieldType().getMethod("getValue", String.class)
                    .invoke(null, val.toString());
              } else {
                val = Class
                    .forName(this.getClass().getName().replaceAll(this.getClass().getSimpleName(),
                        "fieldtype." + valType.getSimpleName() + "Type"))
                    .getMethod("getValue", String.class).invoke(null, val.toString());
              }
            }
          } catch (Exception ex) {
            log.info("Get cell value [" + i + "," + column + "] error: " + ex.toString());
            val = null;
          }
          // set entity value
          String propertyName = null;
          if (os[1] instanceof Field) {
            propertyName = ((Field) os[1]).getName();
            ReflectionUtils.invokeMethod(
                BeanUtils.getPropertyDescriptor(e.getClass(), propertyName)
                    .getWriteMethod(), e, val);
          } else if (os[1] instanceof Method) {
            String mthodName = ((Method) os[1]).getName();
            if ("get".equals(mthodName.substring(0, 3))) {
              propertyName = mthodName.substring(3, mthodName.length());
              mthodName = "set" + propertyName;
              propertyName = StringUtils.uncapitalize(propertyName);
            }

            ReflectionUtils.invokeMethod(cls.getMethod(mthodName, valType), e, val);
          }
          ConstraintViolationException exception = validateProperty(e, propertyName,
              validateGroups);
          if (exception != null && errors.size() < 10) {//最多显示10条错误
            errors.add(new CellError(i + headerNum + 1, ef.sort(), exception));
            if (errors.size() == 10) {
              throw new ExcelImportException(errors);
            }
          }
        }
        sb.append(val).append(", ");
      }
      dataList.add(e);
      log.debug("Read success: [" + i + "] " + sb.toString());
    }
    if (!errors.isEmpty()) {
      throw new ExcelImportException(errors);
    }
    return dataList;
  }

  /**
   * 校验对象
   *
   * @param object 待校验对象
   * @param groups 待校验的组
   */
  private ConstraintViolationException validateProperty(Object object, String propertyName,
      Class<?>... groups) {
    Set<ConstraintViolation<Object>> constraintViolations = validator
        .validateProperty(object, propertyName, groups);
    if (constraintViolations.isEmpty()) {
      return null;
    } else {
      return new ConstraintViolationException(constraintViolations);
    }
  }

  /**
   * @param validateGroups 验证 groups
   */
  public void validateGroups(Class<?>... validateGroups) {
    this.validateGroups = validateGroups;
  }
}
