package cn.bestwu.simpleframework.data;

import cn.bestwu.simpleframework.data.binding.BinderCustomizer;
import cn.bestwu.simpleframework.data.dsl.EntityPathWrapper;
import com.baomidou.mybatisplus.entity.TableFieldInfo;
import com.baomidou.mybatisplus.entity.TableInfo;
import com.baomidou.mybatisplus.mapper.BaseMapper;
import com.baomidou.mybatisplus.toolkit.TableInfoHelper;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.springframework.util.Assert;

/**
 * 资源元数据
 *
 * @author Peter Wu
 */
@SuppressWarnings("unchecked")
public class RepositoryMetadata {

  private final Object repository;
  private final Map<String, String> cachedFieldsMap;
  private Class<?> modelClass;
  private Class<? extends Serializable> idType;
  private Class<? extends EntityPathWrapper<?, ?>> queryDslType;

  //  	private Method findAllMethod;
  private Method wrapperBinderMethod;
  private Method findOneMethod;
  //	private Method saveMethod;
  //	private Method deleteMethod;
  //--------------------------------------------

  /**
   * 创建资源元数据
   *
   * @param repository 仓库
   * @param repositoryInterface repositoryInterface
   * @throws NoSuchMethodException NoSuchMethodException
   * @throws NoSuchFieldException NoSuchFieldException
   */
  public RepositoryMetadata(BaseMapper repository,
      Class<? extends BaseMapper> repositoryInterface)
      throws NoSuchMethodException, NoSuchFieldException {
    this.repository = repository;

    init(repositoryInterface);
    Assert.notNull(modelClass, "未成功设置 modelClass，repository:" + repositoryInterface.getName()
        + " 未继承 " + BaseMapper.class.getTypeName());

    findOneMethod = repositoryInterface.getMethod("selectById", Serializable.class);
    Map<String, String> cachedFieldsMap = new HashMap<>();
    TableInfo tableInfo = TableInfoHelper.getTableInfo(modelClass);
    String keyProperty = tableInfo.getKeyProperty();
    cachedFieldsMap.put(keyProperty, tableInfo.getKeyColumn());
    for (TableFieldInfo tableFieldInfo : tableInfo.getFieldList()) {
      cachedFieldsMap.put(tableFieldInfo.getProperty(), tableFieldInfo.getColumn());
    }
    this.idType = (Class<? extends Serializable>) modelClass.getDeclaredField(keyProperty)
        .getType();
    this.cachedFieldsMap = Collections.unmodifiableMap(cachedFieldsMap);
  }

  private void init(Class<?> repositoryInterface) throws NoSuchMethodException {
    Type[] genericInterfaces = repositoryInterface.getGenericInterfaces();
    boolean hasBinderCustomizer = BinderCustomizer.class.isAssignableFrom(repositoryInterface);

    for (Type genericInterface : genericInterfaces) {
      if (genericInterface instanceof ParameterizedType && ((ParameterizedType) genericInterface)
          .getRawType().getTypeName().equals(BaseMapper.class.getName())) {
        Type[] typeArguments = ((ParameterizedType) genericInterface)
            .getActualTypeArguments();
        modelClass = (Class) typeArguments[0];
      } else if (hasBinderCustomizer && genericInterface instanceof ParameterizedType
          && ((ParameterizedType) genericInterface).getRawType().getTypeName()
          .equals(BinderCustomizer.class.getName())) {
        Type[] typeArguments = ((ParameterizedType) genericInterface).getActualTypeArguments();
        queryDslType = (Class) typeArguments[0];
        wrapperBinderMethod = repositoryInterface.getMethod("customize", queryDslType);
        break;
      } else if (genericInterface instanceof Class
          && ((Class) genericInterface).getGenericInterfaces().length > 0) {
        init((Class<?>) genericInterface);
      }
    }
  }

  //--------------------------------------------

  public Object invokeFindOne(Serializable id)
      throws InvocationTargetException, IllegalAccessException {
    return findOneMethod.invoke(repository, id);
  }

  public void invokeCustomize(EntityPathWrapper<?, ?> entityPathWrapper)
      throws InvocationTargetException, IllegalAccessException {
    if (wrapperBinderMethod != null) {
      wrapperBinderMethod.invoke(repository, entityPathWrapper);
    }
  }

  //--------------------------------------------

  public Class<?> getModelClass() {
    return modelClass;
  }

  public Class<? extends Serializable> getIdType() {
    return idType;
  }

  public Method getWrapperBinderMethod() {
    return wrapperBinderMethod;
  }

  public Method getFindOneMethod() {
    return findOneMethod;
  }

  public Object getRepository() {
    return repository;
  }

  public Map<String, String> getCachedFieldsMap() {
    return cachedFieldsMap;
  }

  public Class<? extends EntityPathWrapper<?, ?>> getQueryDslType() {
    return queryDslType;
  }

  public EntityPathWrapper<?, ?> getWrapper()
      throws IllegalAccessException, InstantiationException {
    return queryDslType == null ? new EntityPathWrapper<>() : queryDslType.newInstance();
  }
}
