package cn.bestwu.simpleframework.config;

import java.nio.charset.Charset;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.boot.autoconfigure.context.MessageSourceAutoConfiguration;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.ResourceBundleMessageSource;
import org.springframework.core.Ordered;
import org.springframework.util.StringUtils;

@ConditionalOnWebApplication
@Configuration
@AutoConfigureBefore(MessageSourceAutoConfiguration.class)
@ConditionalOnMissingBean(value = MessageSource.class, search = SearchStrategy.CURRENT)
@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE)
@EnableConfigurationProperties
@ConfigurationProperties(prefix = "spring.messages")
public class MessageSourceConfiguration {

  public static final String BASE_MESSAGES = "base-messages";

  /**
   * Comma-separated list of basenames, each following the ResourceBundle convention.
   * Essentially a fully-qualified classpath location. If it doesn't contain a package
   * qualifier (such as "org.mypackage"), it will be resolved from the classpath root.
   */
  private String basename = "messages";

  /**
   * Message bundles encoding.
   */
  private Charset encoding = Charset.forName("UTF-8");

  /**
   * Loaded resource bundle files cache expiration, in seconds. When set to -1, bundles
   * are cached forever.
   */
  private int cacheSeconds = -1;

  /**
   * Set whether to fall back to the system Locale if no files for a specific Locale
   * have been found. if this is turned off, the only fallback will be the default file
   * (e.g. "messages.properties" for basename "messages").
   */
  private boolean fallbackToSystemLocale = true;

  /**
   * Set whether to always apply the MessageFormat rules, parsing even messages without
   * arguments.
   */
  private boolean alwaysUseMessageFormat = false;

  @Bean
  public MessageSource messageSource() {
    ResourceBundleMessageSource messageSource = new ResourceBundleMessageSource();
    if (StringUtils.hasText(this.basename)) {
      messageSource.setBasenames(StringUtils.commaDelimitedListToStringArray(
          StringUtils.trimAllWhitespace(this.basename)));
    }
    messageSource.addBasenames(BASE_MESSAGES);
    if (this.encoding != null) {
      messageSource.setDefaultEncoding(this.encoding.name());
    }
    messageSource.setFallbackToSystemLocale(this.fallbackToSystemLocale);
    messageSource.setCacheSeconds(this.cacheSeconds);
    messageSource.setAlwaysUseMessageFormat(this.alwaysUseMessageFormat);
    return messageSource;
  }

  public String getBasename() {
    return this.basename;
  }

  public void setBasename(String basename) {
    this.basename = basename;
  }

  public Charset getEncoding() {
    return this.encoding;
  }

  public void setEncoding(Charset encoding) {
    this.encoding = encoding;
  }

  public int getCacheSeconds() {
    return this.cacheSeconds;
  }

  public void setCacheSeconds(int cacheSeconds) {
    this.cacheSeconds = cacheSeconds;
  }

  public boolean isFallbackToSystemLocale() {
    return this.fallbackToSystemLocale;
  }

  public void setFallbackToSystemLocale(boolean fallbackToSystemLocale) {
    this.fallbackToSystemLocale = fallbackToSystemLocale;
  }

  public boolean isAlwaysUseMessageFormat() {
    return this.alwaysUseMessageFormat;
  }

  public void setAlwaysUseMessageFormat(boolean alwaysUseMessageFormat) {
    this.alwaysUseMessageFormat = alwaysUseMessageFormat;
  }
}
