package cn.benma666.sjzt.mqtt;

import cn.benma666.domain.SysSjglSjzt;
import cn.benma666.domain.SysSjglZnjh;
import cn.benma666.exception.MyException;
import cn.benma666.iframe.InterfaceLog;
import cn.benma666.iframe.Result;
import cn.benma666.myutils.ClassUtil;
import cn.benma666.myutils.FileUtil;
import cn.benma666.sjzt.*;
import com.alibaba.druid.util.Utils;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.eclipse.paho.client.mqttv3.*;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;

/**
 * mqtt数据载体
 */
public class Mqtt extends BasicSjzt {

    /**
     * 默认客户端
     */
    private static Mqtt mqtt = null;
    /**
     * 客户端对象池
     */
    private final GenericObjectPool objectPool;

    protected Mqtt(String name, SysSjglSjzt sjzt) {
        //参考hdfs
        super(name, sjzt);
        GenericObjectPoolConfig<MqttClient> conf = new GenericObjectPoolConfig<>();
        conf.setTestOnCreate(true);
        conf.setTestOnBorrow(true);
        conf.setMinIdle(1);
        conf.setMaxIdle(2);
        conf.setMaxTotal(5);
        //最大等待60秒
        conf.setMaxWait(Duration.ofSeconds(300));
        //支持在载体中配置
        ClassUtil.plMethodInvoke(conf,getSjzt().getKzxxObj().getJSONObject("ljcpz"));
        objectPool = new GenericObjectPool(new SjztPooledObjectFactory(sjzt), conf);
        MqttClient client = null;
        try {
            //测试获取连接
            client = borrowClient();
        } catch (Exception e) {
            throw new MyException(name + "初始化失败", e);
        } finally {
            returnClient(client);
        }
        if (mqtt == null) {
            mqtt = this;
        }
        cache.put(name,this);
    }
    /**
     * 退回客户端
     * @param client 需要退回的客户端
     */
    public void returnClient(MqttClient client) {
        log.trace("释放回连接池：{}",name);
        if (client != null) {
            objectPool.returnObject(client);
        }
    }

    /**
     * 获取客户端
     */
    public MqttClient borrowClient() throws Exception {
        log.trace("从连接池获取：{}",name);
        return (MqttClient) objectPool.borrowObject();
    }

    public static Mqtt use(String name) {
        return use(name,getSjzt(name));
    }
    public static Mqtt use(String name,SysSjglSjzt sjzt) {
        Mqtt mqtt = (Mqtt) cache.get(name);
        if (mqtt == null) {
            mqtt = new Mqtt(name, sjzt);
        }
        return mqtt;
    }
    public static Result cszt(SysSjglSjzt sjzt) {
        try {
            Result r = success("测试成功");
            MqttClient client = createClient(sjzt);
            if(!validateClient(sjzt,client)){
                r = failed("测试失败");
            }
            destroyClient(sjzt,client);
            return r;
        }catch (Throwable t){
            slog.debug("{}测试失败",sjzt,t);
            return failed("载体测试不通过："+t.getMessage());
        }
    }

    /**
     * 验证客户端是否可用
     */
    public static boolean validateClient(SysSjglSjzt sjzt,Object client) {
        try {
            return ((MqttClient) client).isConnected();
        }catch (Throwable t){
            slog.debug("验证无效：{}",sjzt.getMc(),t);
            return false;
        }
    }

    /**
     * 关闭客户端
     */
    public static void destroyClient(SysSjglSjzt sjzt,Object client) throws Exception {
        MqttClient mqttClient = (MqttClient) client;
        mqttClient.disconnect();
        mqttClient.close();
    }
    public static MqttClient createClient(SysSjglSjzt sjzt) {
        try {
            //创建MQTT客户端对象
            MqttClient mqttClient = new MqttClient(sjzt.getLjc(), sjzt.getDm(), new MemoryPersistence());
            //连接设置
            MqttConnectOptions options = new MqttConnectOptions();
            //是否清空session，设置false表示服务器会保留客户端的连接记录（订阅主题，qos）,客户端重连之后能获取到服务器在客户端断开连接期间推送的消息
            //设置为true表示每次连接服务器都是以新的身份
            options.setCleanSession(false);
            //设置连接用户名
            options.setUserName(sjzt.getYhm());
            //设置连接密码
            options.setPassword(sjzt.getMm().toCharArray());
            //设置超时时间，单位为秒
            options.setConnectionTimeout(100);
            //设置心跳时间 单位为秒，表示服务器每隔 1.5*20秒的时间向客户端发送心跳判断客户端是否在线
            options.setKeepAliveInterval(20);
            //设置遗嘱消息的话题，若客户端和服务器之间的连接意外断开，服务器将发布客户端的遗嘱信息
            options.setWill("willTopic", (sjzt.getDm() + "与服务器断开连接").getBytes(), 0, false);
            //设置回调
            mqttClient.setCallback(new MqttCallBack());
            JSONObject mttpz = sjzt.getKzxxObj().getJSONObject("mttpz");
            if(mttpz!=null){
                ClassUtil.plMethodInvoke(mqttClient,mttpz.getJSONObject("MqttClient"));
                ClassUtil.plMethodInvoke(options,mttpz.getJSONObject("MqttConnectOptions"));
            }
            mqttClient.connect(options);
            if(mqttClient.isConnected()){
                return mqttClient;
            }else {
                throw new MyException("mqtt连接失败，未知异常");
            }
        } catch (MqttException e) {
            throw new MyException("mqtt数据载体创建失败："+sjzt.getDm(),e);
        }
    }

    public Object exec(SjztExecRunnable<MqttClient> exec) {
        MqttClient client = null;
        try {
            client = borrowClient();
            return exec.exec(client);
        } catch (Exception e) {
            throw new MyException("执行异常", e);
        } finally {
            if (client != null) {
                objectPool.returnObject(client);
            }
        }
    }

    @Override
    public List<IFile> listFiles(SysSjglZnjh znjhConfig) throws Exception {
        throw new MyException("mqtt不支持获取文件列表");
    }

    @Override
    public InputStream getInputStream(IFile file) throws Exception {
        //才开ftp封装消费者为输入流
        if(file instanceof MqttFile){
            return new ByteArrayInputStream(((MqttFile)file).getFile().getPayload());
        }
        throw new MyException("不支持非mqttfile");
    }

    @Override
    public boolean delete(IFile file) throws Exception {
        if(file instanceof MqttFile){
            //消费后就不会读取到了
            return true;
        }
        throw new MyException("不支持非Mqttfile");
    }


    /**
     * 订阅消息
     * @param topicFilter 主题
     * @param messageListener 消息监听器
     */
    public void subscribe(String topicFilter, IMqttMessageListener messageListener){
        subscribe(topicFilter,MqttQos.ZHYC.getCode(),messageListener);
    }

    /**
     * 订阅消息
     * @param topicFilter 主题
     * @param qos 质量
     * @param messageListener 消息监听器
     */
    public void subscribe(String topicFilter, int qos, IMqttMessageListener messageListener){
        exec(client -> {
            client.subscribe(topicFilter,qos,messageListener);
            return null;
        });
    }
    /**
     * 发布消息
     * @param topic 主题
     * @param msg 消息内容
     * @return 发布状态
     * @throws Exception 异常
     */
    public boolean pub(String topic,String msg) throws Exception {
        return pub(topic,msg,MqttQos.ZSYC);
    }

    /**
     * 发布消息
     * @param topic 主题
     * @param msg 消息内容
     * @param qos 消息质量
     * @return 发布状态
     * @throws Exception 异常
     */
    public boolean pub(String topic,String msg,MqttQos qos) throws Exception {
        return save(new ByteArrayInputStream(msg.getBytes(StandardCharsets.UTF_8)),
                new MqttFile(topic,qos));
    }

    @Override
    public boolean save(InputStream is, IFile file) throws Exception {
        return (boolean) exec(client -> {
            //默认至少一次
            MqttQos qos = MqttQos.ZSYC;
            if(file instanceof MqttFile){
                qos = ((MqttFile) file).getQos();
            }
            String topic = file.getParent();
            try {
                MqttMessage mqttMessage = new MqttMessage();
                mqttMessage.setQos(qos.getCode());
                mqttMessage.setPayload(Utils.readByteArray(is));
                MqttTopic mqttTopic = client.getTopic(topic);
                MqttDeliveryToken token = mqttTopic.publish(mqttMessage);
                token.waitForCompletion();
                return true;
            }catch (Throwable t){
                throw new MyException("消息推送异常：" + topic,t);
            } finally {
                FileUtil.closeStream(is);
            }
        });
    }

    @Override
    public String getRootPath() {
        return this.sjzt.getDxgsNew();
    }

    @Override
    public long getSize(IFile file) throws Exception {
        if(file instanceof MqttFile){
            return ((MqttFile)file).getFile().getPayload().length;
        }
        throw new SjztBzcwjdxExecption("不支持非mqttfile");
    }

    @Override
    public void sjztjt(SysSjglZnjh znjhConfig, InterfaceLog log) {
        //从消费者获取所有历史消息
        exec(client -> {
            String topic = znjhConfig.getSrml();
            try {
                client.subscribe(topic,MqttQos.ZHYC.getCode(),
                        (s, mqttMessage) -> {
                            IFile file = new MqttFile(topic, mqttMessage, this,
                                    s.replace("/","_")+"_"+ mqttMessage.getId()+".json");
                            file.setGzml(topic);
//                            log.debug("收到mqtt消息：{}->{}",mqttMessage.getId(), new String(mqttMessage.getPayload()));
                            znjhConfig.getTp().run(() -> znjh.znjh(znjhConfig, log,file));
                        });
                while (client.isConnected()) {
                    Thread.sleep(100000L);
                }
            } catch (InterruptedException e){
                log.info("退出遍历监听：{}",znjhConfig.getJhmc());
                //取消订阅
                client.unsubscribe(topic);
            } catch (Throwable e) {
                log.error("遍历监听异常："+znjhConfig.getJhmc()+"，"+e.getMessage(),e);
                //取消订阅
                client.unsubscribe(topic);
            }
            jtqMap.remove(znjhConfig.getId());
            return null;
        });
    }

    @Override
    public void close() throws IOException {
        //关闭所有连接
        if (!this.objectPool.isClosed()) {
            this.objectPool.close();
        }
        cache.remove(name);
        if (this == mqtt) {
            mqtt = null;
        }
    }
}
