/*
 Project Name:myutils
 Date:2017年5月31日
 Copyright (c) 2017, jingma All Rights Reserved.
*/

package cn.benma666.sjzt;

import cn.benma666.constants.UtilConst;
import cn.benma666.crypt.DesUtil;
import cn.benma666.domain.SysQxYhxx;
import cn.benma666.domain.SysSjglBlob;
import cn.benma666.domain.SysSjglSjzt;
import cn.benma666.domain.SysSjglZnjh;
import cn.benma666.exception.MyException;
import cn.benma666.exception.SuccessException;
import cn.benma666.json.JsonUtil;
import cn.benma666.iframe.Conf;
import cn.benma666.iframe.MyParams;
import cn.benma666.iframe.PageInfo;
import cn.benma666.iframe.Result;
import cn.benma666.myutils.*;
import com.alibaba.druid.DbType;
import com.alibaba.druid.filter.Filter;
import com.alibaba.druid.filter.logging.Slf4jLogFilter;
import com.alibaba.druid.filter.stat.StatFilter;
import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.util.JdbcUtils;
import com.alibaba.druid.util.Utils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.util.TypeUtils;
import lombok.Getter;
import lombok.Setter;
import org.beetl.sql.clazz.NameConversion;
import org.beetl.sql.clazz.kit.BeanKit;
import org.beetl.sql.core.*;
import org.beetl.sql.core.db.*;
import org.beetl.sql.core.nosql.HBaseStyle;
import org.beetl.sql.core.nosql.HiveStyle;
import org.beetl.sql.core.query.LambdaQuery;

import javax.sql.DataSource;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.*;

/**
 * 数据库工具<br/>
 * 基于BeetlSql和druid实现<br/>
 * date: 2017年5月31日 <br/>
 *
 * @author jingma
 * @version 0.2
 */
@Setter
@Getter
public class Db extends BasicSjzt {
    /**
     * 数据库方言映射集合，可以将本系统引入的druid和beetlsql两大数据库工具关联起来
     */
    public static Map<DbType, Class<?>> dbStyleMap = new HashMap<>();
    /**
     * 我的默认名称转换
     */
    public static NameConversion defaultNameConversion = new MyNameConversion();
    /**
     * 默认数据库
     */
    private static Db db = null;
    /**
     * 数据库类型
     */
    private DbType dbType;
    /**
     * 连接池
     */
    private DataSource dataSource;
    /**
     * beetlSql数据库操作对象
     */
    private SQLManager sqlManager;

    static {
        dbStyleMap.put(DbType.h2, H2Style.class);
        dbStyleMap.put(DbType.mysql, MySqlStyle.class);
        dbStyleMap.put(DbType.oracle, OracleStyle.class);
        dbStyleMap.put(DbType.postgresql, PostgresStyle.class);
        dbStyleMap.put(DbType.gaussdb, OpenGaussStyle.class);
        dbStyleMap.put(DbType.greenplum, GreenplumDBStyle.class);
        dbStyleMap.put(DbType.polardb, PolarDBStyle.class);
        dbStyleMap.put(DbType.sqlserver, SqlServerStyle.class);
        dbStyleMap.put(DbType.sqlite, SQLiteStyle.class);
        dbStyleMap.put(DbType.dm, DamengStyle.class);
        dbStyleMap.put(DbType.derby, DerbyStyle.class);
        //nosql
        dbStyleMap.put(DbType.hive, HiveStyle.class);
        dbStyleMap.put(DbType.hbase, HBaseStyle.class);
        dbStyleMap.put(DbType.vertica,VerticaDBStyle.class);
    }
    public Db(){
        super(null);
    }
    /**
     * 直接创建数据库操作对象 <br/>
     *
     * @param name 名称
     * @param url  数据库连接
     * @param user 数据库用户
     * @param pwd  数据库密码
     * @author jingma
     */
    private Db(String name, String url, String user, String pwd) {
        this(name, createDruidDataSource(url, user, pwd));
    }

    /**
     * 使用已有的数据源创建数据库操作对象 <br/>
     *
     * @param name       名称
     * @param dataSource 已有的数据源
     * @author jingma
     */
    private Db(String name, DruidDataSource dataSource) {
        this(name, dataSource, JdbcUtils.getDbType(dataSource.getUrl(), null));
    }

    /**
     * 使用数据源创建数据库操作对象 <br/>
     *
     * @param name       名称
     * @param dataSource beetlSql的数据源
     * @param dbType     数据库类型
     * @author jingma
     */
    private Db(String name, DataSource dataSource, String dbType) {
        super(name);
        if (!isSupported(dbType)) {
            throw new MyException("该数据库类型暂时不支持：" + dbType);
        }
        this.dbType = DbType.of(dbType);
        this.dataSource = dataSource;
        SQLManagerBuilder builder = new SQLManagerBuilder(ConnectionSourceHelper.getSingle(dataSource));
        builder.setName(name);
        try{
            builder.setDbStyle((DBStyle) dbStyleMap.get(this.dbType).getConstructor().newInstance());
        }catch (Exception e){
            throw new MyException("数据库方言实例化失败",e);
        }
        builder.setNc(defaultNameConversion);
        //开发环境监测sql文件变化
        builder.setProduct(!Conf.getUtilConfig().isDebug());
        builder.addIdAutoGen("idGenerator", (IDAutoGen<String>) params -> StringUtil.getUUIDUpperStr());
        this.sqlManager = builder.build();
        if (Db.db == null) {
            Db.db = this;
        }
        cache.put(name,this);
    }

    /**
     * 是否支持的载体类型
     * @param dbType 载体类型代码
     * @return 是否支持
     */
    public static boolean isSupported(String dbType) {
        return DbType.of(dbType) != null && dbStyleMap.containsKey(DbType.of(dbType));
    }

    /**
     * 取默认数据库操作对象 <br/>
     *
     * @author jingma
     */
    public static Db use() throws MyException {
        if (db == null) {
            throw new MyException("默认数据源未初始化，请初始化后在使用");
        }
        return db;
    }

    /**
     * 获取默认sqlManager <br/>
     *
     * @return 默认beetlSql操作对象
     * @author jingma
     */
    public static SQLManager useSqlManager() throws MyException {
        return use().getSqlManager();
    }

    public static Db use(String name) {
        if(DEFAULT.equals(name)){
            return use();
        }
        Object db = cache.get(name);
        if (db != null) {
            return (Db) db;
        }
        //根据需要动态加载，避免全部初始化加载
        return use(name,getSjzt(name));
    }
    /**
     * 数据名称取数据库操作对象 <br/>
     *
     * @param name 数据库名称
     * @return 对应的数据库对象
     * @author jingma
     */
    //这个锁是为了避免同一个数据源被多次创建，但这个写法不同数据源也会被锁
    public synchronized static Db use(String name,SysSjglSjzt sjzt) {
        Object db = cache.get(name);
        if (db != null) {
            return (Db) db;
        }
        //测试是否可用
        Result r = testDb(sjzt, false);
        if(!r.isStatus()){
            //测试未通过
            throw new MyException(name+r.getMsg());
        }
        //创建数据源
        DruidDataSource ds = createDruidDataSource(sjzt.getLjc(), sjzt.getYhm(),
                sjzt.getMm(), sjzt.getSjkqd(), sjzt.getCsyj(),sjzt.getLx());
        ds.setName(name);
        configDataSource(sjzt, ds, "druid");
        if (StringUtil.isNotBlank(Conf.getAppdm())) {
            configDataSource(sjzt, ds, "druid." + Conf.getAppdm());
        }
        return use(name,ds,sjzt.getLx());
    }
    public static Db use(String name, DruidDataSource ds) {
        return use(name,ds,JdbcUtils.getDbType(ds.getUrl(), null));
    }
    public synchronized static Db use(String name, DataSource ds,String lx) {
        Object db = cache.get(name);
        if (db != null) {
            return (Db) db;
        }
        return new Db(name, ds, lx);
    }
    /**
     * 测试数据载体 <br/>
     *
     * @param sjzt 数据载体
     * @return 测试结果
     * @author jingma
     */
    public static Result cszt(SysSjglSjzt sjzt) {
        return testDb(sjzt, false);
    }

    @Override
    public List<IFile> listFiles(SysSjglZnjh znjhConfig) throws Exception {
        throw new MyException("数据库暂不支持遍历");
    }

    @Override
    public InputStream getInputStream(IFile file) throws Exception {
        if(file instanceof DbFile){
            DbFile dbfile = ((DbFile) file);
            LambdaQuery<SysSjglBlob> query = this.getSqlManager().lambdaQuery(SysSjglBlob.class);
            SysSjglBlob blob = query.andEq(SysSjglBlob::getId, dbfile.getFile().getId()).singleSimple();
            if(blob==null){
                throw new MyException("该文件没找到："+ file.getParent());
            }
            return new ByteArrayInputStream(blob.getNr());
        }
        throw new MyException("不支持非DbFile");
    }

    @Override
    public boolean delete(IFile file) throws Exception {
        if(file instanceof DbFile){
            DbFile dbfile = ((DbFile) file);
            if(StringUtil.isBlank(dbfile.getFile().getId())){
                throw new MyException("没有设置文件主键，无法删除文件");
            }
            return sqlManager().deleteById(SysSjglBlob.class,dbfile.getFile().getId())>0;
        }
        throw new MyException("不支持非DbFile");
    }

    @Override
    public boolean save(InputStream is, IFile file) throws Exception {
        if(file instanceof DbFile){
            try {
                DbFile dbfile = ((DbFile) file);
                SysSjglBlob blob = SysSjglBlob.builder().nr(Utils.readByteArray(is)).id(dbfile.getFile().getId()).build();
                return sqlManager().insertTemplate(blob)>0;
            }finally {
                FileUtil.closeStream(is);
            }
        }
        throw new MyException("不支持非DbFile");
    }

    @Override
    public String getRootPath() {
        return UtilConst.FXG;
    }

    @Override
    public long getSize(IFile file) throws Exception {
        return Utils.readByteArray(getInputStream(file)).length;
    }

    /**
     * 获取sqlManager <br/>
     *
     * @return 指定的beetlSql操作对象
     * @author jingma
     */
    public static SQLManager useSqlManager(String name) {
        return use(name).getSqlManager();
    }

    /**
     * 配置数据源 <br/>
     *
     * @param sjzt            数据载体
     * @param druidDataSource druid数据源
     * @param druidKey        配置key
     * @author jingma
     */
    public static void configDataSource(SysSjglSjzt sjzt, DruidDataSource druidDataSource,
                                        String druidKey) {
        Object dcObj = sjzt.getKzxxObj().getString("$." + druidKey);
        if (dcObj != null) {
            //需支持连接池相关配置
            Properties p = new Properties();
            p.putAll(JSON.parseObject(dcObj.toString()));
            druidDataSource.configFromPropety(p);
        }
    }

    /**
     * 测试数据库是否可用 <br/>
     *
     * @param yobj 数据库对象
     * @param mmjm 是否密码加密
     * @return 测试结果
     * @author jingma
     */
    public static Result testDb(SysSjglSjzt yobj, boolean mmjm) {
        if (yobj == null) {
            return Result.failed("数据载体为空");
        }
        String mm = yobj.getMm();
        if (mmjm && StringUtil.isNotBlank(getSjztEjmm())) {
            try {
                mm = DesUtil.decrypt(mm, getSjztEjmm());
            } catch (Exception e) {
                return Result.failed("数据载体密码解析错误" + yobj, e);
            }
        }
        try {
            if(StringUtil.isBlank(yobj.getSjkqd())){
                try {
                    yobj.setSjkqd(JdbcUtils.getDriverClassName(yobj.getLjc()));
                } catch (SQLException e) {
                    slog.debug(yobj+"驱动获取失败",e);
                    return Result.failed("驱动获取失败："+e.getMessage(), e);
                }
            }
            SQLManager sm;
            try{
                sm = SQLManager.newBuilder(yobj.getSjkqd(),yobj.getLjc(),yobj.getYhm(), mm)
                        .setDbStyle((DBStyle) dbStyleMap.get(DbType.of(yobj.getLx())).newInstance()).build();
            }catch (Exception e){
                throw new MyException("构建SQL执行器失败："+e.getMessage(),e);
            }
            if(StringUtil.isBlank(yobj.getCsyj())){
                yobj.setCsyj(sm.getSQLResult(SqlId.of("util","getDate14"),null).jdbcSql);
            }
            sm.execute(new SQLReady(yobj.getCsyj()), JSONObject.class);
            return Result.success("测试成功");
        } catch (Exception e) {
            slog.debug(yobj.toString(),e);
            return Result.failed("测试不通过：" + e.getMessage(), e);
        }
    }

    /**
     * 判断数据源是否存在 <br/>
     *
     * @param dbCode 数据库对象名称
     * @return 是否存在
     * @author jingma
     */
    public static boolean isCz(String dbCode) {
        return cache.containsKey(dbCode);
    }

    /**
     * 执行更新语句 <br/>
     *
     * @param db     数据库操作类
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @return 影响行数
     * @author jingma
     */
    public static int update(Db db, String sql, Object... params) {
        return db.update(sql, params);
    }

    /**
     * 执行更新语句 <br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 影响行数
     * @author jingma
     */
    public static int update(Db db, String sqlTemplate, Map<String, Object> params) {
        return db.update(sqlTemplate, params);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param db     数据库操作类
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @return 查询结果列表
     * @author jingma
     */
    public static List<JSONObject> find(Db db, String sql, Object... params) {
        return db.find(sql, params);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param db          数据库操作类
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 查询结果列表
     * @author jingma
     */
    public static List<JSONObject> find(Db db, String sqlTemplate, Map<String, Object> params) {
        return db.find(sqlTemplate, params);
    }

    /**
     * 获取第一个对象 <br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @return 查询结果
     * @author jingma
     */
    public static JSONObject findFirst(Db db, String sql, Object... params) {
        return db.findFirst(sql, params);
    }

    /**
     * 获取第一个对象 <br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 查询结果
     * @author jingma
     */
    public static JSONObject findFirst(Db db, String sqlTemplate, Map<String, Object> params) {
        return db.findFirst(sqlTemplate, params);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param keyName key字段名称
     * @param sql     要执行的预编译原生sql语句
     * @param params  参数
     * @return 结果列表基于指定的key转换为map
     * @author jingma
     */
    public static Map<String, JSONObject> findMap(Db db, String keyName, String sql, Object... params) {
        return db.findMap(keyName, sql, params);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param keyName     key字段名称
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 结果列表基于指定的key转换为map
     * @author jingma
     */
    public static Map<String, JSONObject> findMap(Db db, String keyName, String sqlTemplate, Map<String, Object> params) {
        return db.findMap(keyName, sqlTemplate, params);
    }

    /**
     * 查询一个字符串<br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @author jingma
     */
    public static String queryStr(Db db, String sql, Object... params) {
        return db.queryStr(sql, params);
    }

    /**
     * 查询一个字符串<br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @author jingma
     */
    public static String queryStr(Db db, String sqlTemplate, Map<String, Object> params) {
        return db.queryStr(sqlTemplate, params);
    }

    /**
     * 查询一个数字<br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @author jingma
     */
    public static int queryInt(Db db, String sql, Object... params) {
        return db.queryInt(sql, params);
    }

    /**
     * 查询一个数字<br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @author jingma
     */
    public static int queryInt(Db db, String sqlTemplate, Map<String, Object> params) {
        return db.queryInt(sqlTemplate, params);
    }

    /**
     * 列表转Map <br/>
     *
     * @param list    列表
     * @param keyName key字段名称
     * @return 转换结果Map
     * @author jingma
     */
    public static <T> Map<String, T> listToMap(List<T> list, String keyName) {
        Map<String, T> result = new LinkedHashMap<>();
        for (T jo : list) {
            Object o = null;
            if(jo instanceof Map){
                o = ((Map)jo).get(keyName);
            }
            o = ClassUtil.getVal(keyName,o,jo);
            if (o != null) {
                result.put(o.toString(), jo);
            }
        }
        return result;
    }

    /**
     * 将参数列表转换为Map，key默认为p+序号（1开始）,如：p1、p2、p3<br/>
     *
     * @param params 参数列表
     * @return 列表转换出的JSONObject
     * @author jingma
     */
    public static JSONObject buildMap(Object... params) {
        JSONObject ps = new JSONObject();
        for (int i = 0; i < params.length; i++) {
            ps.put("p" + (i + 1), params[i]);
        }
        return ps;
    }
    /**
     * 将含key的参数列表转换为Map，key和值成对出现<br/>
     * @param params 含key的参数列表，eg：key1,xxx1,key2,xxx3
     * @return 列表转换出的JSONObject
     * @author jingma
     */
    public static JSONObject buildKeyMap(Object... params) {
        JSONObject ps = new JSONObject();
        for (int i = 0; i < params.length; i=i+2) {
            ps.set(params[i].toString(),params[i+1]);
        }
        return ps;
    }

    /**
     * 解析sql字符串得到sql所在数据库 <br/>
     *
     * @param sql sql字符串，可以在常规sql后追加;ds=xxx来设置数据源
     * @return 0:sql,1:ds
     * @author jingma
     */
    public static String[] parseDictExp(String sql) {
        return parseDictExp(sql, null);
    }

    /**
     * 解析sql字符串得到sql所在数据库 <br/>
     * @param sql sql字符串，可以在常规sql后追加;ds=xxx来设置数据源
     * @return 0:ds,1:sql
     * @author jingma
     */
    public static String[] parseDictExp(String sql, String defaultSjzt) {
        if (StringUtil.isBlank(sql)) {
            return null;
        }
        String[] result = new String[2];
        String[] strs = sql.split(";");
        if (strs.length > 1&&strs[strs.length-1].startsWith("ds=")) {
            result[0] = strs[strs.length-1].substring(3).trim();
            result[1] = sql.substring(0,sql.lastIndexOf(";"));
        } else if (StringUtil.isNotBlank(defaultSjzt)) {
            result[0] = defaultSjzt;
        } else {
            result[0] = UtilConst.DEFAULT;
        }
        if(isBlank(result[1])){
            result[1] = sql;
        }
        return result;
    }

    /**
     * 获取字典中的sql模板，采用sqlKey.sjztlx方式寻找sql模板
     * @param myParams 相关参数
     * @param sqlKey sql对应的key
     * @param sjztlx 数据载体类型
     * @return 对应sql模板
     */
    public static String getZdSqlTmpl(MyParams myParams, String sqlKey, String sjztlx) {
        //先找对应数据载体类型的sql
        String key= sqlKey + "." + sjztlx;
        String sqlTmpl = Conf.getVal(key);
        if (StringUtil.isBlank(sqlTmpl)) {
            //最基础默认sql-所有数据库通用
            key = sqlKey;
            sqlTmpl = Conf.getVal(key);
        }
        String sql;
        if(StringUtil.isNotBlank(sqlTmpl)){
            //先生成默认SQL
            try{
                sql = TmplUtil.buildStrSql(sqlTmpl, myParams).trim();
                if (sql.startsWith("error:")) {
                    //用于在模板处理中，直接返回信息到前端
                    throw new MyException(sql.substring("error:".length()));
                }else if (sql.startsWith("success:")) {
                    //用于在模板处理中，直接返回信息到前端
                    throw new SuccessException(sql.substring("success:".length()));
                }else if (sql.startsWith("json:")) {
                    //用于在模板处理中，直接返回信息到前端
                    throw new SuccessException("请求成功",JSON.parseObject(sql.substring("json:".length())));
                }else if (sql.startsWith("result:")) {
                    //用于在模板处理中，直接返回信息到前端
                    throw Result.parse(JSON.parseObject(sql.substring("result:".length()))).newMyException();
                }
                myParams.sql().setDefaultSql(sql);
                return sql;
            }catch (MyException e){
                throw e;
            }catch (Exception e){
                slog.error("模板渲染失败",e);
                throw new MyException(e.getMessage(),e);
            }
        }else{
            //没有配置对应模板则直接返回默认sql
            return myParams.sql().getDefaultSql();
        }
    }

    /**
     * 创建Druid数据源 <br/>
     *
     * @param url  数据库连接
     * @param user 数据库用户
     * @param pwd  数据库密码
     * @return 创建的Druid数据源
     * @author jingma
     */
    public static DruidDataSource createDruidDataSource(String url, String user, String pwd) {
        return createDruidDataSource(url, user, pwd, null, null,null);
    }

    /**
     * 创建druid数据源 <br/>
     *
     * @param url             连接串
     * @param user            用户名
     * @param pwd             密码
     * @param driverClassName 数据库驱动
     * @param validationQuery 测试语句
     * @param lx 数据库类型
     * @return 创建出来的数据源
     * @author jingma
     */
    public static DruidDataSource createDruidDataSource(String url, String user, String pwd,
            String driverClassName, String validationQuery,String lx) {
        try {
            boolean isDebug = Conf.getUtilConfig().isDebug();
            DruidDataSource ds = new DruidDataSource();
            ds.setInitialSize(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.initial-size","2")));
            ds.setMinIdle(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.min-idle","2")));
            ds.setMaxActive(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.max-active","150")));
            ds.setMaxWait(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.max-wait","600000")));
            ds.setTestOnBorrow(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.test-on-borrow"),false)));
            ds.setTestWhileIdle(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.test-while-idle"),true)));
            ds.setSocketTimeout(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.socket-timeout","600000")));
            ds.setPhyTimeoutMillis(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.phy-timeout-millis","6000000")));
            //归还连接时执行validationQuery检测连接是否有效，做了这个配置会降低性能。
            ds.setTestOnReturn(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.test-on-return"),false)));
            ds.setTimeBetweenEvictionRunsMillis(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.time-between-eviction-runs-millis","300000")));
            ds.setMinEvictableIdleTimeMillis(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.min-evictable-idle-time-millis","300000")));
            ds.setMaxEvictableIdleTimeMillis(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.max-evictable-idle-time-millis","900000")));
            ds.setKeepAlive(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.keep-alive"),true)));
            ds.setKeepAliveBetweenTimeMillis(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.keep-alive-between-time-millis","600000")));
            //超时未使用，自动断开连接，
            ds.setRemoveAbandoned(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.remove-abandoned"),false)));
            ds.setRemoveAbandonedTimeout(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.remove-abandoned-timeout","3600")));
            ds.setLogAbandoned(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.log-abandoned"),true)));
            ds.setBreakAfterAcquireFailure(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.break-after-acquire-failure"),false)));
            ds.setConnectionErrorRetryAttempts(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.connection-error-retry-attempts","3")));
            ds.setTimeBetweenConnectErrorMillis(TypeUtils.castToInt(Conf.getVal(
                    "spring.datasource.druid.time-between-connect-error-millis","300000")));
            if(StringUtil.isNotBlank(lx)){
                ds.setDbType(DbType.of(lx));
            }
            ds.setUrl(url);
            ds.setUsername(user);
            ds.setPassword(pwd);
            //为空时druid会有默认值
            ds.setValidationQuery(validationQuery);
            //为空时druid会有默认值
            ds.setDriverClassName(driverClassName);

            List<Filter> filters = new ArrayList<>();
            StatFilter stat = new StatFilter();
            stat.setLogSlowSql(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.stat.log-slow-sql"),true)));
            stat.setSlowSqlMillis(TypeUtils.castToInt(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slow-sql-millis"),"30000")));
            stat.setMergeSql(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.stat.merge-sql"),true)));
            filters.add(stat);

            Slf4jLogFilter slf4j = new Slf4jLogFilter();
            slf4j.setStatementExecutableSqlLogEnable(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-executable-sql-log-enable"),isDebug)));
            slf4j.setStatementCreateAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-create-after-log-enabled"),isDebug)));
            slf4j.setStatementPrepareAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-prepare-after-log-enabled"),isDebug)));
            slf4j.setStatementParameterSetLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-parameter-set-log-enabled"),isDebug)));
            slf4j.setStatementExecuteAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-execute-after-log-enabled"),isDebug)));
            slf4j.setStatementCloseAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-close-after-log-enabled"),isDebug)));
            slf4j.setStatementExecuteBatchAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-execute-batch-after-log-enabled"),isDebug)));
            slf4j.setStatementExecuteQueryAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-execute-query-after-log-enabled"),isDebug)));
            slf4j.setStatementExecuteUpdateAfterLogEnabled(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-execute-update-after-log-enabled"),isDebug)));
            slf4j.setStatementParameterClearLogEnable(TypeUtils.castToBoolean(valByDef(Conf.getValByConfig(
                    "spring.datasource.druid.filter.slf4j.statement-parameter-clear-log-enable"),isDebug)));
            filters.add(slf4j);
            ds.setProxyFilters(filters);
            if(db!=null){
                ds.setConnectProperties(Conf.getPropertiesByDict());
            }

            return ds;
        } catch (Exception e) {
            throw new MyException("创建数据源失败", e);
        }
    }

    /**
     * 根据数据源获取数据库类型 <br/>
     *
     * @param dataSource 数据源
     * @return 数据库类型
     * @author jingma
     */
    public static String getDbtypeByDatasource(DataSource dataSource) {
        String dbType = null;
        if (dataSource instanceof DruidDataSource) {
            dbType = ((DruidDataSource) dataSource).getDbType();
        }
        return dbType;
    }

    /**
     * @return defaultNameConversion
     */
    public static NameConversion getDefaultNameConversion() {
        return defaultNameConversion;
    }

    /**
     * @param defaultNameConversion the defaultNameConversion to set
     */
    public static void setDefaultNameConversion(NameConversion defaultNameConversion) {
        Db.defaultNameConversion = defaultNameConversion;
    }

    /**
     * 关闭数据源 <br/>
     *
     * @author jingma
     */
    public void close() {
        if (dataSource instanceof DruidDataSource) {
            ((DruidDataSource) dataSource).close();
        }
        cache.remove(getName());
        if (db == this) {
            db = null;
        }
    }

    /**
     * 获取当前数据库的当前时间的14位时间字符串
     */
    public String getDate14(){
        return queryStr(SqlId.of("util","getDate14"),buildMap());
    }

    /**
     * 获取源sql脚本后采用TmplUtil再次渲染后返回脚本
     */
    public String getSourceSql(SqlId sqlId, Map<String, Object> params){
        //重载
        if(Conf.getUtilConfig().isDebug()){
            sqlManager.getSqlTemplateEngine().getSqlTemplate(sqlId);
        }
        return TmplUtil.buildStrSql(sqlManager.getScript(sqlId).getExecuteContext().sqlSource.getTemplate(),params);
    }

    /**
     * 执行更新语句 <br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @return 影响行数
     * @author jingma
     */
    public int update(String sql, Object... params) {
        return sqlManager.executeUpdate(new SQLReady(sql, params));
    }

    /**
     * 执行更新语句 <br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 影响行数
     * @author jingma
     */
    public int update(String sqlTemplate, Map<String, Object> params) {
        return sqlManager.executeUpdate(sqlTemplate, params);
    }

    /**
     * 执行更新语句 <br/>
     *
     * @param sqlId  sql在文件中的id
     * @param params 参数
     * @return 影响行数
     * @author jingma
     */
    public int update(SqlId sqlId, Map<String, Object> params) {
        return update(getSourceSql(sqlId,params),params);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @return 结果列表
     * @author jingma
     */
    public List<JSONObject> find(String sql, Object... params) {
        return sqlManager.execute(new SQLReady(sql, params), JSONObject.class);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 结果列表
     * @author jingma
     */
    public List<JSONObject> find(String sqlTemplate, Map<String, Object> params) {
        return sqlManager.execute(sqlTemplate, JSONObject.class, params);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param sqlId  sql在文件中的id
     * @param params 参数
     * @return 结果列表
     * @author jingma
     */
    public List<JSONObject> find(SqlId sqlId, Map<String, Object> params) {
        return find(getSourceSql(sqlId,params),params);
    }

    /**
     * 获取第一个对象 <br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @return 查询出的第一行数据
     * @author jingma
     */
    public JSONObject findFirst(String sql, Object... params) {
        List<JSONObject> list = find(sql, params);
        if (list.size() > 0) {
            return list.get(0);
        } else {
            return null;
        }
    }

    /**
     * 获取第一个对象 <br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 查询出的第一行数据
     * @author jingma
     */
    public JSONObject findFirst(String sqlTemplate, Map<String, Object> params) {
        List<JSONObject> list = find(sqlTemplate, params);
        if (list.size() > 0) {
            return list.get(0);
        } else {
            return null;
        }
    }

    /**
     * 获取第一个对象 <br/>
     *
     * @param sqlId  sql在文件中的id
     * @param params 参数
     * @return 查询出的第一行数据
     * @author jingma
     */
    public JSONObject findFirst(SqlId sqlId, Map<String, Object> params) {
        List<JSONObject> list = find(sqlId, params);
        if (list.size() > 0) {
            return list.get(0);
        } else {
            return null;
        }
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param keyName key字段名称
     * @param sql     要执行的预编译原生sql语句
     * @param params  参数
     * @return 结果列表基于指定的key转换为map
     * @author jingma
     */
    public Map<String, JSONObject> findMap(String keyName, String sql, Object... params) {
        List<JSONObject> list = find(sql, params);
        return listToMap(list, keyName);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param keyName     key字段名称
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 结果列表基于指定的key转换为map
     * @author jingma
     */
    public Map<String, JSONObject> findMap(String keyName, String sqlTemplate, Map<String, Object> params) {
        List<JSONObject> list = find(sqlTemplate, params);
        return listToMap(list, keyName);
    }

    /**
     * 获取对象列表  <br/>
     *
     * @param keyName key字段名称
     * @param sqlId   sql在文件中的id
     * @param params  参数
     * @return 结果列表基于指定的key转换为map
     * @author jingma
     */
    public Map<String, JSONObject> findMap(String keyName, SqlId sqlId, Map<String, Object> params) {
        List<JSONObject> list = find(sqlId, params);
        return listToMap(list, keyName);
    }

    /**
     * 查询一个字符串<br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @author jingma
     */
    public String queryStr(String sql, Object... params) {
        JSONObject obj = findFirst(sql, params);
        if (obj != null && obj.size() > 0) {
            return obj.values().toArray()[0] + "";
        } else {
            return null;
        }
    }

    /**
     * 查询一个字符串<br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @author jingma
     */
    public String queryStr(String sqlTemplate, Map<String, Object> params) {
        JSONObject obj = findFirst(sqlTemplate, params);
        if (obj != null && obj.size() > 0) {
            return obj.values().toArray()[0] + "";
        } else {
            return null;
        }
    }

    /**
     * 查询一个字符串<br/>
     *
     * @param sqlId  sql在文件中的id
     * @param params 参数
     * @author jingma
     */
    public String queryStr(SqlId sqlId, Map<String, Object> params) {
        JSONObject obj = findFirst(sqlId, params);
        if (obj != null && obj.size() > 0) {
            return obj.values().toArray()[0] + "";
        } else {
            return null;
        }
    }

    /**
     * 查询一个数字<br/>
     *
     * @param sql    要执行的预编译原生sql语句
     * @param params 参数
     * @author jingma
     */
    public int queryInt(String sql, Object... params) {
        Integer integer = TypeUtils.castToInt(queryStr(sql, params));
        if(isBlank(integer)){
            return 0;
        }
        return integer;
    }

    /**
     * 查询一个数字<br/>
     *
     * @param sqlTemplate sql模板
     * @param params      参数
     * @author jingma
     */
    public int queryInt(String sqlTemplate, Map<String, Object> params) {
        Integer integer = TypeUtils.castToInt(queryStr(sqlTemplate, params));
        if(isBlank(integer)){
            return 0;
        }
        return integer;
    }

    /**
     * 查询一个数字<br/>
     *
     * @param sqlId  sql在文件中的id
     * @param params 参数
     * @author jingma
     */
    public int queryInt(SqlId sqlId, Map<String, Object> params) {
        Integer integer = TypeUtils.castToInt(queryStr(sqlId, params));
        if(isBlank(integer)){
            return 0;
        }
        return integer;
    }

    /**
     * 查询结果直接转换为JSON对象,直接传入SQL <br/>
     *
     * @param page        分页对象
     * @param sqlTemplate sql模板
     * @param params      参数
     * @return 分页查询结果
     * @author jingma
     */
    public PageInfo<JSONObject> queryPage(PageInfo<JSONObject> page, String sqlTemplate, Map<String, Object> params) {
        //底层存在报多线程问题。beetl会修改该对象，为避免相互影响，这里克隆一个传入
        params = JsonUtil.clone(params);
        return (PageInfo<JSONObject>) sqlManager.executePageQuery(sqlTemplate,
                JSONObject.class, params, page);
    }

    /**
     * 查询结果直接转换为JSON对象 <br/>
     * sql脚本会多进行一次TmplUtil模板渲染
     * @param page   分页对象
     * @param sqlId  sql在文件中的id
     * @param params 参数
     * @return 分页查询结果
     * @author jingma
     */
    public PageInfo<JSONObject> queryPage(PageInfo<JSONObject> page, SqlId sqlId, Map<String, Object> params) {
        return queryPage(page,getSourceSql(sqlId,params),params);
    }

    /**
     * 返回自定义的lambdaQuery，自动处理有效性、更新时间、创建人等，不需要这些特性时请采用原生LambdaQuery
     * @param clazz 实体类
     * @param <T> 泛型
     * @return LambdaQuery对象
     */
    public <T> MyLambdaQuery<T> lambdaQuery(Class<T> clazz) {
        return lambdaQuery(clazz,null);
    }
    /**
     * 返回自定义的lambdaQuery，自动处理有效性、更新时间、创建人等，不需要这些特性时请采用原生LambdaQuery
     * @param clazz 实体类
     * @param <T> 泛型
     * @param user 用户信息
     * @return LambdaQuery对象
     */
    public <T> MyLambdaQuery<T> lambdaQuery(Class<T> clazz, SysQxYhxx user) {
        if (BeanKit.queryLambdasSupport) {
            MyLambdaQuery<T> q = new MyLambdaQuery<>(sqlManager, clazz);
            q.setUser(user);
            return q;
        } else {
            throw new UnsupportedOperationException("需要Java8以上");
        }
    }
    /**
     * @return name
     */
    public String getName() {
        return sqlManager.getName();
    }

    /**
     * @return 默认数据源是否初始化了
     */
    public static boolean isDefaultInit(){
        return db!=null;
    }

}
