/**
* Project Name:KettleUtil
* Date:2016年7月14日
* Copyright (c) 2016, jingma All Rights Reserved.
*/

package cn.benma666.myutils;

import de.innosystec.unrar.Archive;
import de.innosystec.unrar.exception.RarException;
import de.innosystec.unrar.rarfile.FileHeader;
import net.lingala.zip4j.exception.ZipException;
import org.apache.commons.io.IOUtils;

import java.io.*;


/**
 * Rar压缩文件处理工具类 <br/>
 * date: 2016年7月14日 下午4:13:34 <br/>
 * @author jingma
 * @version
 */
public class RarUtilPwd {
    /**
    * 是否将rar文件的名称作为对应解压文件父级目录
    */
    private static boolean isRarNameToDir = false;
    public static final String SEPARATOR = File.separator;

    /**
     * 使用给定密码解压指定目录下的rar压缩文件到指定目录
     * <p>
     * 如果指定目录不存在,可以自动创建,不合法的路径将导致异常被抛出
     * @param rarDir 指定的rar压缩文件
     * @param dest 解压目录
     * @param pwd 密码
     * @throws ZipException 压缩文件有损坏或者解压缩失败抛出
     */
    public static void unRarDir(String rarDir, String dest,String pwd) throws Exception {
        File zips = new File(rarDir);
        for(File zip:zips.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                if(name.endsWith(".rar")){
                    return true;
                }
                return false;
            }
        })){
            unrar(zip.getAbsolutePath(),dest,pwd);
        }
    }
    /**
     * 解压指定RAR文件到当前文件夹
     * @param srcRar 指定解压
     *  @param password 压缩文件时设定的密码
     * @throws IOException
     */
    public static void unrar(String srcRar, String password) throws IOException {
     unrar(srcRar, null, password);
    }

    /**
     * 解压指定的RAR压缩文件到指定的目录中
     * @param srcRar 指定的RAR压缩文件
     * @param destPath 指定解压到的目录
     *  @param password 压缩文件时设定的密码
     * @throws IOException
     */
    public static void unrar(String srcRar, String destPath, String password) throws IOException {
     File srcFile = new File(srcRar);
     if (!srcFile.exists()) {
      return;
     }
     if (null == destPath || destPath.length() == 0) {
      unrar(srcFile, srcFile.getParent(), password);
      return;
     }
     unrar(srcFile,destPath, password);
    }

    /**
     * 解压指定RAR文件到当前文件夹
     * @param srcRarFile 解压文件
     *  @param password 压缩文件时设定的密码
     * @throws IOException
     */
    public static void unrar(File srcRarFile, String password) throws IOException {
     if (null == srcRarFile || !srcRarFile.exists()) {
      throw new IOException("指定文件不存在.");
     }
     unrar(srcRarFile, srcRarFile.getParent(),password);
    }

    /**
     * 解压指定RAR文件到指定的路径
     * @param srcRarFile 需要解压RAR文件
     * @param destPath 指定解压路径
     * @param password 压缩文件时设定的密码
     * @throws IOException
     */
    public static void unrar(File srcRarFile, String destPath, String password) throws IOException {
     if (null == srcRarFile || !srcRarFile.exists()) {
      throw new IOException("指定压缩文件不存在.");
     }
     if (!destPath.endsWith(SEPARATOR)) {
      destPath += SEPARATOR;
     }
     Archive archive = null;
     OutputStream unOut = null;
     try {
      archive = new Archive(srcRarFile, password, false);
      FileHeader fileHeader = archive.nextFileHeader();
      while(null != fileHeader) {
       if (!fileHeader.isDirectory()) {
        // 1 根据不同的操作系统拿到相应的 destDirName 和 destFileName
        String destFileName = "";
        String destDirName = "";
        String fileName = fileHeader.getFileNameW();
        if(StringUtil.isBlank(fileName)){
            fileName = fileHeader.getFileNameString();
        }
        if (SEPARATOR.equals("/")) {  // 非windows系统
         destFileName = (destPath + fileName).replaceAll("\\\\", "/");
         destDirName = destFileName.substring(0, destFileName.lastIndexOf("/"));
        } else {  // windows系统
         destFileName = (destPath + fileName).replaceAll("/", "\\\\");
         destDirName = destFileName.substring(0, destFileName.lastIndexOf("\\"));
        }
        // 2创建文件夹
        File dir = new File(destDirName);
        if (!dir.exists() || !dir.isDirectory()) {
         dir.mkdirs();
        }
        // 抽取压缩文件
        unOut = new FileOutputStream(new File(destFileName));
        archive.extractFile(fileHeader, unOut);
        unOut.flush();
        unOut.close();
       }
       fileHeader = archive.nextFileHeader();
      }
      archive.close();
     } catch (RarException e) {
      e.printStackTrace();
     } finally {
      IOUtils.closeQuietly(unOut);
     }
    }

    /**
     * @return isRarNameToDir
     */
    public static boolean isRarNameToDir() {
        return isRarNameToDir;
    }
    /**
     * @param isRarNameToDir the isRarNameToDir to set
     */
    public static void setRarNameToDir(boolean isRarNameToDir) {
        RarUtilPwd.isRarNameToDir = isRarNameToDir;
    }
}
