package cn.benma666.myutils;

import java.util.List;

/**
 * 经纬度工具
 */
public class JWDUtil {

    private static final double pi = 3.1415926535897932384626;
    private static final double x_pi = 3.14159265358979324 * 3000.0 / 180.0;
    private static final double EARTH_RADIUS = 6378245.0;
    private static final double ee = 0.00669342162296594323;

    private static double transformLat(double x, double y) {
        double ret = -100.0 + 2.0 * x + 3.0 * y + 0.2 * y * y + 0.1 * x * y
                + 0.2 * Math.sqrt(Math.abs(x));
        ret += (20.0 * Math.sin(6.0 * x * pi) + 20.0 * Math.sin(2.0 * x * pi)) * 2.0 / 3.0;
        ret += (20.0 * Math.sin(y * pi) + 40.0 * Math.sin(y / 3.0 * pi)) * 2.0 / 3.0;
        ret += (160.0 * Math.sin(y / 12.0 * pi) + 320 * Math.sin(y * pi / 30.0)) * 2.0 / 3.0;
        return ret;
    }

    private static double transformLon(double x, double y) {
        double ret = 300.0 + x + 2.0 * y + 0.1 * x * x + 0.1 * x * y + 0.1
                * Math.sqrt(Math.abs(x));
        ret += (20.0 * Math.sin(6.0 * x * pi) + 20.0 * Math.sin(2.0 * x * pi)) * 2.0 / 3.0;
        ret += (20.0 * Math.sin(x * pi) + 40.0 * Math.sin(x / 3.0 * pi)) * 2.0 / 3.0;
        ret += (150.0 * Math.sin(x / 12.0 * pi) + 300.0 * Math.sin(x / 30.0
                * pi)) * 2.0 / 3.0;
        return ret;
    }

    private static double[] transform(double lat, double lon) {
        if (outOfChina(lat, lon)) {
            return new double[]{lat, lon};
        }
        double dLat = transformLat(lon - 105.0, lat - 35.0);
        double dLon = transformLon(lon - 105.0, lat - 35.0);
        double radLat = lat / 180.0 * pi;
        double magic = Math.sin(radLat);
        magic = 1 - ee * magic * magic;
        double sqrtMagic = Math.sqrt(magic);
        dLat = (dLat * 180.0) / ((EARTH_RADIUS * (1 - ee)) / (magic * sqrtMagic) * pi);
        dLon = (dLon * 180.0) / (EARTH_RADIUS / sqrtMagic * Math.cos(radLat) * pi);
        double mgLat = lat + dLat;
        double mgLon = lon + dLon;
        return new double[]{mgLat, mgLon};
    }

    private static boolean outOfChina(double lat, double lon) {
        if (lon < 72.004 || lon > 137.8347)
            return true;
        if (lat < 0.8293 || lat > 55.8271)
            return true;
        return false;
    }

    /**
     * Gps84 -> Gcj02
     *
     * @param lat
     * @param lon
     * @return
     */
    public static double[] Gps84_To_Gcj02(double lat, double lon) {
        if (outOfChina(lat, lon)) {
            return new double[]{lat, lon};
        }
        double dLat = transformLat(lon - 105.0, lat - 35.0);
        double dLon = transformLon(lon - 105.0, lat - 35.0);
        double radLat = lat / 180.0 * pi;
        double magic = Math.sin(radLat);
        magic = 1 - ee * magic * magic;
        double sqrtMagic = Math.sqrt(magic);
        dLat = (dLat * 180.0) / ((EARTH_RADIUS * (1 - ee)) / (magic * sqrtMagic) * pi);
        dLon = (dLon * 180.0) / (EARTH_RADIUS / sqrtMagic * Math.cos(radLat) * pi);
        double mgLat = lat + dLat;
        double mgLon = lon + dLon;
        double[] Gcj02 = {mgLat, mgLon};
        return Gcj02;
    }

    /**
     * Gcj02 -> Gps84
     *
     * @param lat
     * @param lon
     * @return
     */
    public static double[] Gcj02_To_Gps84(double lat, double lon) {
        double[] gps = transform(lat, lon);
        double latitude = lat * 2 - gps[0];
        double lontitude = lon * 2 - gps[1];
        double[] Gps84 = {latitude, lontitude};
        return Gps84;
    }

    /**
     * Gcj02 -> Bd09
     *
     * @param lat
     * @param lon
     * @return
     */
    public static double[] Gcj02_To_Bd09(double lat, double lon) {
        double x = lon, y = lat;
        double z = Math.sqrt(x * x + y * y) + 0.00002 * Math.sin(y * x_pi);
        double theta = Math.atan2(y, x) + 0.000003 * Math.cos(x * x_pi);
        double tempLat = z * Math.sin(theta) + 0.006;
        double tempLon = z * Math.cos(theta) + 0.0065;
        double[] Bd09 = {tempLat, tempLon};
        return Bd09;
    }

    /**
     * Bd09 -> Gcj02
     *
     * @param lat
     * @param lon
     * @return
     */
    public static double[] Bd09_To_Gcj02(double lat, double lon) {
        double x = lon - 0.0065, y = lat - 0.006;
        double z = Math.sqrt(x * x + y * y) - 0.00002 * Math.sin(y * x_pi);
        double theta = Math.atan2(y, x) - 0.000003 * Math.cos(x * x_pi);
        double tempLon = z * Math.cos(theta);
        double tempLat = z * Math.sin(theta);
        double[] Gcj02 = {tempLat, tempLon};
        return Gcj02;
    }

    /**
     * Gps84 -> Bd09
     *
     * @param lat
     * @param lon
     * @return
     */
    public static double[] Gps84_To_Bd09(double lat, double lon) {
        double[] Gcj02 = Gps84_To_Gcj02(lat, lon);
        double[] Bd09 = Gcj02_To_Bd09(Gcj02[0], Gcj02[1]);
        return Bd09;
    }

    /**
     * Bd09 -> Gps84
     *
     * @param lat
     * @param lon
     * @return
     */
    public static double[] Bd09_To_Gps84(double lat, double lon) {
        double[] Gcj02 = Bd09_To_Gcj02(lat, lon);
        double[] Gps84 = Gcj02_To_Gps84(Gcj02[0], Gcj02[1]);
        // 保留小数点后六位
        Gps84[0] = retain6(Gps84[0]);
        Gps84[1] = retain6(Gps84[1]);
        return Gps84;
    }

    /**
     * 保留小数点后六位
     * @param num
     * @return
     */
    private static double retain6(double num) {
        String result = String.format("%.6f", num);
        return Double.valueOf(result);
    }

    public static class LatLng {
        public double latitude;
        public double longitude;
        public LatLng(double latitude, double longitude) {
            this.latitude = latitude;
            this.longitude = longitude;
        }
    }

    /**
     * 计算两个经纬度点的中心点
     * @param points 两个点
     * @return 中心点
     */
    public static LatLng calculateCenter(List<LatLng> points) {
        double x = 0;
        double y = 0;
        double z = 0;
        for (LatLng point : points) {
            double latitude = Math.toRadians(point.latitude);
            double longitude = Math.toRadians(point.longitude);

            x += Math.cos(latitude) * Math.cos(longitude);
            y += Math.cos(latitude) * Math.sin(longitude);
            z += Math.sin(latitude);
        }
        int totalPoints = points.size();
        x /= totalPoints;
        y /= totalPoints;
        z /= totalPoints;
        double centralLongitude = Math.atan2(y, x);
        double centralSquareRoot = Math.sqrt(x * x + y * y);
        double centralLatitude = Math.atan2(z, centralSquareRoot);
        return new LatLng(Math.toDegrees(centralLatitude), Math.toDegrees(centralLongitude));
    }
    private static double rad(double d) {
        return d * Math.PI / 180.0;
    }

    /**
     * 根据两个位置的经纬度，来计算两地的距离（单位为米）
     * @param lat1 经度1
     * @param lng1 纬度1
     * @param lat2 经度2
     * @param lng2 纬度2
     * @return 两地距离（单位为米）
     */
    public static double getDistance(double lng1, double lat1, double lng2, double lat2) {
        double radLat1 = rad(lat1);
        double radLat2 = rad(lat2);
        double a = radLat1 - radLat2;
        double b = rad(lng1) - rad(lng2);
        double s = 2 * Math.asin(Math.sqrt(Math.pow(Math.sin(a / 2), 2) +
                Math.cos(radLat1) * Math.cos(radLat2) * Math.pow(Math.sin(b / 2), 2)));
        s = s * EARTH_RADIUS;
        s = Math.round(s * 10000) / 10000;
        return s;
    }

    public static void main(String[] args) {
        double lat1 = 40.7128; // 第一个点的纬度
        double lon1 = -74.0060; // 第一个点的经度

        double lat2 = 51.5074; // 第二个点的纬度
        double lon2 = -0.1278; // 第二个点的经度

        double distanceInMeters = getDistance(lat1, lon1, lat2, lon2);
        System.out.println("两个点之间的距离为：" + distanceInMeters + " 米");
    }
}
