/**
 * Project Name:ftpManager
 * File Name:ftpManager.java
 * Package Name:ftpManager
 * Date:2014年7月11日
 * Copyright (c) 2014, jingma All Rights Reserved.
 *
 */

package cn.benma666.sjzt.ftp;

import cn.benma666.constants.UtilConst;
import cn.benma666.domain.SysSjglSjzt;
import cn.benma666.domain.SysSjglZnjh;
import cn.benma666.exception.MyException;
import cn.benma666.iframe.Conf;
import cn.benma666.iframe.Result;
import cn.benma666.myutils.ClassUtil;
import cn.benma666.myutils.FileUtil;
import cn.benma666.myutils.StringUtil;
import cn.benma666.sjzt.BasicSjzt;
import cn.benma666.sjzt.IFile;
import cn.benma666.sjzt.SjztExecRunnable;
import cn.benma666.sjzt.SjztPooledObjectFactory;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.util.TypeUtils;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.net.ProtocolCommandEvent;
import org.apache.commons.net.ProtocolCommandListener;
import org.apache.commons.net.ftp.*;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;

import java.io.*;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static cn.benma666.myutils.FileUtil.closeStream;
import static cn.benma666.myutils.FileUtil.getFilePath;

/**
 * ftp管理工具<br/>
 * 继承本地文件对象，与ftp交互需要一个本地文件作为缓存，后续文件的处理也是继承自本地文件
 * Date: 2014年7月11日 上午12:21:16 <br/>
 * @author jingma
 */
@Setter
@Getter
public class Ftp extends BasicSjzt {
    /**
     * ftp url解析正则
     */
    public static Pattern ftpUrlZz = Pattern.compile("^(ftp://)?([^:]*)(:(\\d+))?$");
    /**
     * 默认ftp操作对象
     */
    public static Ftp ftp;
    /**
     * 根目录
     */
    private String root;
    /**
     * ftp客户端对象池
     */
    private GenericObjectPool objectPool;

    /**
     * @param name 代码
     * @param sjzt 数据载体对象
     */
    private Ftp(String name,SysSjglSjzt sjzt){
        this(name,createFtpFactory(sjzt));
    }
    /**
     * @param name ftp名称
     * @param ftpFactory ftp工厂
     */
    private Ftp(String name, SjztPooledObjectFactory ftpFactory){
        //设置本地缓存目录
        super(name);
        GenericObjectPoolConfig<FTPClient> conf = new GenericObjectPoolConfig<>();
        conf.setTestOnCreate(true);
        conf.setTestOnBorrow(true);
        conf.setMinIdle(1);
        conf.setMaxIdle(3);
        conf.setMaxTotal(5);
        conf.setMaxWait(Duration.ofMinutes(10));
//        conf.setTestWhileIdle(true);
        setSjzt(ftpFactory.getSjzt());
        //支持在载体中配置
        ClassUtil.plMethodInvoke(conf,getSjzt().getKzxxObj().getJSONObject("ljcpz"));
        this.objectPool =new GenericObjectPool(ftpFactory,conf);
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            this.root = valByDef(pwd(ftpClient),UtilConst.FXG);
            log.info("{}根路径：{}",name,root);
        } catch (Exception e) {
            throw new MyException(name+"初始化失败",e);
        }finally {
            returnClient(ftpClient);
        }
        if (ftp == null) {
            ftp = this;
        }
        cache.put(name,this);
    }
    /**
     * 使用其他ftp <br/>
     * @author jingma
     * @param name ftp的数据载体代码
     * @return 工具对象
     */
    public static Ftp use(String name) {
        return use(name,getSjzt(name));
    }
    public synchronized static Ftp use(String name,SysSjglSjzt sjzt) {
        Ftp ftp = (Ftp) cache.get(name);
        if (ftp == null) {
            ftp = new Ftp(name,sjzt);
        }
        return ftp;
    }
    public static Result cszt(SysSjglSjzt sjzt) {
        try {
            parseSjztFtp(sjzt);
            FTPClient fc = createClient(sjzt);
            fc.printWorkingDirectory();
            fc.disconnect();
            return success("测试成功");
        }catch (Throwable t){
            return failed("载体测试不通过："+t.getMessage());
        }
    }

    /**
     * 验证客户端是否可用
     */
    public static boolean validateClient(SysSjglSjzt sjzt,Object client) {
        try {
            FTPClient ftpClient = (FTPClient) client;
            if(!ftpClient.isConnected()){
                return false;
            }
            slog.trace("模式：{},网络连接超时:{}，默认超时时长：{},缓存大小：{},编码：{},AutodetectUTF8:{},OptsUtf8:{}",
                    ftpClient.getDataConnectionMode(),ftpClient.getSoTimeout(),ftpClient.getDefaultTimeout(),
                    ftpClient.getBufferSize(),ftpClient.getControlEncoding(),ftpClient.getAutodetectUTF8(),
                    FTPReply.isPositiveCompletion(ftpClient.sendCommand("OPTS UTF8", "ON")));
            //获取当前目录正常
            return StringUtil.isNotBlank(ftpClient.printWorkingDirectory());
        }catch (Throwable t){
            slog.debug("ftp验证无效：{}",sjzt.getMc(),t);
            return false;
        }
    }

    /**
     * 关闭客户端
     */
    public static void destroyClient(SysSjglSjzt sjzt,Object client) throws Exception {
        FTPClient ftpClient = (FTPClient) client;
        ftpClient.disconnect();
    }
    public static FTPClient createClient(SysSjglSjzt sjzt) {
//        slog.trace("连接的数据载体：{}",sjzt);
        //创建一个ftp操作对象
        FTPClient ftpClient = new FTPClient();
        FTPClientConfig conf = null;
        //取消服务器获取自身Ip地址和提交的host进行匹配，否则当不一致时报出以上异常。
        ftpClient.setRemoteVerificationEnabled(false);
        ftpClient.setAutodetectUTF8(true);
        ftpClient.setControlEncoding("UTF-8");
        JSONObject ftppz = sjzt.getKzxxObj().getJSONObject("ftppz");
        if(ftppz!=null){
            if(!StringUtil.isBlank(ftppz.getString("$.FTPClientConfig.serverSystemKey"))){
                //设置操作系统
                conf = new FTPClientConfig(ftppz.getString("$.FTPClientConfig.serverSystemKey"));
            }
            ClassUtil.plMethodInvoke(ftpClient,ftppz.getJSONObject("FTPClient"));
            if(conf!=null){
                ClassUtil.plMethodInvoke(conf,ftppz.getJSONObject("FTPClientConfig"));
                ftpClient.configure(conf);
            }
        }
        return createClient(ftpClient,sjzt.getIp(), sjzt.getPort(),sjzt.getYhm(),
                sjzt.getMm(),TypeUtils.castToBoolean(Conf.getVal("benma666.debug")),sjzt);
    }

    /**
     * @param ftpClient 客户端
     * @param host 主机
     * @param port 端口
     * @param username 用户名
     * @param password 密码
     * @param isPrintCommmand 是否输出处理详情
     * @param sjzt 数据载体
     */
    public static FTPClient createClient(FTPClient ftpClient, String host, int port, String username,
                                             String password, boolean isPrintCommmand, SysSjglSjzt sjzt) {
        if (StringUtil.valByDef(isPrintCommmand,false)) {
            //设置详细信息打印
            ftpClient.addProtocolCommandListener(new ProtocolCommandListener() {
                @Override
                public void protocolCommandSent(ProtocolCommandEvent protocolCommandEvent) {
                    slog.trace(ftpClient.getRemoteAddress()+":"+protocolCommandEvent.getMessage());
                }
                @Override
                public void protocolReplyReceived(ProtocolCommandEvent protocolCommandEvent) {
                    slog.trace(ftpClient.getRemoteAddress()+":"+protocolCommandEvent.getMessage());
                }
            });
        }
        try {
            //连接服务器
            ftpClient.connect(host, port);
            if (FTPReply.isPositiveCompletion(ftpClient.getReplyCode())) {
                //登陆ftp
                if(!ftpClient.login(username, password)){
                    throw new MyException("账号密码登录失败");
                }
                ftpClient.setFileType(FTP.BINARY_FILE_TYPE);
                //默认设为被动模式，必须登录后设置
                ftpClient.enterLocalPassiveMode();
                ClassUtil.plMethodInvoke(ftpClient,sjzt.getKzxxObj()
                        .getJSONObject("$.ftppz.FTPClientDlh"));
            }else{
                if (ftpClient.isConnected()) {
                    ftpClient.disconnect();
                }
                throw new MyException("连接失败");
            }
            slog.trace("模式：{},网络连接超时:{}，默认超时时长：{},缓存大小：{},编码：{},AutodetectUTF8:{},OptsUtf8:{}",
                    ftpClient.getDataConnectionMode(),ftpClient.getSoTimeout(),ftpClient.getDefaultTimeout(),
                    ftpClient.getBufferSize(),ftpClient.getControlEncoding(),ftpClient.getAutodetectUTF8(),
                    FTPReply.isPositiveCompletion(ftpClient.sendCommand("OPTS UTF8", "ON")));
        } catch (Exception e) {
            throw new MyException("ftp登录失败："+ host+","+port+","+username,e);
        }
        return ftpClient;
    }

    /**
     * 退回客户端
     * @param ftpClient 需要退回的客户端
     */
    public void returnClient(FTPClient ftpClient) {
        log.trace("释放ftp回连接池：{}",name);
        if (ftpClient != null) {
            objectPool.returnObject(ftpClient);
        }
    }

    /**
     * 获取客户端
     */
    public FTPClient borrowClient() throws Exception {
        log.trace("从连接池获取ftp：{}",name);
        Object r = StringUtil.requireNonNull(objectPool,"对象池为空"+name).borrowObject();
        return (FTPClient)StringUtil.requireNonNull(r,"ftp连接池返回的客户端为空"+name);
    }

    /**
     * 关闭连接
     */
    @Override
    public void close() throws IOException {
        if (!this.objectPool.isClosed()) {
            this.objectPool.close();
        }
        cache.remove(name);
        if(this==ftp){
            ftp=null;
        }
    }

    /**
     * 直接使用内部客户端
     * @param exec 实现执行接口
     * @return 执行结果
     * @throws Exception 处理异常
     */
    public Object exec(SjztExecRunnable<FTPClient> exec) throws Exception {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            return exec.exec(ftpClient);
        }finally {
            returnClient(ftpClient);
        }
    }

    @Override
    public List<IFile> listFiles(SysSjglZnjh znjhConfig) throws Exception{
        return (List<IFile>) exec(ftpClient -> listFiles(znjhConfig.getSrml(),ftpClient,znjhConfig));
    }

    private List<IFile> listFiles(String path, FTPClient ftpClient,SysSjglZnjh znjhConfig) throws Exception {
        List<IFile> list = new ArrayList<>();
        String p = getFilePath(root, path);
        log.trace("开始遍历目录：{}",p);
        FTPFile[] files = ftpClient.listFiles(p);
        for (FTPFile file : files) {
            log.trace("遍历到文件：{}",file.getName());
            if (file.isFile()) {
                long t = file.getTimestamp()==null?0:file.getTimestamp().getTimeInMillis();
                if (sftg(znjhConfig,t,file.getName())) continue;
                list.add(new FtpFile(path, file,this));
            } else if (file.isDirectory()&&!file.getName().contains(".")) {
                if(znjhConfig.isBlbhml()){
                    long t = file.getTimestamp()==null?0:file.getTimestamp().getTimeInMillis();
                    if (!sftg(znjhConfig,t,file.getName())){
                        list.add(new FtpFile(path, file,this));
                    }
                }
                if(znjhConfig.isDgbl()){
                    list.addAll(listFiles(getFilePath(path , file.getName()),ftpClient,znjhConfig));
                }
            }
        }
        return list;
    }

    @Override
    public InputStream getInputStream(IFile file) throws Exception {
        int i = 0;
        InputStream is = null;
        FTPClient ftpClient = null;
        try{
            ftpClient = borrowClient();
            while (is==null&&++i<20){
                try {
                    is = ftpClient.retrieveFileStream(getAbsolutePath(file));
                }catch (Exception e){
                    slog.trace("获取输入流异常：{}",file.getAbsolutePath(),e);
                }
                Thread.sleep(100);
            }
            if(is==null){
                throw new MyException("没有获取到输入流："+getAbsolutePath(file));
            }
            return new FtpInputStream(is,this,ftpClient);
        }catch (Throwable t){
            //获取流异常时退回连接
            returnClient(ftpClient);
            throw new MyException("获取输入流异常："+getAbsolutePath(file),t);
        }
    }

    @Override
    public boolean delete(IFile file) throws Exception {
        return (boolean) exec(ftpClient -> ftpClient.deleteFile(getAbsolutePath(file)));
    }

    @Override
    public boolean save(InputStream is, IFile file) throws Exception {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            makeDir(FileUtil.getFilePath(file.getGzml(),file.getParent()),ftpClient);
            int i = 0;
            while (i++<20&&!ftpClient.storeFile(getAbsolutePath(file)+UNIMASTMP, is)){
                Thread.sleep(100);
            }
            if(i>20){
                throw new MyException("文件保存失败");
            }
            i = 0;
            while (i++<20&&!ftpClient.rename(getAbsolutePath(file)+UNIMASTMP,getAbsolutePath(file))){
                Thread.sleep(100);
            }
            if(i>20){
                throw new MyException("文件重命名失败："+getAbsolutePath(file));
            }
        }finally {
            returnClient(ftpClient);
            closeStream(is);
        }
        return true;
    }

    @Override
    public String getRootPath() {
        return root;
    }

    @Override
    public long getSize(IFile file) throws Exception {
        return (long) exec(ftpClient -> {
            String path = getAbsolutePath(file);
            long size = 0;
            int i = 0;
            while (size==0&&i<20){
                i++;
                try{
                    //ftp真的烦，各种问题
                    size = TypeUtils.castToLong(ftpClient.getSize(path));
                }catch (Exception e){
                    //忽略
                    slog.trace("获取文件大小失败："+path,e);
                }
            }
            return size;
        });
    }

    /**
     * 从FTP服务器下载文件
     * @param ftpPath 要下载的远程路径
     * @param fileName 文件名
     * @param localPath 下载后保存到本地的路径
     * @return 下载的文件
     */
    public File download(String ftpPath,String fileName, String localPath) {
        OutputStream is = null;
        File localFile = new File(getFilePath(localPath , fileName));
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            setPath(ftpPath,ftpClient);
            File f = new File(localPath);
            if (!f.exists()&&!f.mkdirs()) {
                throw new MyException("目录创建失败："+f.getAbsolutePath());
            }
            is = new FileOutputStream(localFile);
            ftpClient.retrieveFile(zhFileName(fileName), is);
        } catch (Exception e) {
            log.error("文件下载失败", e);
            throw new MyException("文件下载失败",e);
        }finally {
            FileUtil.closeStream(is);
            returnClient(ftpClient);
        }
        return localFile;
    }

    /**
     * 向FTP服务器上传文件
     * @param ftpPath FTP服务器保存目录
     * @param filename 上传到FTP服务器上的文件名
     * @param input 输入流
     * @return 上传状态
     */
    public boolean upload(String ftpPath, String filename, InputStream input) {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            makeDir(ftpPath, ftpClient);
            setPath(ftpPath, ftpClient);
            return ftpClient.storeFile(zhFileName(filename), input);
        } catch (Exception e) {
            log.error("上传ftp失败", e);
            throw new MyException("上传ftp失败",e);
        }finally {
            FileUtil.closeStream(input);
            returnClient(ftpClient);
        }
    }

    /**
     * 删除FTP上的文件
     * @param ftpPath 远程文件路径
     * @param fileName 待删除的文件名
     * @return 删除状态
     */
    public boolean delete(String ftpPath, String fileName) {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            return ftpClient.deleteFile(zhFileName(getFilePath(root, ftpPath , fileName)));
        } catch (Exception e) {
            log.error("删除文件失败", e);
            throw new MyException("删除文件失败",e);
        }finally {
            returnClient(ftpClient);
        }
    }
    /**
     * Description: 从FTP服务器下载文件
     * @param ftpPath 要下载的远程路径
     * @param localPath 下载后保存到本地的路径
     * @return 下载的文件列表
     */
    public List<File> downloadPath(String ftpPath, String localPath) {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            return downloadPath(ftpPath,localPath,ftpClient);
        } catch (Exception e) {
            log.error("文件下载失败", e);
            throw new MyException("文件下载失败",e);
        }finally {
            returnClient(ftpClient);
        }
    }
    /**
     * Description: 从FTP服务器下载文件
     * @param ftpPath 要下载的远程路径
     * @param localPath 下载后保存到本地的路径
     * @return 下载的文件列表
     */
    public List<File> downloadPath(String ftpPath, String localPath,FTPClient ftpClient) {
        OutputStream os = null;
        List<File> fileList = new ArrayList<>();
        try {
            File lo = new File(localPath);
            if (!lo.exists()&&!lo.mkdirs()) {
                throw new MyException("目录创建失败："+lo.getAbsolutePath());
            }
            setPath(ftpPath, ftpClient);
            FTPFile[] fs = ftpClient.listFiles();
            for (FTPFile ff : fs) {
                if (ff.isDirectory()) {
                    //目录，递归调用
                    fileList.addAll(downloadPath(getFilePath(ftpPath , ff.getName()),
                            getFilePath(localPath ,ff.getName()),ftpClient));
                } else {
                    //文件
                    File localFile = new File(getFilePath(localPath ,ff.getName()));
                    os = new FileOutputStream(localFile);
                    ftpClient.retrieveFile(zhFileName(ff.getName()), os);
                    fileList.add(localFile);
                }
            }
        } catch (Exception e) {
            log.error("文件下载失败", e);
            throw new MyException("文件下载失败",e);
        } finally {
            FileUtil.closeStream(os);
        }
        return fileList;
    }

    /**
     * 向FTP服务器上传文件
     * @param ftpPath FTP服务器保存目录
     * @param localPath 本地路径
     */
    public void uploadPath(String ftpPath,String localPath) {
        File file = new File(localPath);
        FileInputStream in;
        try {
            String fp;
            if (file.isFile()) {
                in = new FileInputStream(file);
                upload(ftpPath, file.getName(),in);
            } else if (file.isDirectory()) {
                File[] list = file.listFiles();
                if(list!=null){
                    for (File f : list) {
                        fp = ftpPath;
                        if(f.isDirectory()){
                            fp = getFilePath(ftpPath,f.getName());
                        }
                        uploadPath(fp, f.getAbsolutePath());
                    }
                }
            }
        } catch (Exception e) {
            log.error("文件找不到",e);
            throw new MyException("文件找不到",e);
        }
    }
    /**
     * 远程FTP删除目录下的所有文件包含目录本身
     * @param ftpPath ftp目录
     */
    public void deletePath(String ftpPath) {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            FTPFile[] ftpFiles = ftpClient.listFiles(getFilePath(root,ftpPath));
            for (FTPFile file : ftpFiles) {
                if (file.isDirectory()) {
                    deletePath(getFilePath(ftpPath , file.getName()));
                } else {
                    ftpClient.deleteFile(zhFileName(getFilePath(root,ftpPath,file.getName())));
                }
            }
            ftpClient.removeDirectory(zhFileName(getFilePath(root,ftpPath)));

        } catch (Exception e) {
            log.error("从FTP服务器删除文件异常：", e);
            throw new MyException("从FTP服务器删除文件异常",e);
        }finally {
            returnClient(ftpClient);
        }
    }
    /**
     * 远程FTP上传文件
     * @param ftpPath 上传的目录
     * @param files 要上传的文件列表
     */
    public void upload(String ftpPath, List<File> files){
        FileInputStream is = null;
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            makeDir(ftpPath, ftpClient);
            for (File file : files) {
                is = new FileInputStream(file);
                ftpClient.storeFile(zhFileName(getFilePath(root,ftpPath,file.getName())), is);
                is.close();
            }
        } catch (Exception e) {
            log.error("上传FTP文件异常:", e);
            throw new MyException("文件上传失败",e);
        }finally {
            FileUtil.closeStream(is);
            returnClient(ftpClient);
        }
    }

    /**
     * 远程FTP上创建目录
     * @param ftpPath 要创建的目录
     */
    public void makeDir(String ftpPath) throws Exception {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            makeDir(ftpPath,ftpClient);
        }finally {
            returnClient(ftpClient);
        }
    }
    /**
     * 远程FTP上创建目录
     * @param ftpPath 要创建的目录
     * @param ftpClient ftp客户端
     */
    public void makeDir(String ftpPath, FTPClient ftpClient) {
        try {
            if(existDir(ftpPath,ftpClient)){
                return;
            }
            File f = new File(ftpPath);
            if(!existDir(getFilePath(f.getParent()),ftpClient)){
                //父目录不存在，先创建父目录
                makeDir(getFilePath(f.getParent()),ftpClient);
            }
            if(!ftpClient.makeDirectory(zhFileName(getFilePath(root, ftpPath)))){
                throw new MyException("创建目录失败："+ftpPath);
            }
        } catch (Exception ex) {
            slog.error("远程FTP生成目录异常:"+ftpPath, ex);
            throw new MyException("远程FTP生成目录异常:"+ftpPath,ex);
        }
    }
    /**
     * 判断目录是否存在 <br/>
     */
    public void existDir(String ftpPath) throws Exception {
        FTPClient ftpClient = null;
        try {
            ftpClient = borrowClient();
            existDir(ftpPath,ftpClient);
        }finally {
            returnClient(ftpClient);
        }
    }
    /**
    * 判断目录是否存在 <br/>
    * @author jingma
    * @param ftpPath 判断的目录
     * @param ftpClient ftp客户端
     * @return true：存在，false：不存在
    */
    public boolean existDir(String ftpPath, FTPClient ftpClient){
        try {
            setPath(ftpPath, ftpClient);
        } catch (Exception e) {
            return false;
        }
        return true;
    }
    /**
     * 编码文件名为：ISO_8859_1 <br/>
     * @author jingma
     * @param fileName 文件名
     * @return 转码的文件名
     */
    private static String zhFileName(String fileName) {
        //win下执行转换会乱码，不转中文不乱码
//        return new String(fileName.getBytes(), StandardCharsets.ISO_8859_1);
        return fileName;
    }

    /**
     * 设置当前文件夹
     * @param ftpPath 路径
     * @param ftpClient ftp客户端
     */
    private void setPath(String ftpPath, FTPClient ftpClient) throws Exception {
        String fp = getFilePath(root,ftpPath);
        String pwd;
        int j=0;
        ftpClient.changeWorkingDirectory(zhFileName(fp));
        pwd = pwd(ftpClient);
        if(pwd==null){
            throw new MyException("获取当前目录失败："+ftpPath);
        }
        if(!fp.equals(pwd)){
            throw new MyException("切换目录失败："+fp+"->"+pwd);
        }
    }
    public static String pwd(FTPClient ftpClient){
        try {
            String pwd = ftpClient.printWorkingDirectory();
            int i = 0;
            while (pwd==null&&i++<20){
                Thread.sleep(100);
                pwd = ftpClient.printWorkingDirectory();
            }
            return pwd;
        }catch (Exception e){
            throw new MyException("获取目录失败",e);
        }
    }
    public static SjztPooledObjectFactory createFtpFactory(SysSjglSjzt sjzt) {
        parseSjztFtp(sjzt);
        return new SjztPooledObjectFactory(sjzt);
    }

    /**
     * 解析ftp连接 <br/>
     * 解析出ip、port、encodeing
     * @author jingma
     * @param sjztObj 数据载体对象
     * @return 解析添加了相关信息的数据载体
     */
    public static SysSjglSjzt parseSjztFtp(SysSjglSjzt sjztObj) {
        Matcher m = ftpUrlZz.matcher(sjztObj.getLjc());
        if(m.find()){
            sjztObj.setIp(m.group(2));
            sjztObj.setPort(TypeUtils.castToInt(valByDef(m.group(4),21)));
        }else{
            throw new MyException("ftp连接串解析出错");
        }
        return sjztObj;
    }
}
