package cn.benma666.myutils;

import cn.benma666.constants.UtilConst;
import cn.benma666.crypt.DesUtil;
import cn.benma666.crypt.MD5Util;
import cn.benma666.dict.Jmfs;
import cn.benma666.dict.Ljpd;
import cn.benma666.domain.SysSjglSjzd;
import cn.benma666.exception.BusinessException;
import cn.benma666.exception.MyException;
import cn.benma666.iframe.Conf;
import cn.benma666.sm.FMSM1;
import cn.benma666.sm.FMSM4;
import cn.benma666.sm.SM3;
import cn.benma666.sm.SM4;
import cn.benma666.sm.sm2.SM2EncDecUtils;
import cn.hutool.log.LogFactory;
import com.alibaba.druid.util.Utils;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.util.TypeUtils;
import com.github.stuxuhai.jpinyin.PinyinException;
import com.github.stuxuhai.jpinyin.PinyinFormat;
import com.github.stuxuhai.jpinyin.PinyinHelper;
import org.apache.commons.lang.StringUtils;
import org.beetl.ext.fn.EmptyExpressionFunction;

import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 字符串工具类 <br/>
 * date: 2021年1月6日 <br/>
 *
 * @author jingma
 * @version 0.1
 */
public class StringUtil extends StringUtils {
    /**
     * 子节点KEY
     */
    public static final String KEY_CHILDREN = "children";
    public static final EmptyExpressionFunction isEmpty = new EmptyExpressionFunction();

    /**
     * @return 随机32位大写UUID
     * @author jingma
     */
    public static String getUUIDUpperStr() {
        return UUID.randomUUID().toString().replace("-", "").toUpperCase();
    }

    /**
     * 判断对象是否为空或空字符串 <br/>
     *
     * @param obj 对象
     * @return true：对象为空或空字符串，false：非空
     * @author jingma
     */
    public static boolean isBlank(Object obj) {
        return isEmpty.call(new Object[]{obj},null) || isBlank(obj.toString());
    }
    public static <T> T requireNonNull(T obj,String msg) {
        if(isBlank(obj)){
            throw new BusinessException(msg);
        }
        return obj;
    }
    public static boolean isNotBlank(Object obj) {
        return !isBlank(obj);
    }

    /**
     * 当传入的值为空时，采用默认值 <br/>
     *
     * @param val 传入值
     * @param def 默认值
     * @return val为空时采用def
     * @author jingma
     */
    public static <T> T valByDef(T val, T def) {
        if (isBlank(val)) {
            return def;
        } else {
            return val;
        }
    }

    /**
     * 获取第一个不为空的值
     * @param ts 值列表
     * @param <T> 泛型
     * @return 第一个不为空的值
     */
    public static <T> T fastNotNull(T... ts){
        for(T t : ts){
            if(!isBlank(t)){
                return t;
            }
        }
        return null;
    }

    /**
     * 字符串拼接
     * @param list 字符串列表
     * @param separator 分隔符
     * @return 拼接结果
     */
    public static String strJoin(List<String> list,String separator){
        return StringUtils.join(list,separator);
    }

    /**
     * 校验字符串是否纯数字，无长度限制
     *
     * @param str 待验证字符串
     * @return true：数字，false：非数字
     */
    public static Boolean validateNumber(String str) {
        if (isBlank(str)) {
            return false;
        }
        Pattern pattern = Pattern.compile("\\d+(\\.\\d+)?$");
        Matcher matcher = pattern.matcher(str);
        return matcher.matches();
    }
    /**
     * @Description 将驼峰转为下划线
     */
    public static String humpToUnderline(String str) {
        if (isBlank(str)) {
            return null;
        }
        Pattern compile = Pattern.compile("[A-Z]");
        Matcher matcher = compile.matcher(str);
        StringBuffer sb = new StringBuffer();
        while(matcher.find()) {
            matcher.appendReplacement(sb,  "_" + matcher.group(0).toLowerCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * 下划线规则的命名转换为驼峰式命名 <br/>
     *
     * @param str 原名称
     * @return 转换后名称
     * @author jingma
     */
    public static String underlineTohump(String str) {
        if (isBlank(str)) {
            return null;
        }
        str = str.toLowerCase();
        Pattern compile = Pattern.compile("_[a-z]");
        Matcher matcher = compile.matcher(str);
        StringBuffer sb = new StringBuffer();
        while(matcher.find()) {
            matcher.appendReplacement(sb,  matcher.group(0).toUpperCase().replace("_",""));
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * 将字符串首字母转换为大写 <br/>
     *
     * @param name 原名称
     * @return 转换结果
     * @author jingma
     */
    public static String upperCaseFast(String name) {
        if (isBlank(name)) {
            return null;
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }
    /**
     * 将字符串首字母转换为小写 <br/>
     *
     * @param name 原名称
     * @return 转换结果
     * @author jingma
     */
    public static String lowerCaseFast(String name) {
        if (isBlank(name)) {
            return null;
        }
        return name.substring(0, 1).toLowerCase() + name.substring(1);
    }

    /**
     * beetl-支持通配符的判断 <br/>
     *
     * @param zddm  字段名称
     * @param value 值
     * @author jingma
     */
    public static String addInput(String zddm, Object value) {
        if (isBlank(value)) {
            return "";
        }
        if (value.toString().contains("%")) {
            return " like #{sql.kzcs." + zddm + "_like} ";
        } else {
            return " = #{yobj." + zddm + "} ";
        }
    }

    /**
     * beetl-时间范围查询 <br/>
     *
     * @param field 字段对象
     * @param yobj  前端传入表单
     * @author jingma
     */
    public static void clEqDate(SysSjglSjzd field, JSONObject yobj) {
        String zddm = field.getZddm();
        if(isBlank(yobj.get(zddm))){
            return;
        }
        String start;
        String end = null;
        if(yobj.get(zddm) instanceof List){
            JSONArray value = yobj.getJSONArray(zddm);
            if (isBlank(value)) {
                return;
            }
            start = value.getString(0);
            if(value.size()>1){
                end = value.getString(1);
            }
        }else {
            String[] values = yobj.getString(zddm).split(";");
            start = values[0];
            if(values.length>1){
                end = values[1];
            }
        }
        //获取后端时间格式
        String hdgs = field.getKzxxObj().getString("$.yzgz.insert.date.value");
        if ("date".equals(hdgs)||isBlank(hdgs)) {
            //库中为date类型时，这里还是转换为14位时间
            hdgs = DateUtil.DATE_FORMATTER14;
        }
        if (isNotBlank(start)) {
            yobj.put(zddm + "_start", DateUtil.doFormatDate(start, hdgs));
        }
        if (isNotBlank(end)) {
            yobj.put(zddm + "_end", DateUtil.doFormatDate(end, hdgs));
        }
    }

    /**
     * 基于字段配置用des算法对字段进行加密 <br/>
     *
     * @param data  待加密的数据
     * @param field 字段信息
     * @return 加密结果
     * @author jingma
     */
    public static String desEnByField(String data, SysSjglSjzd field) {
        String jmfs = field.getKzxxObj().getString("$.kjkz.jmsf");
        Jmfs jmfsE;
        try {
            jmfsE = Jmfs.valueOf(valByDef(jmfs,"des"));
        }catch (IllegalArgumentException e){
            jmfsE = Jmfs.des;
        }
        return encodeing(data,jmfsE,Conf.getUtilConfig().getData().getPassword(),
                field.getKzxxObj().getString("$.kjkz.mm"));
    }

    /**
     * 加密
     * @param data 待加密数据
     * @param jmsf 加密方式
     * @param mm1 一级密码
     * @param mm2 二级密码
     * @return 结果
     */
    public static String encodeing(String data, Jmfs jmsf, String mm1, String mm2) {
        //加密算法
        SM4 sm4 = new SM4();
        switch (jmsf){
            case md5:
                return MD5Util.encode(data);
            case sm3:
                try{
                    return SM3.decode(data);
                }catch (Exception e){
                    throw new MyException("SM3加密失败："+data,e);
                }
            case des:
                if (isBlank(mm2)) {
                    mm2 = mm1;
                } else {
                    mm2 = DesUtil.encrypt(mm2, mm1);
                }
                return DesUtil.encrypt(data, mm2);
            case sm4ecb:
                //国产加密算法
                if (isBlank(mm2)) {
                    mm2 = mm1;
                } else {
                    sm4.setSecretKey(mm1);
                    mm2 = sm4.encryptDataToString_ECB(mm2);
                }
                sm4.setSecretKey(mm2);
                return sm4.encryptDataToString_ECB(data);
            case sm4cbc:
                //国产加密算法
                sm4.setSecretKey(mm1);
                if (isBlank(mm2)) {
                    sm4.setIv(mm1);
                } else {
                    sm4.setIv(mm2);
                }
                return sm4.encryptDataToString_CBC(data);
            case fmsm1:
                //渔翁加密机sm1
                FMSM1 fmsm1 = new FMSM1(mm1);
                return Base64.getEncoder().encodeToString(fmsm1.InternalSM1Enc(Integer.parseInt(mm2),
                        "CBC",true,data.getBytes(StandardCharsets.UTF_8)));
            case fmsm4:
                //渔翁加密机sm4
                FMSM4 fmsm4 = new FMSM4(mm1);
                return Base64.getEncoder().encodeToString(fmsm4.InternalSM4Enc(Integer.parseInt(mm2),
                        "CBC",true,data.getBytes(StandardCharsets.UTF_8)));
            case sm2:
//                SM2 sm02 = new SM2();
//                ECPoint publicKey = SM2.stringToPublicKey(mm2);
//                return ByteUtil.byteToHex(sm02.encrypt(data,publicKey));
                try {
                    return SM2EncDecUtils.encrypt(ByteUtil.hexToByte(mm2),data.getBytes(StandardCharsets.UTF_8));
                } catch (IOException e) {
                    throw new MyException("sm2加密失败",e);
                }
            case sm2Dcmy:
                //sm2-生成秘钥
                try {
                    return SM2EncDecUtils.generateKeyPair().toString();
                } catch (Exception e) {
                    throw new MyException("sm2生成秘钥失败失败",e);
                }
            default:
                throw new MyException("暂不支持的算法："+jmsf);
        }
    }
    /**
     * 生成秘钥
     * @param val 待处理数据
     * @param jmsf 加密方式
     * @return 结果
     */
    public static String generateKeyPair(String val,String jmsf) {
        switch (jmsf){
            case "sm2":
                //生成秘钥d对应
                try {
                    return SM2EncDecUtils.generateKeyPair().toString();
                } catch (Exception e) {
                    throw new MyException("生成秘钥失败失败",e);
                }
            default:
                throw new MyException("暂不支持的算法："+jmsf);
        }
    }

    /**
     * 基于字段配置用des算法对字段进行解密密 <br/>
     *
     * @param data  待解密数据
     * @param field 字段信息
     * @return 解密结果
     * @author jingma
     */
    public static String desDeByField(String data, SysSjglSjzd field) {
        Jmfs jmfs;
        try {
            jmfs = Jmfs.valueOf(valByDef(field.getKzxxObj().getString("$.kjkz.jmsf"),Jmfs.des.name()));
        }catch (IllegalArgumentException e){
            jmfs = Jmfs.des;
        }
        return decodeing(data,jmfs,Conf.getUtilConfig().getData().getPassword(),
                field.getKzxxObj().getString("$.kjkz.mm"));
    }
    /**
     * 解密
     * @param data 待解密数据
     * @param jmsf 解密方式
     * @param mm1 一级密码
     * @param mm2 二级密码
     * @return 结果
     */
    public static String decodeing(String data, Jmfs jmsf,String mm1,String mm2) {
        //加密算法
        SM4 sm4 = new SM4();
        switch (jmsf){
            case md5:
                throw new MyException("md5加密信息无法解密");
            case sm3:
                throw new MyException("sm3加密信息无法解密");
            case des:
                if (isBlank(mm2)) {
                    mm2 = mm1;
                } else {
                    mm2 = DesUtil.encrypt(mm2, mm1);
                }
                return DesUtil.decrypt(data, mm2);
            case sm4ecb:
                //国产加密算法
                if (isBlank(mm2)) {
                    mm2 = mm1;
                } else {
                    sm4.setSecretKey(mm1);
                    mm2 = sm4.encryptDataToString_ECB(mm2);
                }
                sm4.setSecretKey(mm2);
                return sm4.decryptDataToString_ECB(data);
            case sm4cbc:
                //国产加密算法
                sm4.setSecretKey(mm1);
                if (isBlank(mm2)) {
                    sm4.setIv(mm1);
                } else {
                    sm4.setIv(mm2);
                }
                return sm4.decryptDataToString_CBC(data);
            case fmsm1:
                //渔翁加密机sm1
                FMSM1 fmsm1 = new FMSM1(mm1);
                return new String(fmsm1.InternalSM1Dec(Integer.parseInt(mm2),"CBC",
                        true,Base64.getDecoder().decode(data)));
            case fmsm4:
                //渔翁加密机sm4
                FMSM4 fmsm4 = new FMSM4(mm1);
                return new String(fmsm4.InternalSM4Dec(Integer.parseInt(mm2),"CBC",
                        true,Base64.getDecoder().decode(data)));
            case sm2:
//                SM2 sm02 = new SM2();
//                BigInteger privateKey = SM2.stringToPrivateKey(mm2);
//                return sm02.decrypt(ByteUtil.hexToByte(data.toUpperCase()), privateKey);
                try {
                    return new String(SM2EncDecUtils.decrypt(ByteUtil.hexToByte(mm2),ByteUtil.hexToByte(data.toUpperCase())), StandardCharsets.UTF_8);
                } catch (IOException e) {
                    throw new MyException("sm2加密失败",e);
                }
            default:
                throw new MyException("暂不支持的算法："+jmsf);
        }
    }

    public static String getSimpleSpell(Reader source) throws PinyinException {
        return getSimpleSpell(Utils.read(source));
    }
    /**
     * 获取字符串简拼 <br/>
     *
     * @param source 来源字符串
     * @return 来源字符串的简拼
     * @throws PinyinException 拼音异常
     * @author jingma
     */
    public static String getSimpleSpell(String source) throws PinyinException {
        if (StringUtil.isBlank(source)) {
            return null;
        }
        return PinyinHelper.getShortPinyin(source).toUpperCase();
    }

    public static String getFullSpell(Reader source) throws PinyinException {
        return getFullSpell(Utils.read(source));
    }
    /**
     * 获取字符串全拼 <br/>
     *
     * @param source 来源字符串
     * @return 来源字符串的全拼
     * @throws PinyinException 拼音异常
     * @author jingma
     */
    public static String getFullSpell(String source) throws PinyinException {
        if (StringUtil.isBlank(source)) {
            return null;
        }
        return PinyinHelper.convertToPinyinString(source, "",
                PinyinFormat.WITHOUT_TONE).toUpperCase();
    }

    /**
     * Convert byte[] to hex
     * string.这里我们可以将byte转换成int，然后利用Integer.toHexString(int)来转换成16进制字符串。
     *
     * @param byteArray byte[] data
     * @return hexString string
     */
    public static String bytesToHexString(byte[] byteArray) {
        if (byteArray == null || byteArray.length < 1)
            throw new IllegalArgumentException(
                    "this byteArray must not be null or empty");

        final StringBuilder hexString = new StringBuilder();
        for (byte b : byteArray) {
            if ((b & 0xff) < 0x10)// 0~F前面不零
                hexString.append("0");
            hexString.append(Integer.toHexString(0xFF & b));
        }
        return hexString.toString().toLowerCase();
    }

    /**
     * Convert hex string to byte[]
     *
     * @param hexString the hex string
     * @return byte[]
     */
    public static byte[] hexStringToBytes(String hexString) {
        if (StringUtil.isEmpty(hexString))
            throw new IllegalArgumentException("this hexString must not be empty");
        hexString = hexString.toLowerCase();
        final byte[] byteArray = new byte[hexString.length() / 2];
        int k = 0;
        for (int i = 0; i < byteArray.length; i++) {// 因为是16进制，最多只会占用4位，转换成字节需要两个16进制的字符，高位在先
            byte high = (byte) (Character.digit(hexString.charAt(k), 16) & 0xff);
            byte low = (byte) (Character.digit(hexString.charAt(k + 1), 16) & 0xff);
            byteArray[i] = (byte) (high << 4 | low);
            k += 2;
        }
        return byteArray;
    }

    /**
     * 转为utf8 <br/>
     *
     * @param s 待转字符串
     * @return 转换结果
     * @author jingma
     */
    public static String toUtf8(String s) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (c >= 0 && c <= 255) {
                sb.append(c);
            } else {
                byte[] b;
                try {
                    b = Character.toString(c).getBytes(StandardCharsets.UTF_8);
                } catch (Exception ex) {
                    LogFactory.get().debug(s + "转换异常", ex);
                    b = new byte[0];
                }
                for (int value : b) {
                    int k = value;
                    if (k < 0) k += 256;
                    sb.append("%").append(Integer.toHexString(k).toUpperCase());
                }
            }
        }
        return sb.toString();
    }

    public static int whether(Object obj) {
        if(obj==null){
            return Ljpd.FALSE.getCode();
        }
        return (TypeUtils.castToBoolean(obj)? Ljpd.TURE.getCode() :Ljpd.FALSE.getCode());
    }
    /**
     * 构建的树数据,要求根节点为空，默认根据为：_root
     * @param list 数据列表
     * @param upkey 父节点字段
     * @param key 与父节点对应的字段
     */
    public static JSONArray buildTree(List<JSONObject> list,String upkey,String key) {
        return buildTree(list,upkey,key,null);
    }

    /**
     * 构建的树数据,要求根节点为空
     * @param list 数据列表
     * @param upkey 父节点字段
     * @param key 与父节点对应的字段
     * @param rootKey 根节点key，默认：_root
     */
    public static JSONArray buildTree(List<JSONObject> list,String upkey,String key,String  rootKey) {
        if(isBlank(rootKey)){
            //根节点key
            rootKey = "_root";
        }
        //按父节点拆分
        JSONObject zdUpnode = new JSONObject(true);
        for(JSONObject zd : list){
            //父节点，默认值尽量为非常规值
            String upnode = valByDef(zd.getString(upkey),rootKey);
            //父节点对象
            JSONArray upnodeObj = zdUpnode.getJSONArray(upnode);
            if(upnodeObj==null){
                upnodeObj = new JSONArray();
                zdUpnode.put(upnode,upnodeObj);
            }
            //设置到父节点的对象中
            upnodeObj.add(zd);
        }
        JSONArray zdTree = new JSONArray();
        if(!zdUpnode.containsKey(rootKey)){
            //无根节点
            return zdTree;
        }
        zdTree.addAll(zdUpnode.getJSONArray(rootKey));
        zdUpnode.remove(rootKey);
        buildTree(zdTree,zdUpnode,key);
        return zdTree;
    }

    /**
     * 构建树形结构
     * @param list 当前字典树
     * @param zdUpnode 父节点结合
     * @param key 与父节点对应的字段
     */
    private static void buildTree(JSONArray list,JSONObject zdUpnode,String key){
        for(int i=0;i<list.size();i++){
            JSONObject zd = list.getJSONObject(i);
            JSONArray children = zdUpnode.getJSONArray(zd.getString(key));
            if(children==null){
                continue;
            }
            //使用过的就移除，避免重复使用形成循环依赖
            zdUpnode.remove(zd.getString(key));
            zd.put(KEY_CHILDREN,children);
            buildTree(children,zdUpnode,key);
        }
    }

    /**
     * 按字节计算长度进行截取
     * @param obj 待截取字符串
     * @param maxLength 最大长度
     * @return 结果
     */
    public static String substrByByte(Object obj,int maxLength){
        if(isBlank(obj)){
            return "";
        }
        String str = obj.toString();
        long length = str.getBytes(StandardCharsets.UTF_8).length;
        if(length<=maxLength){
            return str;
        }
        str = str.substring(0, (int) ((long)maxLength*str.length()/length));
        if(str.getBytes(StandardCharsets.UTF_8).length>maxLength){
            return substrByByte(str,maxLength);
        }
        return str;
    }
}
