/**
* Project Name:myutils
* Date:2017年2月26日
* Copyright (c) 2017, jingma All Rights Reserved.
*/

package cn.benma666.myutils;

import cn.benma666.constants.Charset;
import cn.benma666.exception.MyException;
import cn.benma666.iframe.BasicObject;
import cn.benma666.iframe.Conf;
import com.alibaba.druid.util.Utils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.*;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.cert.X509Certificate;
import java.util.Map;

/**
 * Http工具 <br/>
 * date: 2017年2月26日 <br/>
 * @author jingma
 * @version 0.1
 */
@Setter
@Getter
@AllArgsConstructor
@NoArgsConstructor
@Builder
public class HttpUtil extends BasicObject {
    /**
     * 请求方法key
     */
    public static final String REQUEST_METHOD = "RequestMethod";
    /**
     * 权限认证请求头key
     */
    public static final String AUTHORIZATION = "Authorization";
    /**
     * 请求内容格式请求头
     */
    public static final String CONTENT_TYPE = "Content-Type";
    /**
     * 默认请求编码
     */
    public static final String DEFAULT_QQBM = "UTF-8";
    /**
     * 默认超时时长
     */
    public static final int DEFAULT_CSSC = 120000;
    /**
     * 单例
     */
    protected static HttpUtil hu;
    /**
     * 请求编码
     */
    @Builder.Default
    private String qqbm = DEFAULT_QQBM;
    /**
     * 超时时长
     */
    @Builder.Default
    private int cssc = DEFAULT_CSSC;
    /**
     * 请求信息头
     */
    @Builder.Default
    private JSONObject rp = new JSONObject();
    /**
     * 返回信息头
     */
    @Builder.Default
    private JSONObject repHeards = new JSONObject();
    /**
     * http请求最终发起工具
     */
    @Builder.Default
    private HttpInterface hi = defaultHi();
    /**
     * 请求地址
     */
    private String url;
    /**
     * 请求参数
     */
    private Object param;
    /**
     * 上传文件类接口的文件参数名
     */
    @Builder.Default
    private String fileKey = "file";
    /**
     * 请求方式
     */
    @Builder.Default
    private String qqfs = HttpPost.METHOD_NAME;

    /**
     * 要避免一个实例多用途，避免出现参数相互影响
     */
    public HttpUtil(HttpInterface hi){
        this.setHi(hi);
    }

    /**
     * 获取单例
     */
    private static HttpUtil getInstance(){
        if(hu==null){
            hu = builder().build();
        }
        return hu;
    }
    /**
    * post请求-表单入参返回json结果 <br/>
    * @author jingma
    * @param url 连接
    * @param param 参数
    * @return json对象
    */
    public static JSONObject doJosnByFrom(String url, Object param){
        JSONObject rp = new JSONObject();
        rp.put(CONTENT_TYPE,"application/x-www-form-urlencoded;charset="+DEFAULT_QQBM);
        String result = doStr(url, param, DEFAULT_QQBM, DEFAULT_CSSC,rp);
        if(StringUtil.isNotBlank(result)){
            return JSON.parseObject(result);
        }else {
            return new JSONObject();
        }
    }
    public JSONObject josnByFrom(){
        return josnByFrom(url,param);
    }
    public JSONObject josnByFrom(String url, Object param){
        rp.put(CONTENT_TYPE,"application/x-www-form-urlencoded;charset="+qqbm);
        String result = str(url, param, qqbm, cssc,rp);
        if(StringUtil.isNotBlank(result)){
            return JSON.parseObject(result);
        }else {
            return new JSONObject();
        }
    }
    /**
     * post请求-json入参返回json结果 <br/>
     * @author jingma
     * @param url 连接
     * @param param 参数
     * @return json对象
     */
    public static JSONObject doJosnByJson(String url, Object param){
        JSONObject rp = new JSONObject();
        rp.put(CONTENT_TYPE,WebUtil.CONTENTTYPE_APPLICATION_JSON);
        String result = doStr(url, param, DEFAULT_QQBM, DEFAULT_CSSC,rp);
        if(StringUtil.isNotBlank(result)){
            try {
                return JSON.parseObject(result);
            }catch (Exception e){
                throw new MyException("请求结果解析失败："+result,e);
            }
        }else {
            return new JSONObject();
        }
    }
    public JSONObject josnByJson(){
        return josnByJson(url,param);
    }
    public JSONObject josnByJson(String url, Object param){
        rp.put(CONTENT_TYPE,WebUtil.CONTENTTYPE_APPLICATION_JSON);
        String result = str(url, param, qqbm, cssc,rp);
        if(StringUtil.isNotBlank(result)){
            try {
                return JSON.parseObject(result);
            }catch (Exception e){
                throw new MyException("请求结果解析失败："+result,e);
            }
        }else {
            return new JSONObject();
        }
    }
    /**
     * get请求
     * @param url 连接
     * @param param 参数
     * @return 字符串
     */
    public static String doStrByGet(String url, Object param) {
        JSONObject rp = new JSONObject();
        rp.put(REQUEST_METHOD, HttpGet.METHOD_NAME);
        return doStr(url,param,DEFAULT_QQBM,DEFAULT_CSSC,rp);
    }
    public String strByGet() {
        return strByGet(url,param);
    }
    public String strByGet(String url, Object param) {
        rp.put(REQUEST_METHOD, HttpGet.METHOD_NAME);
        return str(url, param, qqbm, cssc, rp);
    }
    /**
    * post请求-表单入参返回字符串结果 <br/>
    * @author jingma
    * @param url 连接
    * @param param 参数
    * @return 字符串
    */
    public static String doStr(String url, Object param) {
        return doStr(url,param,DEFAULT_QQBM,DEFAULT_CSSC,new JSONObject());
    }
    public String str() {
        return str(url,param);
    }
    public String str(String url, Object param) {
        return str(url, param, qqbm, cssc,rp);
    }
    /**
    * post请求 <br/>
    * @author jingma
    * @param url 连接
    * @param param 参数,请求参数应该是 name1=value1&name2=value2 的形式。
    * @param qqbm 请求编码
    * @param cssc 超时时长
    * @param rp 请求属性
    * @return 字符串
    */
    public static String doStr(String url, Object param,String qqbm,int cssc,
                               JSONObject rp) {
        return doStr(url,param,qqbm,cssc,rp,new JSONObject());
    }
    public String str(String url, Object param,String qqbm,int cssc,
            JSONObject rp) {
        repHeards.clear();
        return str(url, param, qqbm, cssc, rp, repHeards);
    }
    /**
     * post请求 <br/>
     * @author jingma
     * @param url 连接
     * @param param 参数,请求参数应该是 name1=value1&name2=value2 的形式，或json串，需要自行设置请求头信息。
     * @param qqbm 请求编码
     * @param cssc 超时时长
     * @param rp 请求属性，不能为空，直接调用本方法时，采用new一个json对象
     * @param repHeards 响应头信息,需要提前new好对象
     * @return 字符串
     */
    public static String doStr(String url, Object param,String qqbm,int cssc,
                               JSONObject rp,JSONObject repHeards) {
        return getInstance().str(url,param,qqbm,cssc,rp,repHeards);
    }
    public String str(String url, Object param,String qqbm,int cssc,
                               JSONObject rp,JSONObject repHeards) {
        InputStream in = null;
        try {
            in = doHttp(getHi(),url, param, qqbm, cssc, rp, repHeards);
            byte[] bytes = Utils.readByteArray(in);
            String r = "";
            if(!isBlank(bytes)){
                r = new String(bytes,qqbm);
            }
            log.trace("请求结果：{}",r);
            return r;
        } catch (UnsupportedEncodingException e) {
            throw new MyException("不支持的编码："+qqbm,e);
        } catch (IOException e) {
            throw new MyException("读取输入流失败",e);
        }finally {
            //关闭输入流
            FileUtil.closeStream(in);
        }
    }
    public InputStream getInputStream() {
        return doHttp(getHi(),url, null, getQqbm(),
                getCssc(), getRp(), getRepHeards());
    }
    public InputStream getInputStream(String url) {
        return doHttp(getHi(),url, null, getQqbm(),
                getCssc(), getRp(), getRepHeards());
    }
    public InputStream getInputStream(String url, Object param, String qqbm, int cssc,
                                      JSONObject rp, JSONObject repHeards) {
        return doHttp(getHi(), url, param, qqbm, cssc, rp, repHeards);
    }
    public static InputStream inputStream(String url) {
        return inputStream(url, null, DEFAULT_QQBM,
                DEFAULT_CSSC, null, new JSONObject());
    }
    public static InputStream inputStream(String url, Object param, String qqbm, int cssc,
                                          JSONObject rp, JSONObject repHeards) {
        return doHttp(getInstance().getHi(),url, param, qqbm, cssc, rp, repHeards);
    }

    /**
     * 请求预处理
     */
    public static InputStream doHttp(HttpInterface hi,String url, Object param, String qqbm, int cssc,
                                    JSONObject rp, JSONObject repHeards) {
        if(rp ==null){
            rp = new JSONObject();
        }
        if(repHeards ==null){
            repHeards = new JSONObject();
        }
        if(!isBlank(param)&&param instanceof Map &&(isBlank(rp.getString(CONTENT_TYPE))
                ||rp.getString(CONTENT_TYPE).contains(ContentType.APPLICATION_FORM_URLENCODED.getMimeType()))){
            //参数为map对象，且请求方式是GET时，进行参数格式转换
            StringBuilder sb = new StringBuilder();
            for (Map.Entry e:((Map<?, ?>) param).entrySet()){
                if(isBlank(e.getValue())){
                    continue;
                }
                try {
                    sb.append(e.getKey()).append("=").append(URLEncoder.encode(e.getValue().toString(), Charset.UTF_8.getCode())).append("&");
                } catch (UnsupportedEncodingException unsupportedEncodingException) {
                    throw new MyException("参数编码异常："+e.getKey()+"="+e.getValue(),e);
                }
            }
            param = sb.toString();
        }
        if(isBlank(url)){
            throw new MyException("请求地址不能为空");
        }
        slog.trace("请求地址：{}，请求参数：{}",url,param);
        return hi.inputStream(url, param, qqbm, cssc, rp, repHeards);
    }

    /**
     * 跳过ssl证书
     */
    static SSLContext trustAllHttpsCertificates(){
        try {
            TrustManager[] trustAllCerts = new TrustManager[1];
            trustAllCerts[0] = new TrustAllManager();
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, null);
            return sc;
        }catch (Exception e){
            throw new MyException("创建忽略证书对象失败",e);
        }
    }
    static class TrustAllManager implements X509TrustManager {
        public X509Certificate[] getAcceptedIssuers() {
            return new X509Certificate[]{};
        }
        public void checkServerTrusted(X509Certificate[] certs, String authType) {
        }
        public void checkClientTrusted(X509Certificate[] certs, String authType) {
        }
    }

    //获取HostnameVerifier
    static HostnameVerifier getHostnameVerifier() {
        return (s, sslSession) -> {
            //未真正校检服务器端证书域名
            return true;
        };
    }

    private static HttpInterface defaultHi(){
        //默认http工具
        String httpgj = valByDef(Conf.getValByConfig("benma666.default-http-util"), "huc");
        //如下即是简单工厂模式
        switch (httpgj){
            case "huc":
                return HttpByHUC.getInstance();
            case "ok":
                return HttpByOk.getInstance();
            case "hc":
                return HttpByHttpClient.getInstance();
            default:
                throw new MyException("不支持的http工具："+httpgj);
        }
    }

    /**
     * 构建者
     */
    public static class HttpUtilBuilder {
        /**
         * 便捷添加请求属性
         * @param key 键
         * @param val 值
         * @return 构建者
         */
        public HttpUtilBuilder addRp(String key,Object val){
            if(rp$value==null){
                rp$value = new JSONObject();
                rp$set = true;
            }
            rp$value.put(key,val);
            return this;
        }
    }
}
