/**
 * Project Name:myutils
 * Date:2018年12月13日
 * Copyright (c) 2018, jingma All Rights Reserved.
 */

package cn.benma666.iframe;

import cn.benma666.constants.UtilConst;
import cn.benma666.dict.Cllx;
import cn.benma666.dict.Ljpd;
import cn.benma666.domain.SysSjglSjzd;
import cn.benma666.exception.VerifyRuleException;
import cn.benma666.myutils.DateUtil;
import cn.benma666.myutils.SfzhUtil;
import cn.benma666.myutils.StringUtil;
import cn.benma666.myutils.TmplUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.util.TypeUtils;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map.Entry;

/**
 * 验证规则 <br/>
 * date: 2018年12月13日 <br/>
 *
 * @author jingma
 */
public class VerifyRule extends BasicObject {
    /**
     * 值
     */
    public static final String VALUE = "value";
    /**
     * 提示信息
     */
    public static final String TS = "ts";

    /**
     * 规则校验 <br/>
     *
     * @param val      值
     * @param myParams 相关参数
     * @param rootRule     验证规则
     * @param cllx     处理类型
     * @return 验证结果
     * @author jingma
     */
    public static Object ruleVerify(Object val, MyParams myParams, JSONObject rootRule,String cllx) {
        JSONObject yzgz = rootRule.getJSONObject(cllx);
        if(yzgz==null){
            return val;
        }
        Object xxmsObj = yzgz.get("xxms");
        String xxms = xxmsObj==null?"该值":xxmsObj.toString();
        if (val==null) {
            //notNull
            if (yzgz.containsKey("notNull")) {
                //规则要求非空，且没有排除非空规则
                throw new VerifyRuleException(xxms+"必须传");
            } else {
                return null;
            }
        }
        if (StringUtil.isBlank(val)) {
            //notBlank
            if (yzgz.containsKey("notBlank")) {
                //规则要求非空，且没有排除非空规则
                throw new VerifyRuleException(xxms+"不能为空值");
            } else {
                return UtilConst.NULL_STR;
            }
        }
        String zdlb = null;
        MyParams zdParams = new MyParams();
        zdParams.sys().setCllx(cllx);
        zdParams.put(UtilConst.KEY_USER,myParams.get(UtilConst.KEY_USER));
        zdParams.other().setYobj(myParams.getJSONObject(UtilConst.KEY_YOBJ).clone());
        zdParams.put(UtilConst.KEY_OBJ,myParams.get(UtilConst.KEY_OBJ));
        try {
            for (String key : yzgz.keySet()) {
                if("xxms".equals(key)||"notNull".equals(key)||"notBlank".equals(key)){
                    //前面已经处理
                    continue;
                }
                if(StringUtil.isBlank(yzgz.getString(key))){
                    //规则为空，跳过
                    continue;
                }
                JSONObject gzObj = yzgz.getJSONObject(key);
                switch (key) {
                    case "null":
                        if (StringUtil.isNotBlank(val)) {
                            throw new VerifyRuleException(xxms+"必须为空");
                        }
                        break;
                    case "mustBe":
                        //mustBe:123,必须在字典规则的后面
                        mustBe(xxms, val.toString(), zdlb, gzObj);
                        break;
                    case "length":
                        //length:max:min
                        length(xxms, val.toString(), gzObj);
                        break;
                    case "zzbds":
                        //zzbds:[0-9]{8}:必须是8位的数字
                        //正则表达式
                        if (!val.toString().matches(gzObj.getString(VALUE))) {
                            throw new VerifyRuleException(xxms+gzObj.getString(TS));
                        }
                        break;
                    case "number":
                        number(xxms, val.toString());
                        break;
                    case "date":
                        val = date(xxms, val, gzObj);
                        break;
                    case "dateRange":
                        //时间范围校验
                        val = dataRange(val, myParams, xxms, gzObj);
                        break;
                    case "sfzh":
                        val = sfzh(val, xxms, val.toString());
                        break;
                    case "email":
                        if (!val.toString().contains("@")) {
                            throw new VerifyRuleException(xxms+"不是正确的邮箱");
                        }
                        break;
                    case "zd":
                        val = zd(xxms, val.toString(), gzObj,myParams,zdParams);
                        break;
                    case "zdpd":
                        //字典判断
                        zdpd(xxms, val.toString(), zdParams, gzObj);
                        break;
                    case "tmpl":
                        //设置值，便于在模板中使用
                        tmpl(myParams, xxms, val, gzObj);
                        break;
                    default:
                        throw new VerifyRuleException(xxms+"不支持验证规则:" + key);
                }
            }
        }catch (VerifyRuleException e){
            throw new VerifyRuleException(e.getMessage()+"，值为："+val);
        }
        return val;
    }

    private static Object zd(String xxms, String value, JSONObject gzObj, MyParams myParams,MyParams zdParams) {
        String zdlb = gzObj.getString(VALUE);
        zdParams.set("$.yobj.zdlb",zdlb);
        if (gzObj.getBooleanValue("zddx")) {
            if(Cllx.sjplsc.name().equals(myParams.sys().getCllx())){
                //数据批量上传场景，支持从名称转回代码
                value = DictManager.zdDmByMoreMc(zdlb,value);
            }
            zdParams.set("$.yobj.dm",value);
            //字段多选
            if (StringUtils.isNotEmpty(value)) {
                String[] dms = value.split(",");
                for (String s : dms) {
                    zdParams.set("$.yobj.dm",s);
                    JSONObject valObj = DictManager.zdObj(zdParams);
                    if (valObj == null) {
                        zdYcts(xxms, zdlb, gzObj);
                    }
                }
            }
            return value;
        } else {
            value = DictManager.zdDmByMc(zdlb,value);
            zdParams.set("$.yobj.dm",value);
            JSONObject obj = DictManager.zdObj(zdParams);
            if (obj == null) {
                zdYcts(xxms, zdlb, gzObj);
            }
            assert obj != null;
            return obj.getString("dm");
        }
    }
    private static void mustBe(String xxms, String value, String zdlb, JSONObject gzObj) {
        if (!value.equals(gzObj.getString(VALUE))) {
            String v = gzObj.getString(VALUE);
            if (zdlb != null) {
                v = DictManager.zdMcByDm(zdlb, v);
            }
            if(!value.equals(v)){
                throw new VerifyRuleException(xxms +"必须为：" + gzObj.getString(VALUE));
            }
        }
    }

    private static void length(String xxms, String value, JSONObject gzObj) {
        if (value.getBytes().length > gzObj.getIntValue("max")) {
            throw new VerifyRuleException(xxms +"长度不得大于："
                    + gzObj.getIntValue("max"));
        } else if (value.getBytes().length < gzObj.getIntValue("min")) {
            throw new VerifyRuleException(xxms +"长度不得小于："
                    + gzObj.getIntValue("min"));
        }
    }

    private static void number(String xxms, String value) {
        try {
            TypeUtils.castToDouble(value);
        } catch (NumberFormatException e) {
            throw new VerifyRuleException(xxms +"要求是数字");
        }
    }

    private static Object date(String xxms, Object value, JSONObject gzObj) {
        Object val;
        Date d = DateUtil.parseDate(value,valByDef(gzObj
                .getBoolean("lenient"),true));
        if (d == null) {
            throw new VerifyRuleException(xxms +"必须是时间格式");
        }
        String f = DateUtil.DATE_FORMATTER14;
        if (gzObj.containsKey(VALUE)) {
            f = gzObj.getString(VALUE);
        }
        val = DateUtil.doFormatDate1(d, f);
        return val;
    }

    private static Object sfzh(Object val, String xxms, String value) {
        if (!SfzhUtil.validateCard(value)) {
            throw new VerifyRuleException(xxms +"不是正确的身份证号码");
        }
        //自动转18位，且x转为大写
        String sfzh = SfzhUtil.conver15CardTo18(value);
        if (StringUtil.isNotBlank(sfzh)) {
            val = sfzh.toUpperCase();
        }
        return val;
    }

    private static void tmpl(MyParams myParams, String xxms, Object value, JSONObject gzObj) {
        myParams.set("$.sys.value", value);
        myParams.set("$.sys.gzObj", gzObj);
        if (!TypeUtils.castToBoolean(TmplUtil.buildStr(gzObj.getString(VALUE), myParams))) {
            throw new VerifyRuleException(xxms + gzObj.getString(TS));
        }
    }

    private static void zdpd(String xxms, String value, MyParams zdParams, JSONObject gzObj) {
        zdParams.set("$.yobj.zdlb", gzObj.getString(VALUE));
        zdParams.set("$.yobj.dm", value);
        JSONObject obj = DictManager.zdObjNoCache(zdParams);
        if (obj==null) {
            throw new VerifyRuleException(xxms +"没有配置对应的字典判断字典类别");
        }
        if (Ljpd.FALSE.getCode() == obj.getIntValue("mc")) {
            throw new VerifyRuleException(xxms + gzObj.getString(TS));
        }
    }

    /**
     * 时间范围校验
     */
    private static Object dataRange(Object val, MyParams myParams, String xxms, JSONObject gzObj) {
        List<String> dr;
        if(val instanceof String){
            dr = Arrays.asList(val.toString().split(";"));
        }else {
            dr = (List<String>) val;
        }
        Date d1;
        JSONArray dr1 = new JSONArray();
        for (String d2:dr){
            if(isBlank(d2)){
                dr1.add(null);
            }else {
                d1 = DateUtil.parseDate(d2,valByDef(gzObj
                        .getBoolean("lenient"),true));
                if (d1 == null) {
                    throw new VerifyRuleException(xxms +"必须是时间格式");
                }
                String f1 = DateUtil.DATE_FORMATTER14;
                if (gzObj.containsKey(VALUE)) {
                    f1 = gzObj.getString(VALUE);
                }
                dr1.add(DateUtil.doFormatDate1(d1, f1));
            }
        }
        if(gzObj.containsKey("start")){
            String start = TmplUtil.buildStr(gzObj.getString("start"), myParams);
            if(isBlank(dr1.get(0))||dr1.getString(0).compareTo(start)<0){
                throw new VerifyRuleException("开始时间必须大于："+start);
            }
        }
        if(gzObj.containsKey("end")){
            String end = TmplUtil.buildStr(gzObj.getString("end"), myParams);
            if(isBlank(dr1.get(1))||dr1.getString(1).compareTo(end)>0){
                throw new VerifyRuleException("结束时间必须小于："+end);
            }
        }
        if(gzObj.containsKey("range")){
            if(isBlank(dr1.get(0))||isBlank(dr1.get(1))){
                throw new VerifyRuleException("时间范围必须小于："+ gzObj.get("range"));
            }
            if(DateUtil.parseDate(dr1.getString(1)).getTime()-DateUtil.parseDate(
                    dr1.getString(0)).getTime()>DateUtil.scStrToLong(gzObj.getString("range"))){
                throw new VerifyRuleException("时间范围必须小于："+ gzObj.get("range"));
            }
        }
        return dr1;
    }

    /**
     * 字典异常提示
     */
    private static void zdYcts(String xxms, String zdlb, JSONObject gzObj) {
        JSONObject list = DictManager.zdMap(zdlb);
        if (!valByDef(gzObj.getBoolean("zszdx"), true)||list==null) {
            throw new VerifyRuleException(xxms +"不在字典范围内，请在单个申请页面的该数据项的下拉字典中查看相关字典范围");
        }
        List<String> mcs = new ArrayList<>();
        for (Entry<String, Object> e : list.entrySet()) {
            mcs.add(((JSONObject) e.getValue()).getString("mc"));
        }
        throw new VerifyRuleException(xxms +"不在字典范围内，字典如下：" + mcs);
    }

    /**
     * 规则解析 <br/>
     *
     * @param field 字段对象
     * @param yzgz  规则
     * @author jingma
     */
    public static String rulejx(SysSjglSjzd field, JSONObject yzgz) {
        //规则解析结果
        StringBuilder ruleJx = new StringBuilder();
        int idx = 1;
        for (String rule : yzgz.keySet()) {
            if("xxms".equals(rule)){
                continue;
            }
            ruleJx.append(idx).append("、").append(getGzms(field,rule, yzgz.getJSONObject(rule))).append("，");
            idx++;
        }
        if(StringUtil.isNotBlank(field.getZdms())){
            ruleJx.append(idx++).append("、字段描述：").append(field.getZdms()).append("，");
        }
        if(StringUtil.isNotBlank(field.getZdts())){
            ruleJx.append(idx).append("、字段提示：").append(field.getZdts()).append("，");
        }
        return ruleJx.toString();
    }

    /**
     * 获取规则描述 <br/>
     * @param field 字段
     * @param ruleName 规则名称
     * @param ruleObj  规则对象
     * @return 规则描述
     * @author jingma
     */
    public static String getGzms(SysSjglSjzd field, String ruleName, JSONObject ruleObj) {
        if ("zd".equals(ruleName)) {
            String zdxms;
            if (valByDef(ruleObj.getBoolean("zszdx"), true)) {
                JSONObject list = DictManager.zdMap(ruleObj.getString(VALUE));
                List<String> mcs = new ArrayList<>();
                if (list == null || list.size() == 0) {
                    zdxms = "该字典项不支持罗列，请在单个新建页面的该数据项的下拉字典中查看相关字典范围";
                } else {
                    for (Entry<String, Object> e : list.entrySet()) {
                        mcs.add(((JSONObject) e.getValue()).getString("mc"));
                    }
                    zdxms = "字典如下：" + mcs;
                    field.setExcelZdlb(mcs);
                }
            } else {
                zdxms = "请在单个新建页面的该数据项的下拉字典中查看相关字典范围";
            }
            ruleObj.put("zdxms", zdxms);
        }
        return TmplUtil.buildStr(DictManager.zdMcByDm("SYS_SJGL_YZGZ", ruleName), ruleObj);
    }
}
