package cn.benma666.sjzt.rabbit;

import cn.benma666.constants.UtilConst;
import cn.benma666.domain.SysSjglSjzt;
import cn.benma666.domain.SysSjglZnjh;
import cn.benma666.exception.MyException;
import cn.benma666.iframe.InterfaceLog;
import cn.benma666.myutils.FileUtil;
import cn.benma666.sjzt.*;
import cn.benma666.sjzt.bdwj.BdwjFile;
import com.alibaba.druid.util.Utils;
import com.rabbitmq.client.*;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

public class RabbitMQ extends BasicSjzt {

    /**
     * 默认RabbitMQ
     */
    private static RabbitMQ rabbitMQ = null;

    private final GenericObjectPool objectPool;

    protected RabbitMQ(String name, SysSjglSjzt sjzt) {
        super(name, sjzt);
        SjztPooledObjectFactory factory = new SjztPooledObjectFactory(sjzt);
        GenericObjectPoolConfig config = new GenericObjectPoolConfig();
        config.setTestOnCreate(true);
        config.setTestOnBorrow(true);
        config.setMinIdle(2);
        config.setMaxIdle(5);
        config.setMaxTotal(20);
        //最大等待60秒
        config.setMaxWait(Duration.ofSeconds(300));
        objectPool = new GenericObjectPool(factory, config);
        RabbitMQClient rabbitMQClient = null;
        try {
            //测试获取连接
            rabbitMQClient = borrowClient();
        } catch (Exception e) {
            throw new MyException(name + "初始化失败", e);
        } finally {
            returnClient(rabbitMQClient);
        }
        if (rabbitMQ == null) {
            rabbitMQ = this;
        }
        cache.put(name,this);
    }
    /**
     * 退回客户端
     * @param client 需要退回的客户端
     */
    public void returnClient(RabbitMQClient client) {
        log.trace("释放回连接池：{}",name);
        if (client != null) {
            objectPool.returnObject(client);
        }
    }

    /**
     * 获取客户端
     */
    public RabbitMQClient borrowClient() throws Exception {
        log.trace("从连接池获取：{}",name);
        return (RabbitMQClient) objectPool.borrowObject();
    }

    /**
     * 验证客户端是否可用
     */
    public static boolean validateClient(SysSjglSjzt sjzt,Object client) {
        RabbitMQClient adminClient = (RabbitMQClient) client;
        return adminClient != null && adminClient.getChannel() !=null && adminClient.getChannel().isOpen();
    }

    /**
     * 关闭客户端
     */
    public static void destroyClient(SysSjglSjzt sjzt,Object client) throws Exception {
        RabbitMQClient rabbitMQClient = (RabbitMQClient) client;
        rabbitMQClient.close();
    }
    public static RabbitMQClient createClient(SysSjglSjzt sjzt) {
        try {
            return new RabbitMQClient(sjzt);
        } catch (Exception e) {
            throw new MyException("rabbitMQ客户端创建失败",e);
        }
    }

    public Object exec(SjztExecRunnable<RabbitMQClient> exec) {
        RabbitMQClient rabbitMQClient = null;
        try {
            rabbitMQClient = borrowClient();
            return exec.exec(rabbitMQClient);
        } catch (Exception e) {
            throw new MyException("rabbitmq执行异常", e);
        } finally {
            if (rabbitMQClient != null) {
                objectPool.returnObject(rabbitMQClient);
            }
        }
    }
    public static RabbitMQ use(String name) {
        return use(name,getSjzt(name));
    }

    public static RabbitMQ use(String name,SysSjglSjzt sjzt) {
        RabbitMQ rabbitMQ = (RabbitMQ) cache.get(name);
        if (rabbitMQ == null) {
            rabbitMQ = new RabbitMQ(name, sjzt);
        }
        return rabbitMQ;
    }

    /**
     * 发布/订阅模式 消费者接收
     *
     * @param znjhConfig
     * @param log
     */
    @Override
    public void sjztjt(SysSjglZnjh znjhConfig, InterfaceLog log) {
        if (!queueVerification(znjhConfig.getSrml().replace(UtilConst.FXG, ""))) {
            throw new MyException("队列不存在,或队列由另一个连接独占");
        }
        exec(rabbitMQClient -> {
            Channel channel = rabbitMQClient.getChannel();
            Consumer consumer = new MQConsumer(channel, this, znjhConfig, log);
            channel.basicConsume(znjhConfig.getSrml().replace(UtilConst.FXG, ""), false, consumer);
            return null;
        });
    }

    /**
     * 验证队列是否存在
     *
     * @param queueName
     * @return
     */
    public boolean queueVerification(String queueName) {
        return (boolean) exec(rabbitMQClient -> {
            try {
                Channel channel = rabbitMQClient.getChannel();
                channel.queueDeclarePassive(queueName);
                return true;
            } catch (Exception e) {
                slog.info("{}队列不存在,或队列由另一个连接独占", queueName);
                return false;
            }
        });
    }

    /**
     * 消费
     *
     * @param znjh
     * @return
     * @throws Exception
     */
    @Override
    public List<IFile> listFiles(SysSjglZnjh znjh) throws Exception {
        String queueName = znjh.getSrml().replace(UtilConst.FXG, "");
        if (!queueVerification(queueName)) {
            throw new MyException("队列不存在,或队列由另一个连接独占");
        }
        return (List<IFile>) exec(rabbitMQClient -> {
            List<IFile> list = new ArrayList<>();
            Channel channel = rabbitMQClient.getChannel();
            long count = channel.messageCount(queueName);
            if (count > 0) {
                for (long i = 0; i < count; i++) {
                    GetResponse response = channel.basicGet(queueName, false);
                    long deliveryTag = response.getEnvelope().getDeliveryTag();
                    list.add(new RabbitMQFile(UtilConst.FXG, this,UUID.randomUUID().toString(), response));
                    //回复ack
                    channel.basicAck(deliveryTag, false);
                }
            } else {
                slog.info("rabbitmq未获取到消息");
            }
            return list;
        });
    }

    /**
     * 获取ifile中数据
     *
     * @param iFile
     * @return
     * @throws Exception
     */
    @Override
    public InputStream getInputStream(IFile iFile) throws Exception {
        if (iFile instanceof RabbitMQFile) {
            return new ByteArrayInputStream(((RabbitMQFile) iFile).getResponse().getBody());
        }
        throw new MyException("不支持非rabbitmq");
    }

    /**
     * 清空队列消息
     *
     * @param iFile
     * @return
     * @throws Exception
     */
    @Override
    public boolean delete(IFile iFile) throws Exception {
        if (iFile instanceof RabbitMQFile) {
            //消费后就不会读取到了
            return true;
        }
        throw new MyException("不支持非RabbitMQ");
    }

    /**
     * 清空队列数据
     * @param queue
     * @return
     */
    public boolean deleteQueueData(String queue){
        return (boolean) exec(rabbitMQClient -> {
            int messageCount = rabbitMQClient.getChannel().queuePurge(queue).getMessageCount();
            if (messageCount > 0) {
                return true;
            } else {
                return false;
            }
        });
    }

    /**
     * 创建队列
     *
     * @param queue
     * @return
     */
    public boolean createQueue(String queue) {
        return (boolean) exec(rabbitMQClient -> {
            try {
                Channel channel = rabbitMQClient.getChannel();
                channel.queueDeclare(queue, true, false, false, null);
                return true;
            } catch (Exception e) {
                slog.info("创建队列异常：", e);
                return false;
            }
        });
    }

    /**
     * 删除队列
     *
     * @param queue
     * @return
     * @throws Exception
     */
    public boolean deleteQueue(String queue) throws Exception {
        return (boolean) exec(rabbitMQClient -> {
            int messageCount = rabbitMQClient.getChannel().queueDelete(queue).getMessageCount();
            if (messageCount > 0) {
                return true;
            } else {
                return false;
            }
        });
    }

    /**
     * 发布消息
     * @param topic 主题
     * @param msg 消息内容
     * @return 发布状态
     * @throws Exception 异常
     */
    public boolean pub(String topic,String msg) throws Exception {
        return save(new ByteArrayInputStream(msg.getBytes(StandardCharsets.UTF_8)),
                new BdwjFile(topic,null));
    }

    /**
     * 推送消息
     *
     * @param in
     * @param iFile
     * @return
     * @throws Exception
     */
    @Override
    public boolean save(InputStream in, IFile iFile) throws Exception {
        return (boolean) exec(rabbitMQClient -> {
            try {
                Channel channel = rabbitMQClient.getChannel();
                // 为这个通道申明一个队列，如果这个队列不存在，将在服务器上创建
                channel.queueDeclare(iFile.getParent().replace(UtilConst.FXG, ""), true, false, false, null);
                byte[] data = Utils.readByteArray(in);
                channel.confirmSelect();
                //简单模式默认“”
                channel.basicPublish("", iFile.getParent().replace(UtilConst.FXG, ""), null, data);
                boolean flag = channel.waitForConfirms();
                return flag;
            } catch (Exception e) {
                slog.info("消息保存错误", e);
                return false;
            } finally {
                FileUtil.closeStream(in);
            }
        });
    }

    @Override
    public String getRootPath() {
        return this.sjzt.getDxgs();
    }

    /**
     * 获取消息长度
     *
     * @param iFile
     * @return
     * @throws Exception
     */
    @Override
    public long getSize(IFile iFile) {
        if (iFile instanceof RabbitMQFile) {
            return ((RabbitMQFile) iFile).getResponse().getBody().length;
        }
        throw new SjztBzcwjdxExecption("不支持非RabbitMQFile");
    }

    @Override
    public void close() throws IOException {
        if (!this.objectPool.isClosed()) {
            this.objectPool.close();
        }
        cache.remove(name);
        if (this == rabbitMQ) {
            rabbitMQ = null;
        }
    }
}


/**
 * mq 消费
 */
class MQConsumer extends DefaultConsumer {

    SysSjglZnjh znjhConfig;

    InterfaceLog znjhJob;

    BasicSjzt sjzt;

    public MQConsumer(Channel channel, BasicSjzt sjzt, SysSjglZnjh znjhConfig,
                      InterfaceLog znjhJob) {
        super(channel);
        this.sjzt = sjzt;
        this.znjhJob = znjhJob;
        this.znjhConfig = znjhConfig;
    }

    @Override
    public void handleDelivery(String consumerTag, Envelope envelope, AMQP.BasicProperties properties, byte[] body) throws IOException {
        long deliveryTag = envelope.getDeliveryTag();
        try {
            IFile iFile = callBack(consumerTag, envelope, properties, body);
            znjhConfig.getTp().run(() -> sjzt.getZnjh().znjh(znjhConfig,znjhJob,iFile));
            //手动确认
            super.getChannel().basicAck(deliveryTag, false);
        } catch (IOException e) {
            e.printStackTrace();
            try {
                //消息返回队列
                super.getChannel().basicNack(deliveryTag, false, true);
            } catch (IOException ioException) {
                ioException.printStackTrace();
            }
        }
    }

    private IFile callBack(String consumerTag, Envelope envelope, AMQP.BasicProperties properties, byte[] body) {
        GetResponse response = new GetResponse(envelope, properties, body,body.length);
        return new RabbitMQFile(UtilConst.FXG, sjzt, UUID.randomUUID().toString(), response);
    }

    public SysSjglZnjh getZnjhConfig() {
        return znjhConfig;
    }

    public void setZnjhConfig(SysSjglZnjh znjhConfig) {
        this.znjhConfig = znjhConfig;
    }

    public InterfaceLog getZnjhJob() {
        return znjhJob;
    }

    public void setZnjhJob(InterfaceLog znjhJob) {
        this.znjhJob = znjhJob;
    }

    public BasicSjzt getSjzt() {
        return sjzt;
    }

    public void setSjzt(BasicSjzt sjzt) {
        this.sjzt = sjzt;
    }
}
