package cn.benma666.json;

import cn.benma666.constants.UtilConstInstance;
import cn.benma666.myutils.ClassUtil;
import cn.benma666.myutils.StringUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.alibaba.fastjson.parser.Feature;
import com.alibaba.fastjson.serializer.SerializerFeature;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 自定义的JSON对象
 */
@Slf4j
public class MyJSONObject extends JSONObject implements UtilConstInstance {

    public MyJSONObject(){
        this(true);
    }
    public MyJSONObject(Map<String, Object> map){
        super(map==null? new HashMap<>():map);
        if(map!=null){
            for (Entry<String ,Object> e:map.entrySet()){
                //进行属性设置和map设置
                set(e.getKey(),e.getValue());
            }
        }
    }

    public MyJSONObject(boolean ordered){
        super(ordered);
    }

    public MyJSONObject(int initialCapacity){
        super(initialCapacity);
    }

    public MyJSONObject(int initialCapacity, boolean ordered){
        super(initialCapacity,ordered);
    }
    /**
     * 支持JSONPath方式取数据
     * @param key 键
     * @return 值
     */
    @Override
    public Object get(Object key) {
        if(StringUtil.isBlank(key)){
            return getInnerMap().get(key);
        }
        Object val = ClassUtil.getVal(key.toString(), super.get(key), this);
        if(val==null && key.toString().startsWith("$.")) {
            resetMap();
            //如此可以方便的获取多层结构下的值，如：json.getJSONObject("$.sys.key")
            val = JSONPath.eval(getInnerMap(),key.toString());
        }
        return val;
    }

    @Override
    public Object put(String key, Object value) {
        return set(key,value);
    }

    /**
     * 通过JSONPath进行进行多层值设置
     * @param key 路径
     * @param value 值
     * @return 设置结果
     */
    public boolean set(String key, Object value){
        return ClassUtil.setVal(key, value,this,getInnerMap());
    }

    @Override
    public Set<String> keySet() {
        resetMap();
        return super.keySet();
    }

    @Override
    public Set<Entry<String, Object>> entrySet() {
        resetMap();
        return super.entrySet();
    }

    @Override
    public Collection<Object> values() {
        resetMap();
        return super.values();
    }

    @Override
    public int size() {
        resetMap();
        return super.size();
    }

    /**
     * 重新设置map
     */
    private void resetMap() {
        Map<String, Object> map = getInnerMap();
        String field;
        for(Method m:this.getClass().getDeclaredMethods()){
            field = null;
            //获取所有get属性进行设置，只处理get无参的
            if(m.getName().startsWith("get")&&m.getParameterCount()==0){
                field = StringUtil.lowerCaseFast(m.getName().substring(3));
            }else if(m.getName().startsWith("is")){
                field = StringUtil.lowerCaseFast(m.getName().substring(2));
            }
            try {
                if(field!=null&&m.invoke(this)!=null){
                    map.put(field,m.invoke(this));
                }
            } catch (Exception e) {
                log.debug("获取属性失败：{}",m.getName());
            }
        }
    }

    /**
     * 解析字符串为JSON对象
     */
    public static MyJSONObject parseObject(String text, Feature... features) {
        JSONObject obj = JSONObject.parseObject(text, features);
        return new MyJSONObject(obj.getInnerMap());
    }

    @Override
    public String toJSONString() {
        //复制一个对象出来，避免影响参数对象本身
        JSONObject t = this.clone();
        return t.toJSONString();
    }

    @Override
    public String toString(SerializerFeature... features) {
        //复制一个对象出来，避免影响参数对象本身
        JSONObject t = this.clone();
        return t.toString(features);
    }

    @Override
    public String toString() {
        return toJSONString();
    }
}
