package cn.benma666.iframe;

import cn.benma666.constants.UtilConst;
import cn.benma666.dict.Zdlb;
import cn.benma666.domain.SysSjglSjzt;
import cn.benma666.exception.BusinessException;
import cn.benma666.exception.DictException;
import cn.benma666.myutils.StringUtil;
import cn.benma666.myutils.TmplUtil;
import cn.benma666.sjzt.Db;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.beetl.sql.core.SqlId;

import java.util.List;

/**
 * 字典管理 <br/>
 * date: 2016年6月12日 <br/>
 * @author jingma
 * @version 0.1
 */
public class DictManager extends BasicObject {
    /**
     * 不允许缓存
     */
    public static final String NOCACHE = "_nocache";
    /**
     * 字典树缓存
     */
    public static final String TREE = "_tree";
    /**
     * 对应的类别sql缓存
     */
    public static final String LBSQL = "_lbsql";
    /**
     * 字典缓存，指定使用内存缓存。
     */
    private static final JSONObject dictCache = CacheFactory.use("dict");

    /**
     * Creates a new instance of DictManager.
     */
    public DictManager() {
    }

    /**
     * 清理字典缓存 <br/>
     * @author jingma
     */
    public static void clearDict() {
        dictCache.clear();
    }

    /**
     * 清理字典缓存-按字典类别清理 <br/>
     * @author jingma
     */
    public static void clearDict(String zdlb) {
        dictCache.remove(zdlb);
        dictCache.remove(zdlb + NOCACHE);
        dictCache.remove(zdlb + TREE);
        dictCache.remove(zdlb + LBSQL);
    }

    /**
     * 获取字典类别对应的查询sql <br/>
     * @author jingma
     * @param myParams 参数对象
     * @return 字典类别对象
     */
    private static JSONObject zdlbToSql(MyParams myParams) {
        String zdlb = myParams.getString("$.yobj.zdlb");
        JSONObject zdlbObj = dictCache.getJSONObject(zdlb + LBSQL);
        if(zdlbObj==null){
            // 查询类别对象
            zdlbObj = Db.use().findFirst(SqlId.of("util", "findTyzdLbsql"),myParams);
            if (zdlbObj == null) {
                throw new DictException("该字典类别不存在" + myParams.getJSONObject(UtilConst.KEY_YOBJ));
            }
            String lbsql = zdlbObj.getString("lbsql");
            if(!isBlank(lbsql)&&lbsql.startsWith("ds=")){
                //为了支持不同数据库类型，这里只配置数据载体代码，如：ds=xxx，然后sql在大字典中配置
                //原数据载体代码
                String ysjztdm = lbsql.substring(3).trim();
                //数据载体代码
                String sjztdm = TmplUtil.buildStr(ysjztdm,myParams);
                //查询该数据载体
                JSONObject dbObj = DictManager.zdObjByDm(Zdlb.SYS_COMMON_SJZT.name(), sjztdm);
                if(dbObj==null){
                    throw new DictException("没找到数据载体："+sjztdm);
                }
                //设置数据库信息
                myParams.put("sjzt",dbObj.toJavaObject(SysSjglSjzt.class));
                lbsql = Db.getZdSqlTmpl(myParams,"zd."+zdlb,dbObj.getString("lx"));
                if(!lbsql.contains(";ds=")){
                    //补充数据载体代码
                    lbsql += ";ds="+sjztdm;
                }
                if(isBlank(lbsql)){
                    throw new DictException("没有在配置中设置该字典类别数据获取的sql："+zdlb);
                }
                if(sjztdm.equals(ysjztdm)){
                    //数据载体固定则缓存，通过变量在变化则不缓存
                    dictCache.put(zdlb+LBSQL,zdlbObj);
                }
            }else {
                dictCache.put(zdlb+LBSQL,zdlbObj);
            }
            //备份原始类别sql，可能为模板
            zdlbObj.put("lbsqlYs", lbsql);
        }
        if(zdlbObj.getIntValue("sjdj")<valByDef(myParams.getIntValue("$.user.yhdj"),-1)){
            //数据等级高于用户等级则允许访问
            throw new BusinessException("没有访问该字典的权限："+zdlb);
        }
        // sql处理
        String lbsql = zdlbObj.getString("lbsqlYs");
        //获取查询字典数据的默认sql
        String defaultVal = Db.use().getSourceSql(SqlId.of("util", "findTyzdMrsql"), myParams);
        if (StringUtil.isBlank(lbsql)) {
            lbsql = defaultVal ;
        } else {
            myParams.set("$.sql.defaultSql",defaultVal);
            // 进行模板渲染，支持根据前端参数和用户信息进行差异化
            lbsql = TmplUtil.buildStrSql(lbsql, myParams);
            if(lbsql.startsWith("error:")){
                //该SQL模板返回错误的场景
                throw new DictException(lbsql.substring("error:".length()));
            }
        }
        zdlbObj = zdlbObj.clone();
        zdlbObj.put("lbsql", lbsql);
        return zdlbObj;
    }

    /**
     * 获取字典类别的全部数据-不走缓存 <br/>
     * @author jingma
     * @param myParams 参数对象
     * @return null:表示不支持获取列表，JSONObject<dm,JSONObject>:全部字典项
     */
    public static JSONObject zdMapNoCache(MyParams myParams) {
        JSONObject zdlbObj = zdlbToSql(myParams);
        String zdlb = myParams.getString("$.yobj.zdlb");
        if (!zdlbObj.getBooleanValue("cache")) {
            // 不允许缓存时，标记不允许缓存
            if(!dictCache.containsKey(zdlb)){
                dictCache.put(zdlb + NOCACHE, true);
                dictCache.put(zdlb, new JSONObject());
            }
            return null;
        } else {
            String expStr = zdlbObj.getString("lbsql");
            String[] exps = Db.parseDictExp(expStr);
            myParams.set("$.sql.defaultSql",exps[1]);
            JSONObject result = new JSONObject(true);
            result.putAll(db(exps[0]).findMap("dm",SqlId.of("util", "findTyzdSjlb"), myParams));
            return result;
        }
    }

    /**
    * 获取字典类别的全部数据-走缓存 <br/>
    * @author jingma
    * @param myParams 参数对象
    * @return null:表示不支持获取列表，JSONObject<dm,JSONObject>:全部字典项
    */
    public static JSONObject zdMap(MyParams myParams) {
        zdlbToSql(myParams);
        String zdlb = myParams.getString("$.yobj.zdlb");
        if (dictCache.containsKey(zdlb + NOCACHE)) {
            return null;
        }
        JSONObject l = dictCache.getJSONObject(zdlb);
        if(l!=null){
            //有缓存直接返回
            return l;
        }
        //没有缓存需要重新加载
        //这个锁是为了避免同一个字典被重复加载，但不同字典也会被锁
        synchronized (dictCache){
            l = dictCache.getJSONObject(zdlb);
            if (l == null) {
                l = zdMapNoCache(myParams);
                if(l != null){
                    dictCache.put(zdlb, l);
                }
            }
        }
        return l;
    }

    /**
    * 获取字典类别的全部数据-不走缓存 <br/>
    * @author jingma
    * @param zdlb 字典类别
    * @return null:表示不支持获取列表，JSONObject<dm,JSONObject>:全部字典项
    */
    public static JSONObject zdMap(String zdlb) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        return zdMap(myParams);
    }

    /**
     * 获取字典树-不走缓存
     * @param myParams 参数对象
     * @return 构建好的字典树
     */
    public static JSONArray zdTreeNoCache(MyParams myParams){
        String zdlb = myParams.getString("$.yobj.zdlb");
        //构建树场景，树都允许全量加载
        JSONObject zdlbObj = zdlbToSql(myParams);
        String expStr = zdlbObj.getString("lbsql");
        String[] exps = Db.parseDictExp(expStr);
        myParams.set("$.sql.defaultSql",exps[1]);
        List<JSONObject> zdList = db(exps[0]).find(SqlId.of("util", "findTyzdSjlb"),myParams);
        JSONArray l = StringUtil.buildTree(zdList, "upnode", "dm");
        if (zdlbObj.getBooleanValue("cache")) {
            //允许缓存
            dictCache.put(zdlb+TREE, l);
        }
        return l;
    }

    /**
     * 获取字典类别的全部数据-走缓存 <br/>
     * @author jingma
     * @param myParams 参数对象
     * @return JSONObject<dm,JSONObject>:全部字典项
     */
    public static JSONArray zdTree(MyParams myParams) {
        zdlbToSql(myParams);
        String zdlb = myParams.getString("$.yobj.zdlb");
        JSONArray l = dictCache.getJSONArray(zdlb+TREE);
        if (l==null) {
            l = zdTreeNoCache(myParams);
        }
        return l;
    }

    /**
     * 获取字典类别的全部数据-不走缓存 <br/>
     * @author jingma
     * @param zdlb 字典类别
     * @return JSONObject<dm,JSONObject>:全部字典项
     */
    public static JSONArray zdTree(String zdlb) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        return zdTree(myParams);
    }

    /**
     * 获取字典类别对应代码的字典对象-不走缓存 <br/>
     * @author jingma
     * @param myParams 参数对象
     * @return JSONObject:字典项对象
     */
    public static JSONObject zdObjNoCache(MyParams myParams) {
        JSONObject zdlbObj = zdlbToSql(myParams);
        String[] exps = Db.parseDictExp(zdlbObj.getString("lbsql"));
        myParams.set("$.sql.defaultSql",exps[1]);
        return db(exps[0]).findFirst(SqlId.of("util", "findTyzdSjx"),myParams);
    }
    /**
     * 获取字典类别对应代码的字典对象-走缓存 <br/>
     * @author jingma
     * @param myParams 参数对象
     * @return JSONObject:字典项对象
     */
    public static JSONObject zdObj(MyParams myParams) {
        zdlbToSql(myParams);
        String zdlb = myParams.getString("$.yobj.zdlb");
        String dm = myParams.getString("$.yobj.dm");
        String mc = myParams.getString("$.yobj.mc");
        JSONObject result = null;
        JSONObject zc = dictCache.getJSONObject(zdlb);
        if (zc == null) {
            zdMap(myParams);
            zc = dictCache.getJSONObject(zdlb);
        }
        if(zc.getInnerMap().containsKey(dm != null ? dm : mc)){
            //key存在表示已经获取过了，不管是否为null
            result = zc.getJSONObject(dm != null ? dm : mc);
        }else if(dictCache.containsKey(zdlb+NOCACHE)){
            result = zdObjNoCache(myParams);
            zc.put(dm != null ? dm : mc, result);
        }else if(StringUtil.isNotBlank(mc)){
            //允许缓存
            //通过名称获取字典项对象
            for(JSONObject obj : zc.values().toArray(new JSONObject[]{})){
                if(mc.equals(obj.getString("mc"))){
                    result = obj;
                    break;
                }else{
                    //兼容名称
                    JSONArray list = obj.getJSONArray("$.kzxx.jrmc");
                    if(!isBlank(list)){
                        if(list.contains(mc)){
                            result = obj;
                            break;
                        }
                    }
                }
            }
        }
        return result;
    }

    /**
     * 获取字典类别对应代码的字典对象-走缓存 <br/>
     * @param zdlb 字典类别
     * @param dm 字典代码
     * @return JSONObject:字典项对象
     */
    public static JSONObject zdObjByDm(String zdlb, String dm) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        myParams.set("$.yobj.dm",dm);
        return zdObj(myParams);
    }
    /**
     * 获取字典值 <br/>
     * 会走缓存
     * @author jingma
     * @param zdlb 字典类别
     * @param dm 代码
     * @return 值
     */
    public static String zdMcByDm(String zdlb, String dm) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        myParams.set("$.yobj.dm",dm);
        JSONObject r = zdObj(myParams);
        if(r == null){
            return dm;
        }else{
            return r.getString("mc");
        }
    }

    /**
     * 获取字典值 <br/>
     * 会走缓存
     * @author jingma
     * @param zdlb 字典类别
     * @param mc 名称
     * @return 值
     */
    public static String zdDmByMc(String zdlb, String mc) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        myParams.set("$.yobj.mc",mc);
        JSONObject r = zdObj(myParams);
        if(r == null){
            return mc;
        }else{
            return r.getString("dm");
        }
    }

    /**
     * 获取字典值,其中dm用逗号隔开 <br/>
     * 会走缓存
     * @author jingma
     * @param myParams 参数对象
     * @return 值
     */
    public static String zdMcByMoreDm(MyParams myParams) {
        JSONObject r = zdObjByMoreDm(myParams);
        StringBuilder mc = new StringBuilder();
        for(String s : r.keySet()){
            JSONObject obj = r.getJSONObject(s);
            mc.append("、").append(obj==null?s:obj.getString("mc"));
        }
        if(mc.length()>0){
            return mc.substring(1);
        }else {
            return mc.toString();
        }
    }

    /**
     * 获取字典值,其中dm用逗号隔开 <br/>
     * 会走缓存
     * @author jingma
     * @param zdlb 字典类别
     * @param dm 多个代码，逗号分隔
     * @return 值，多个值、分隔
     */
    public static String zdMcByMoreDm(String zdlb, String dm) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        myParams.set("$.yobj.dm",dm);
        return zdMcByMoreDm(myParams);
    }

    /**
     * 获取字典对象,其中dm用逗号隔开 <br/>
     * 会走缓存
     * @author jingma
     * @param myParams 参数对象
     * @return 对象
     */
    public static JSONObject zdObjByMoreDm(MyParams myParams) {
        String dm = myParams.getString("$.yobj.dm");
        JSONObject result = new JSONObject(true);
        if (StringUtils.isNotBlank(dm)) {
            String[] dms = dm.split(",");
            for (String s : dms) {
                myParams.set("$.yobj.dm",s);
                JSONObject obj = zdObj(myParams);
                result.put(s,obj);
            }
        }
        return result;
    }
    /**
     * 获取字典值,其中dm用逗号隔开 <br/>
     * 会走缓存
     * @author jingma
     * @param zdlb 字典类别
     * @param dm 多个代码，逗号分隔
     * @return 值，多个值、分隔
     */
    public static JSONObject zdObjByMoreDm(String zdlb, String dm) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        myParams.set("$.yobj.dm",dm);
        return zdObjByMoreDm(myParams);
    }

    /**
     * 获取字典值,其中dm用顿号隔开 <br/>
     * 会走缓存
     * @author jingma
     * @param myParams 参数对象
     * @return 值
     */
    public static String zdDmByMoreMc(MyParams myParams) {
        String mc = myParams.getString("$.yobj.mc");
        if (StringUtils.isNotBlank(mc)) {
            String[] mcs = mc.replace(",","、").split("、");
            StringBuilder dm = new StringBuilder();
            for (String s : mcs) {
                myParams.set("$.yobj.mc",s);
                JSONObject obj = zdObj(myParams);
                dm.append(",").append(obj==null?s:obj.getString("dm"));
            }
            return dm.substring(1);
        }
        return "";
    }

    /**
     * 获取字典值,其中mc用顿号隔开 <br/>
     * 会走缓存
     * @author jingma
     * @param zdlb 字典类别
     * @param mc 名称
     * @return 代码
     */
    public static String zdDmByMoreMc(String zdlb, String mc) {
        MyParams myParams = new MyParams();
        myParams.set("$.yobj.zdlb",zdlb);
        myParams.set("$.yobj.mc",mc);
        return zdDmByMoreMc(myParams);
    }

    /**
     * 字典搜索 <br/>
     * @author jingma
     * @param myParams 参数对象
     * @return 字典搜索结果列表
     */
    public static PageInfo<JSONObject> zdSearch(MyParams myParams) {
        JSONObject zdlbObj = zdlbToSql(myParams);
        String[] exps = Db.parseDictExp(zdlbObj.getString("lbsql"));
        myParams.sql().setDefaultSql(exps[1]);
        PageInfo<JSONObject> page = myParams.page();
        page = db(exps[0]).queryPage(page,SqlId.of("util", "findTyzdSjss"),myParams);
        return page;
    }
}
