package cn.authing.guard.activity;

import android.content.Intent;
import android.graphics.Bitmap;
import android.os.Bundle;
import android.os.Handler;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.authing.guard.Authing;
import cn.authing.guard.R;
import cn.authing.guard.data.UserInfo;
import cn.authing.guard.flow.AuthFlow;
import cn.authing.guard.internal.CircularAnimatedView;
import cn.authing.guard.social.handler.Amazon;
import cn.authing.guard.social.handler.FaceBook;
import cn.authing.guard.social.handler.GitLab;
import cn.authing.guard.social.handler.Gitee;
import cn.authing.guard.social.handler.Github;
import cn.authing.guard.social.handler.Google;
import cn.authing.guard.social.handler.HuaWei;
import cn.authing.guard.social.handler.Lark;
import cn.authing.guard.social.handler.Line;
import cn.authing.guard.social.handler.Linkedin;
import cn.authing.guard.social.handler.QQ;
import cn.authing.guard.social.handler.Slack;
import cn.authing.guard.social.handler.Weibo;
import cn.authing.guard.util.Const;
import cn.authing.guard.util.DarkModeManager;
import cn.authing.guard.util.ImageUtil;
import cn.authing.guard.util.ToastUtil;
import cn.authing.guard.util.Util;
import cn.authing.guard.webauthn.WebAuthNAuthentication;

public class AuthActivity extends BaseAuthActivity {

    public static final int RC_LOGIN = 1024;
    public static final int OK = 42;
    public static final int BIOMETRIC_BIND_OK = 43;

    public static final String AUTH_FLOW = "auth_flow";
    public static final String CONTENT_LAYOUT_ID = "content_layout_id";

    public static final String EVENT_VERIFY_CODE_ENTERED = "verify_code_entered";
    public static final String EVENT_BIND_FACE_CARE_PERMISSION = "bind_face_carme_permission";

    protected AuthFlow flow;

    private final Map<String, List<EventListener>> eventMap = new HashMap<>();
    private FrameLayout loadingContainer;
    private View loading;
    private Bitmap qrCodeBitmap;
    private WebAuthNAuthentication webAuthNAuthentication;

    public interface EventListener {
        void happened(String what);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Intent intent = getIntent();
        flow = (AuthFlow) intent.getSerializableExtra(AUTH_FLOW);
        if (flow != null) {
            int layoutId = intent.getIntExtra(CONTENT_LAYOUT_ID, 0);
            if (layoutId == 0) {
                layoutId = flow.getIndexLayoutId();
            }
            if (layoutId != 0) {
                setContentView(layoutId);
            }
        }
        DarkModeManager.getInstance().setDarkMode(this);

        if (Authing.isGettingConfig()) {
            FrameLayout rootLayout = findViewById(android.R.id.content);
            loadingContainer = new FrameLayout(this);
            loadingContainer.setBackgroundColor(0xffffffff);
            loading = new CircularAnimatedView(this);
            int size = (int)Util.dp2px(this, 88);
            FrameLayout.LayoutParams lp = new FrameLayout.LayoutParams(size, size);
            lp.gravity = Gravity.CENTER;
            loading.setLayoutParams(lp);
            loading.setVisibility(View.INVISIBLE);
            loadingContainer.addView(loading);
            rootLayout.addView(loadingContainer);

            // some tolerance for blank screen. otherwise the loading will blink
            new Handler().postDelayed(this::showLoading, 500);
        } else if (Authing.getAppId() == null) {
            ToastUtil.showCenter(this, getString(R.string.authing_uninitialized));
        }
        Amazon.getInstance().onCreate(this);
    }

    @Override
    protected void onResume() {
        super.onResume();
        Lark.getInstance().onResume(this);
        Amazon.getInstance().onResume();
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        Lark.getInstance().onNewIntent(this, intent);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (webAuthNAuthentication != null){
            webAuthNAuthentication.onActivityResult(requestCode, resultCode, data);
        }
        if (requestCode == RC_LOGIN && resultCode == OK && data != null) {
            UserInfo userInfo = (UserInfo)data.getSerializableExtra("user");
            Intent intent = new Intent();
            intent.putExtra("user", userInfo);
            setResult(OK, intent);
            finish();
        }
        Lark.getInstance().onActivityResult(this, data);
        Google.getInstance().onActivityResult(requestCode, resultCode, data);
        FaceBook.getInstance().onActivityResult(requestCode, resultCode, data);
        QQ.getInstance().onActivityResult(requestCode, resultCode, data);
        Weibo.getInstance().onActivityResult(this, requestCode, resultCode, data);
        Linkedin.getInstance().onActivityResult(requestCode, resultCode, data);
        Github.getInstance().onActivityResult(requestCode, resultCode, data);
        Gitee.getInstance().onActivityResult(requestCode, resultCode, data);
        GitLab.getInstance().onActivityResult(requestCode, resultCode, data);
        Line.getInstance().onActivityResult(requestCode, resultCode, data);
        Slack.getInstance().onActivityResult(requestCode, resultCode, data);
        HuaWei.getInstance().onActivityResult(requestCode, resultCode, data);
        if (resultCode == BIOMETRIC_BIND_OK){
            Util.biometricBind(this);
        }
    }

    private void showLoading() {
        if (!Authing.isGettingConfig()) {
            loadingContainer.setVisibility(View.GONE);
            return;
        }

        // still requesting config. show loading for some minimum time to avoid blinking
        loading.setVisibility(View.VISIBLE);
        new Handler().postDelayed(()-> Authing.getPublicConfig((config)-> {
            if (config == null) {
                FrameLayout rootLayout = findViewById(android.R.id.content);
                loading.setVisibility(View.GONE);
                TextView tv = new TextView(this);
                tv.setText(R.string.authing_no_network);
                FrameLayout.LayoutParams tvlp = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT);
                tvlp.gravity = Gravity.CENTER;
                tv.setLayoutParams(tvlp);
                rootLayout.addView(tv);
            } else {
                loadingContainer.setVisibility(View.GONE);
            }
        }), 888);
    }

    public AuthFlow getFlow() {
        return flow;
    }

    public void setFlow(AuthFlow flow) {
        this.flow = flow;
    }

    public void subscribe(String channel, EventListener listener) {
        if (!TextUtils.isEmpty(channel) && listener != null) {
            List<EventListener> fans = eventMap.computeIfAbsent(channel, k -> new ArrayList<>());
            fans.add(listener);
        }
    }

    public void fire(String channel, String what) {
        if (TextUtils.isEmpty(channel)) {
            return;
        }

        List<EventListener> fans = eventMap.get(channel);
        if (fans != null) {
            for (EventListener listener : fans) {
                listener.happened(what);
            }
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == Const.REQUEST_CODE_QR && qrCodeBitmap != null){
            ImageUtil.saveImage(this, qrCodeBitmap);
        }
        if (requestCode == Const.REQUEST_MFA_BINDING){
            fire(EVENT_BIND_FACE_CARE_PERMISSION, "");
        }
    }

    public void setQrCodeBitmap(Bitmap qrCodeBitmap) {
        this.qrCodeBitmap = qrCodeBitmap;
    }

    public void setWebAuthNAuthentication(WebAuthNAuthentication webAuthNAuthentication) {
        this.webAuthNAuthentication = webAuthNAuthentication;
    }

}
