package cn.arnohand;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.qiniu.common.QiniuException;
import com.qiniu.http.Response;
import com.qiniu.storage.BucketManager;
import com.qiniu.storage.UpCompletionHandler;
import com.qiniu.storage.UploadManager;
import com.qiniu.storage.model.FileInfo;
import com.qiniu.util.Auth;
import com.qiniu.util.Md5;
import com.qiniu.util.StringMap;

import java.io.File;
import java.io.FileNotFoundException;

/**
 * 七牛工具类
 */
public class QiNiuUtil {

    /**
     * 上传文件
     *
     * @param localFilePath 本地文件路径
     * @return {@link MyPutRet}
     * @throws Exception Exception
     */
    public static MyPutRet upload(String localFilePath, QiNiuConfigModule module) throws Exception {
        return upload(localFilePath, null, null, null, module);
    }

    /**
     * 上传文件
     *
     * @param localFilePath 本地文件路径
     * @param key           上传文件保存的文件名
     * @return {@link MyPutRet}
     * @throws Exception Exception
     */
    public static MyPutRet upload(String localFilePath, String key, QiNiuConfigModule module) throws Exception {
        return upload(localFilePath, key, null, null, module);
    }

    /**
     * 上传文件
     *
     * @param localFilePath 本地文件路径
     * @param key           上传文件保存的文件名,不传或为null则默认为文件md5码
     * @param module        七牛配置
     * @param params        自定义参数
     * @param fileType      文件类型
     * @return {@link MyPutRet}
     * @throws Exception Exception
     */
    public static MyPutRet upload(String localFilePath, String key, StringMap params, String fileType, QiNiuConfigModule module) throws Exception {
        File file = FileUtil.file(localFilePath);
        if (!file.exists()) {
            throw new FileNotFoundException("file not found: " + localFilePath);
        }
        if (StrUtil.isEmpty(key)) {
            key = Md5.md5(file);
        }
        String upToken = module.getUpToken();
        UploadManager uploadManager = module.getUploadManager();
        //默认不指定key的情况下，以文件内容的hash值作为文件名
        Response response = uploadManager.put(file, key, upToken, params, fileType, false);
        //解析上传成功的结果
        return JSON.parseObject(response.bodyString(), MyPutRet.class);
    }

    /**
     * 异步上传文件
     *
     * @param localFilePath 本地文件路径
     * @param handler       上传完成回调
     * @param module        七牛配置
     * @throws Exception Exception
     */
    public static void asyncUpload(String localFilePath, UpCompletionHandler handler, QiNiuConfigModule module) throws Exception {
        asyncUpload(localFilePath, null, handler, module);
    }

    /**
     * 异步上传文件
     *
     * @param localFilePath 本地文件路径
     * @param key           上传文件保存的文件名,不传或为null则默认为文件md5码
     * @param handler       上传完成回调
     * @param module        七牛配置
     * @throws Exception Exception
     */
    public static void asyncUpload(String localFilePath, String key, UpCompletionHandler handler, QiNiuConfigModule module) throws Exception {
        asyncUpload(localFilePath, key, null, null, handler, module);
    }

    /**
     * 异步上传文件
     *
     * @param localFilePath 本地文件路径
     * @param key           上传文件保存的文件名,不传或为null则默认为文件md5码
     * @param params        自定义参数
     * @param fileType      文件类型
     * @param handler       上传完成回调
     * @param module        七牛配置
     * @throws Exception Exception
     */
    public static void asyncUpload(String localFilePath, String key, StringMap params, String fileType, UpCompletionHandler handler, QiNiuConfigModule module) throws Exception {
        File file = FileUtil.file(localFilePath);
        if (!file.exists()) {
            throw new FileNotFoundException("file not found: " + localFilePath);
        }
        if (StrUtil.isEmpty(key)) {
            key = Md5.md5(file);
        }
        UploadManager uploadManager = module.getUploadManager();
        byte[] bytes = IoUtil.readBytes(FileUtil.getInputStream(file));
        uploadManager.asyncPut(bytes, key, module.getUpToken(), params, fileType, false, handler);
    }

    /**
     * 文件信息
     *
     * @param key    文件key
     * @param module 七牛配置
     * @return 文件信息
     */
    public static FileInfo state(String key, QiNiuConfigModule module) {
        try {
            BucketManager bucketManager = module.getBucketManager();
            return bucketManager.stat(module.getBucket(), key);
        } catch (QiniuException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取私有空间文件下载地址
     *
     * @param url    文件地址
     * @param module 七牛配置
     * @return 下载地址
     */
    public static String getPrivateDownloadUrl(String url, QiNiuConfigModule module) {
        Auth auth = module.getAuth();
        return auth.privateDownloadUrl(url);
    }

    /**
     * 删除文件
     *
     * @param key    文件key
     * @param module 七牛配置
     * @return 是否已删除
     */
    public static boolean delete(String key, QiNiuConfigModule module) throws Exception {
        BucketManager bucketManager = module.getBucketManager();
        Response delete = bucketManager.delete(module.getBucket(), key);
        return delete.statusCode == 200;
    }

}
