/*
 * Decompiled with CFR 0.152.
 */
package cn.aradin.spring.redis.starter.cache;

import cn.aradin.spring.redis.starter.properties.RedisCacheConfiguration;
import cn.aradin.spring.redis.starter.properties.RedisCacheManagerProperties;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import org.springframework.dao.PessimisticLockingFailureException;
import org.springframework.data.redis.cache.BatchStrategy;
import org.springframework.data.redis.cache.CacheStatistics;
import org.springframework.data.redis.cache.CacheStatisticsCollector;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisStringCommands;
import org.springframework.data.redis.core.types.Expiration;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;

public class AradinRedisCacheWriter
implements RedisCacheWriter {
    private final RedisConnectionFactory connectionFactory;
    private final Duration sleepTime;
    private final CacheStatisticsCollector statistics;
    private final BatchStrategy batchStrategy;
    private final RedisCacheManagerProperties redisCacheManagerProperties;

    public AradinRedisCacheWriter(RedisConnectionFactory connectionFactory, BatchStrategy batchStrategy, RedisCacheManagerProperties redisCacheManagerProperties) {
        this(connectionFactory, Duration.ZERO, batchStrategy, redisCacheManagerProperties);
    }

    AradinRedisCacheWriter(RedisConnectionFactory connectionFactory, Duration sleepTime, BatchStrategy batchStrategy, RedisCacheManagerProperties redisCacheManagerProperties) {
        this(connectionFactory, sleepTime, CacheStatisticsCollector.none(), batchStrategy, redisCacheManagerProperties);
    }

    AradinRedisCacheWriter(RedisConnectionFactory connectionFactory, Duration sleepTime, CacheStatisticsCollector cacheStatisticsCollector, BatchStrategy batchStrategy, RedisCacheManagerProperties redisCacheManagerProperties) {
        Assert.notNull((Object)connectionFactory, (String)"ConnectionFactory must not be null!");
        Assert.notNull((Object)sleepTime, (String)"SleepTime must not be null!");
        Assert.notNull((Object)cacheStatisticsCollector, (String)"CacheStatisticsCollector must not be null!");
        Assert.notNull((Object)batchStrategy, (String)"BatchStrategy must not be null!");
        this.connectionFactory = connectionFactory;
        this.sleepTime = sleepTime;
        this.statistics = cacheStatisticsCollector;
        this.batchStrategy = batchStrategy;
        this.redisCacheManagerProperties = redisCacheManagerProperties;
    }

    private Duration getTtlOffset(String name) {
        RedisCacheConfiguration cacheConfiguration = this.redisCacheManagerProperties.getConfigs().get(name);
        if (cacheConfiguration == null) {
            return this.redisCacheManagerProperties.getDefaults().getTtlOffset();
        }
        return cacheConfiguration.getTtlOffset();
    }

    private Expiration computeExpiration(String name, Duration ttl) {
        long expiration = ttl.toMillis();
        Duration ttlOffset = this.getTtlOffset(name);
        if (AradinRedisCacheWriter.shouldExpireWithin(ttlOffset)) {
            expiration += ThreadLocalRandom.current().nextLong(ttlOffset.toMillis());
        }
        return Expiration.from((long)expiration, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    public void put(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null!");
        Assert.notNull((Object)key, (String)"Key must not be null!");
        Assert.notNull((Object)value, (String)"Value must not be null!");
        this.execute(name, connection -> {
            if (AradinRedisCacheWriter.shouldExpireWithin(ttl)) {
                connection.stringCommands().set(key, value, this.computeExpiration(name, ttl), RedisStringCommands.SetOption.upsert());
            } else {
                connection.stringCommands().set(key, value);
            }
            return "OK";
        });
        this.statistics.incPuts(name);
    }

    public byte[] get(String name, byte[] key) {
        Assert.notNull((Object)name, (String)"Name must not be null!");
        Assert.notNull((Object)key, (String)"Key must not be null!");
        byte[] result = this.execute(name, connection -> connection.stringCommands().get(key));
        this.statistics.incGets(name);
        if (result != null) {
            this.statistics.incHits(name);
        } else {
            this.statistics.incMisses(name);
        }
        return result;
    }

    public byte[] putIfAbsent(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null!");
        Assert.notNull((Object)key, (String)"Key must not be null!");
        Assert.notNull((Object)value, (String)"Value must not be null!");
        return this.execute(name, connection -> {
            if (this.isLockingCacheWriter()) {
                this.doLock(name, (RedisConnection)connection);
            }
            try {
                boolean put = AradinRedisCacheWriter.shouldExpireWithin(ttl) ? connection.stringCommands().set(key, value, this.computeExpiration(name, ttl), RedisStringCommands.SetOption.ifAbsent()).booleanValue() : connection.stringCommands().setNX(key, value).booleanValue();
                if (put) {
                    this.statistics.incPuts(name);
                    byte[] byArray = null;
                    return byArray;
                }
                byte[] byArray = connection.stringCommands().get(key);
                return byArray;
            }
            finally {
                if (this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
        });
    }

    public void remove(String name, byte[] key) {
        Assert.notNull((Object)name, (String)"Name must not be null!");
        Assert.notNull((Object)key, (String)"Key must not be null!");
        this.execute(name, connection -> connection.keyCommands().del((byte[][])new byte[][]{key}));
        this.statistics.incDeletes(name);
    }

    public void clean(String name, byte[] pattern) {
        Assert.notNull((Object)name, (String)"Name must not be null!");
        Assert.notNull((Object)pattern, (String)"Pattern must not be null!");
        this.execute(name, connection -> {
            boolean wasLocked = false;
            try {
                long deleteCount;
                if (this.isLockingCacheWriter()) {
                    this.doLock(name, (RedisConnection)connection);
                    wasLocked = true;
                }
                for (deleteCount = this.batchStrategy.cleanCache(connection, name, pattern); deleteCount > Integer.MAX_VALUE; deleteCount -= Integer.MAX_VALUE) {
                    this.statistics.incDeletesBy(name, Integer.MAX_VALUE);
                }
                this.statistics.incDeletesBy(name, (int)deleteCount);
            }
            finally {
                if (wasLocked && this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
            return "OK";
        });
    }

    public CacheStatistics getCacheStatistics(String cacheName) {
        return this.statistics.getCacheStatistics(cacheName);
    }

    public void clearStatistics(String name) {
        this.statistics.reset(name);
    }

    public RedisCacheWriter withStatisticsCollector(CacheStatisticsCollector cacheStatisticsCollector) {
        return new AradinRedisCacheWriter(this.connectionFactory, this.sleepTime, cacheStatisticsCollector, this.batchStrategy, this.redisCacheManagerProperties);
    }

    void lock(String name) {
        this.execute(name, connection -> this.doLock(name, (RedisConnection)connection));
    }

    void unlock(String name) {
        this.executeLockFree(connection -> this.doUnlock(name, (RedisConnection)connection));
    }

    private Boolean doLock(String name, RedisConnection connection) {
        return connection.stringCommands().setNX(AradinRedisCacheWriter.createCacheLockKey(name), new byte[0]);
    }

    private Long doUnlock(String name, RedisConnection connection) {
        return connection.keyCommands().del((byte[][])new byte[][]{AradinRedisCacheWriter.createCacheLockKey(name)});
    }

    boolean doCheckLock(String name, RedisConnection connection) {
        return connection.keyCommands().exists(AradinRedisCacheWriter.createCacheLockKey(name));
    }

    private boolean isLockingCacheWriter() {
        return !this.sleepTime.isZero() && !this.sleepTime.isNegative();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T execute(String name, Function<RedisConnection, T> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            this.checkAndPotentiallyWaitUntilUnlocked(name, connection);
            T t = callback.apply(connection);
            return t;
        }
    }

    private void executeLockFree(Consumer<RedisConnection> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            callback.accept(connection);
        }
    }

    private void checkAndPotentiallyWaitUntilUnlocked(String name, RedisConnection connection) {
        if (!this.isLockingCacheWriter()) {
            return;
        }
        long lockWaitTimeNs = System.nanoTime();
        try {
            while (this.doCheckLock(name, connection)) {
                Thread.sleep(this.sleepTime.toMillis());
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new PessimisticLockingFailureException(String.format("Interrupted while waiting to unlock cache %s", name), (Throwable)ex);
        }
        finally {
            this.statistics.incLockTime(name, System.nanoTime() - lockWaitTimeNs);
        }
    }

    private static boolean shouldExpireWithin(@Nullable Duration ttl) {
        return ttl != null && !ttl.isZero() && !ttl.isNegative();
    }

    private static byte[] createCacheLockKey(String name) {
        return (name + "~lock").getBytes(StandardCharsets.UTF_8);
    }
}

