package cn.aotcloud.gmcrypto.jni;

import cn.aotcloud.gmcrypto.GmCryptoNativeUtil;
import cn.aotcloud.gmcrypto.exception.GmCryptoException;
import cn.aotcloud.gmcrypto.util.StringUtils;

public class Sm4Utils {
	
	protected static final GmCryptoNativeUtil gmCryptoNativeUtil = new GmCryptoNativeUtil();

	/**
	 * ECB模式静态工具子类
	 * @author bgu
	 */
	public static class ECB {
		
		/**
		 * SM4_ECB PKCS7Padding模式字符串加密
		 * @param plainText	明文字符串
		 * @param keyHex	密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String encryptFromText(String plainText, String keyHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainText)) {
				throw new GmCryptoException("[SM4_ECB:encryptFromText]invalid plainText");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:encryptFromText]invalid keyHex");
			}
			return gmCryptoNativeUtil.sm4EcbEncryptFromText(plainText, keyHex);
		}
		
		/**
		 * SM4_ECB PKCS7Padding模式十六进制加密
		 * @param plainHex	明文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String encryptFromHex(String plainHex, String keyHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainHex)) {
				throw new GmCryptoException("[SM4_ECB:encryptFromHex]invalid plainHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:encryptFromHex]invalid keyHex");
			}
			return gmCryptoNativeUtil.sm4EcbEncryptFromHex(plainHex, keyHex);
		}
		
		/**
		 * SM4_ECB PKCS7Padding模式字节码加密
		 * @param plainBytes明文字节码
		 * @param keyBytes	密钥字节码
		 * @return 密文字节码
		 * @throws GmCryptoException 
		 */
		public static byte[] encryptFromData(byte[] plainBytes, byte[] keyBytes) throws GmCryptoException {
			if(plainBytes==null || plainBytes.length == 0) {
				throw new GmCryptoException("[SM4_ECB:encryptFromData]invalid plainBytes");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_ECB:encryptFromData]invalid keyBytes");
			}
			return gmCryptoNativeUtil.sm4EcbEncryptFromData(plainBytes, keyBytes);
		}
		
		/**
		 * SM4_ECB NoPadding模式字符串加密
		 * @param plainText	明文字符串
		 * @param keyHex	密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String noPaddingEncryptFromText(String plainText, String keyHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainText)) {
				throw new GmCryptoException("[SM4_ECB:encryptFromText]invalid plainText");
			}
			if(StringUtils.endsWith(plainText, "\\s")) {
				throw new GmCryptoException("[SM4_ECB:encryptFromText]In NoPadding mode, the source Does not support ending with space or tab");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:encryptFromText]invalid keyHex");
			}
			return gmCryptoNativeUtil.sm4EcbNoPaddingEncryptFromText(plainText, keyHex);
		}
		
		/**
		 * SM4_ECB NoPadding模式十六进制加密
		 * @param plainHex	明文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String noPaddingEncryptFromHex(String plainHex, String keyHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainHex)) {
				throw new GmCryptoException("[SM4_ECB:encryptFromHex]invalid plainHex");
			}
			if(StringUtils.endsWith(plainHex, "20")) {
				throw new GmCryptoException("[SM4_ECB:encryptFromHex]In NoPadding mode, the source Does not support ending with space or tab");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:encryptFromHex]invalid keyHex");
			}
			return gmCryptoNativeUtil.sm4EcbNoPaddingEncryptFromHex(plainHex, keyHex);
		}
		
		/**
		 * SM4_ECB NoPadding模式字节码加密
		 * @param plainBytes明文字节码
		 * @param keyBytes	密钥字节码
		 * @return 密文字节码
		 * @throws GmCryptoException 
		 */
		public static byte[] noPaddingEncryptFromData(byte[] plainBytes, byte[] keyBytes) throws GmCryptoException {
			if(plainBytes==null || plainBytes.length == 0) {
				throw new GmCryptoException("[SM4_ECB:encryptFromData]invalid plainBytes");
			}
			if(plainBytes[plainBytes.length - 1] == 0x20) {
				throw new GmCryptoException("[SM4_ECB:encryptFromData]In NoPadding mode, the source Does not support ending with space or tab");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_ECB:encryptFromData]invalid keyBytes");
			}
			return gmCryptoNativeUtil.sm4EcbNoPaddingEncryptFromData(plainBytes, keyBytes);
		}
		
		/**
		 * SM4_ECB PKCS7Padding模式字符串解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static String decryptToText(String cipherHex, String keyHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_ECB:decryptToText]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:decryptToText]invalid keyHex");
			}
			return gmCryptoNativeUtil.sm4EcbDecryptToText(cipherHex, keyHex);
		}
		
		/**
		 * SM4_ECB PKCS7Padding模式十六进制解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 * @throws GmCryptoException 
		 */
		public static String decryptToHex(String cipherHex, String keyHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_ECB:decryptToHex]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:decryptToHex]invalid keyHex");
			}
			String plantHex = gmCryptoNativeUtil.sm4EcbDecryptToHex(cipherHex, keyHex);
			plantHex = removeHexZeroPadding(plantHex);
			return plantHex;
		}

		/**
		 * SM4_ECB PKCS7Padding模式字节码解密
		 * @param cipherBytes	密文字节码
		 * @param keyBytes		密钥字节码
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static byte[] decryptToData(byte[] cipherBytes, byte[] keyBytes) throws GmCryptoException {
			if(cipherBytes==null || cipherBytes.length == 0 || cipherBytes.length%16 != 0) {
				throw new GmCryptoException("[SM4_ECB:decryptToData]invalid cipherBytes");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_ECB:decryptToData]invalid keyBytes");
			}
			byte[] plantData = gmCryptoNativeUtil.sm4EcbDecryptToData(cipherBytes, keyBytes);
			plantData = removeDataZeroPadding(plantData);
			return plantData;
		}
		
		/**
		 * SM4_ECB NoPadding模式字符串解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static String noPaddingDecryptToText(String cipherHex, String keyHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_ECB:decryptToText]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:decryptToText]invalid keyHex");
			}
			return gmCryptoNativeUtil.sm4EcbNoPaddingDecryptToText(cipherHex, keyHex);
		}
		
		/**
		 * SM4_ECB NoPadding模式十六进制解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static String noPaddingDecryptToHex(String cipherHex, String keyHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_ECB:decryptToHex]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_ECB:decryptToHex]invalid keyHex");
			}
			String plantHex = gmCryptoNativeUtil.sm4EcbNoPaddingDecryptToHex(cipherHex, keyHex);
			plantHex = removeHexZeroPadding(plantHex);
			return plantHex;
		}

		/**
		 * SM4_ECB NoPadding模式字节码解密
		 * @param cipherBytes	密文字节码
		 * @param keyBytes		密钥字节码
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static byte[] noPaddingDecryptToData(byte[] cipherBytes, byte[] keyBytes) throws GmCryptoException {
			if(cipherBytes==null || cipherBytes.length == 0 || cipherBytes.length%16 != 0) {
				throw new GmCryptoException("[SM4_ECB:decryptToData]invalid cipherBytes");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_ECB:decryptToData]invalid keyBytes");
			}
			byte[] plantData = gmCryptoNativeUtil.sm4EcbNoPaddingDecryptToData(cipherBytes, keyBytes);
			plantData = removeDataZeroPadding(plantData);
			return plantData;
		}
	}

	/**
	 * CBC模式静态工具子类
	 * @author bgu
	 */
	public static class CBC {
		
		/**
		 * SM4_CBC PKCS7Padding模式字符串加密
		 * @param plainText	明文字符串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String encryptFromText(String plainText, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainText)) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]invalid plainText");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]invalid ivHex");
			}
			return gmCryptoNativeUtil.sm4CbcEncryptFromText(plainText, keyHex, ivHex);
		}
		
		/**
		 * SM4_CBC PKCS7Padding模式十六进制加密
		 * @param plainHex	明文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String encryptFromHex(String plainHex, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainHex)) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]invalid plainHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]invalid ivHex");
			}
			return gmCryptoNativeUtil.sm4CbcEncryptFromHex(plainHex, keyHex, ivHex);
		}
		
		/**
		 * SM4_CBC PKCS7Padding模式字节码加密
		 * @param plainBytes明文字节码
		 * @param keyBytes	密钥字节码
		 * @param ivBytes	盐字节码
		 * @return 密文字节码
		 * @throws GmCryptoException 
		 */
		public static byte[] encryptFromData(byte[] plainBytes, byte[] keyBytes, byte[] ivBytes) throws GmCryptoException {
			if(plainBytes==null || plainBytes.length == 0) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]invalid plainBytes");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]invalid keyBytes");
			}
			if(ivBytes==null || ivBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]invalid ivBytes");
			}
			return gmCryptoNativeUtil.sm4CbcEncryptFromData(plainBytes, keyBytes, ivBytes);
		}

		/**
		 * SM4_CBC NoPadding模式字符串加密
		 * @param plainText	明文字符串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 * @throws GmCryptoException 
		 */
		public static String noPaddingEncryptFromText(String plainText, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainText)) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]invalid plainText");
			}
			if(StringUtils.endsWith(plainText, "\\s")) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]In NoPadding mode, the source Does not support ending with space or tab");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromText]invalid ivHex");
			}
			return gmCryptoNativeUtil.sm4CbcNoPaddingEncryptFromText(plainText, keyHex, ivHex);
		}
		
		/**
		 * SM4_CBC NoPadding模式十六进制加密
		 * @param plainHex	明文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 密文16进制串
		 * @throws GmCryptoException 
		 */
		public static String noPaddingEncryptFromHex(String plainHex, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.isEmpty(plainHex)) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]invalid plainHex");
			}
			if(StringUtils.endsWith(plainHex, "20")) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]In NoPadding mode, the source Does not support ending with space or tab");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:encryptFromHex]invalid ivHex");
			}
			return gmCryptoNativeUtil.sm4CbcNoPaddingEncryptFromHex(plainHex, keyHex, ivHex);
		}
		
		/**
		 * SM4_CBC NoPadding模式字节码加密
		 * @param plainBytes明文字节码
		 * @param keyBytes	密钥字节码
		 * @param ivBytes	盐字节码
		 * @return 密文字节码
		 * @throws GmCryptoException 
		 */
		public static byte[] noPaddingEncryptFromData(byte[] plainBytes, byte[] keyBytes, byte[] ivBytes) throws GmCryptoException {
			if(plainBytes==null || plainBytes.length == 0) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]invalid plainBytes");
			}
			if(plainBytes[plainBytes.length - 1] == 0x20) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]In NoPadding mode, the source Does not support ending with space or tab");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]invalid keyBytes");
			}
			if(ivBytes==null || ivBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:encryptFromData]invalid ivBytes");
			}
			return gmCryptoNativeUtil.sm4CbcNoPaddingEncryptFromData(plainBytes, keyBytes, ivBytes);
		}
		
		/**
		 * SM4_CBC PKCS7Padding模式字符串解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static String decryptToText(String cipherHex, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_CBC:decryptToText]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToText]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToText]invalid ivHex");
			}
			return gmCryptoNativeUtil.sm4CbcDecryptToText(cipherHex, keyHex, ivHex);
		}
		
		/**
		 * SM4_CBC PKCS7Padding模式十六进制解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 明文十六进制串
		 * @throws GmCryptoException 
		 */
		public static String decryptToHex(String cipherHex, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_CBC:decryptToHex]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToHex]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToHex]invalid ivHex");
			}
			String plantHex = gmCryptoNativeUtil.sm4CbcDecryptToHex(cipherHex, keyHex, ivHex);
			plantHex = removeHexZeroPadding(plantHex);
			return plantHex;
		}
		
		/**
		 * SM4_CBC PKCS7Padding模式字节码解密
		 * @param cipherBytes	密文字节码
		 * @param keyBytes		密钥字节码
		 * @param ivBytes		盐字节码
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static byte[] decryptToData(byte[] cipherBytes, byte[] keyBytes, byte[] ivBytes) throws GmCryptoException {
			if(cipherBytes==null || cipherBytes.length == 0 || cipherBytes.length%16 != 0) {
				throw new GmCryptoException("[SM4_CBC:decryptToData]invalid plainBytes");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:decryptToData]invalid keyBytes");
			}
			if(ivBytes==null || ivBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:decryptToData]invalid ivBytes");
			}
			byte[] plantData = gmCryptoNativeUtil.sm4CbcDecryptToData(cipherBytes, keyBytes, ivBytes);
			plantData = removeDataZeroPadding(plantData);
			return plantData;
		}
		
		/**
		 * SM4_CBC NoPadding模式字符串解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static String noPaddingDecryptToText(String cipherHex, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_CBC:decryptToText]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToText]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToText]invalid ivHex");
			}
			return gmCryptoNativeUtil.sm4CbcNoPaddingDecryptToText(cipherHex, keyHex, ivHex);
		}
		
		/**
		 * SM4_CBC NoPadding模式十六进制解密
		 * @param cipherHex	密文十六进制串
		 * @param keyHex	密钥十六进制串
		 * @param ivHex		密钥十六进制串
		 * @return 明文十六进制串
		 * @throws GmCryptoException 
		 * @throws GmCryptoException 
		 */
		public static String noPaddingDecryptToHex(String cipherHex, String keyHex, String ivHex) throws GmCryptoException {
			if(StringUtils.length(keyHex)==0 || cipherHex.length() % 32 != 0) {
				throw new GmCryptoException("[SM4_CBC:decryptToHex]invalid cipherHex");
			}
			if(StringUtils.length(keyHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToHex]invalid keyHex");
			}
			if(StringUtils.length(ivHex) != 32) {
				throw new GmCryptoException("[SM4_CBC:decryptToHex]invalid ivHex");
			}
			String plantHex = gmCryptoNativeUtil.sm4CbcNoPaddingDecryptToHex(cipherHex, keyHex, ivHex);
			plantHex = removeHexZeroPadding(plantHex);
			return plantHex;
		}
		
		/**
		 * SM4_CBC NoPadding模式字节码解密
		 * @param cipherBytes	密文字节码
		 * @param keyBytes		密钥字节码
		 * @param ivBytes		盐字节码
		 * @return 明文字符串
		 * @throws GmCryptoException 
		 */
		public static byte[] noPaddingDecryptToData(byte[] cipherBytes, byte[] keyBytes, byte[] ivBytes) throws GmCryptoException {
			if(cipherBytes==null || cipherBytes.length == 0 || cipherBytes.length%16 != 0) {
				throw new GmCryptoException("[SM4_CBC:decryptToData]invalid plainBytes");
			}
			if(keyBytes==null || keyBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:decryptToData]invalid keyBytes");
			}
			if(ivBytes==null || ivBytes.length != 16) {
				throw new GmCryptoException("[SM4_CBC:decryptToData]invalid ivBytes");
			}
			byte[] plantData = gmCryptoNativeUtil.sm4CbcNoPaddingDecryptToData(cipherBytes, keyBytes, ivBytes);
			plantData = removeDataZeroPadding(plantData);
			return plantData;
		}
	}
	
	protected final static byte[] removeDataZeroPadding(byte[] data) {
        if (data == null || data.length == 0) {
            return data;
        }
        int lastNonZeroIndex = data.length - 1;
        while (lastNonZeroIndex >= 0 && data[lastNonZeroIndex] == 0) {
            lastNonZeroIndex--;
        }
        if (lastNonZeroIndex < 0) {
            return new byte[0];
        }
        byte[] originalData = new byte[lastNonZeroIndex + 1];
        System.arraycopy(data, 0, originalData, 0, originalData.length);
        return originalData;
    }
	
	public static String removeHexZeroPadding(String hex) {
        if (hex == null || hex.isEmpty()) {
            return hex;
        }
        
        while (hex.length() >= 2 && hex.endsWith("00")) {
        	hex = hex.substring(0, hex.length() - 2);
        }
        
        return hex;
    }
	
}
