package cn.aotcloud.gmcrypto.jni;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.aotcloud.gmcrypto.CipherMode;
import cn.aotcloud.gmcrypto.GmCryptoNativeUtil;
import cn.aotcloud.gmcrypto.exception.GmCryptoException;
import cn.aotcloud.gmcrypto.util.StringUtils;

public class Sm2Utils {

	protected final GmCryptoNativeUtil gmCryptoNativeUtil = new GmCryptoNativeUtil();
	
	private CipherMode cipherMode = CipherMode.C1C2C3;

	public Sm2Utils() {
	}

	/**
	 * @param cipherMode
	 *            密文格式:C1C2C3,C1C3C2
	 */
	public Sm2Utils(CipherMode cipherMode) {
		this.cipherMode = cipherMode;
	}

	/**
	 * 产生SM2公私钥对
	 * 
	 * @return
	 */
	public String[] generateKeyPair() {
		return gmCryptoNativeUtil.generateKeyPair();
	}

	public Map<String, String> generateKeyPairMap() {
		String[] keyPair = gmCryptoNativeUtil.generateKeyPair();
		Map<String, String> keyPairMap = new HashMap<String, String>();
		keyPairMap.put("prvkeyhex", keyPair[0]);
		keyPairMap.put("pubkeyhex", keyPair[1]);
		return keyPairMap;
	}
	
	public String getPublicKey(String prvKeyHex) {
		return gmCryptoNativeUtil.getPublicKey(prvKeyHex);
	}
	
	/**
	 * 字符串加密(字符串拼接模式)
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceText 	字符串明文
	 * @return 字符串拼接方式16进制串密文
	 * @throws GmCryptoException 
	 */
	public String encryptFromText(String pubKeyHex, String sourceText) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:encryptFromText]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceText) == 0) {
			throw new GmCryptoException("[SM2:encryptFromText]invalid sourceText");
        }
		if(StringUtils.length(pubKeyHex) < 130 && !pubKeyHex.startsWith("04")) {
			pubKeyHex = "04" + pubKeyHex;
        }
		return gmCryptoNativeUtil.sm2EncryptFromText(this.cipherMode.getValue(), pubKeyHex, sourceText);
	}

	/**
	 * 16进制串加密(字符串拼接模式)
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceHex		16进制串明文
	 * @return 字符串拼接方式16进制串密文
	 * @throws GmCryptoException 
	 */
	public String encryptFromHex(String pubKeyHex, String sourceHex) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:encryptFromHex]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceHex) == 0) {
			throw new GmCryptoException("[SM2:encryptFromHex]invalid sourceHex");
        }
		if(pubKeyHex.length() < 130 && !pubKeyHex.startsWith("04")) {
			pubKeyHex = "04" + pubKeyHex;
        }
		return gmCryptoNativeUtil.sm2EncryptFromHex(this.cipherMode.getValue(), pubKeyHex, sourceHex);
	}
	
	/**
	 * 字节码加密(字符串拼接模式)
	 * @param pubKeyBytes	字节码公钥
	 * @param sourceBytes	字节码明文
	 * @return 字符串拼接方式字节码密文
	 * @throws GmCryptoException 
	 */
	public byte[] encryptFromData(byte[] pubKeyBytes, byte[] sourceBytes) throws GmCryptoException {
		if(this.invalidKey(pubKeyBytes)) {
			throw new GmCryptoException("[SM2:encryptFromBytes]invalid pubKeyBytes");
        }
		if(sourceBytes == null || sourceBytes.length == 0) {
			throw new GmCryptoException("[SM2:encryptFromBytes]invalid sourceBytes");
        }
		return gmCryptoNativeUtil.sm2EncryptFromData(this.cipherMode.getValue(), pubKeyBytes, sourceBytes);
	}
	
	/**
	 * 解密为字符串(字符串拼接模式)
	 * @param prvKeyHex		16进制串私钥
	 * @param cipherHex		16进制串密文
	 * @return 明文字符串
	 * @throws GmCryptoException
	 */
	public String decryptToText(String prvKeyHex, String cipherHex) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:decryptToText]invalid prvKeyHex");
        }
		if(this.invalidCipherObj(cipherHex)) {
			throw new GmCryptoException("[SM2:decryptToText]invalid cipherHex");
        }
		return gmCryptoNativeUtil.sm2DecryptToText(this.cipherMode.getValue(), prvKeyHex, cipherHex);
	}
	
	/**
	 * 解密为16进制串(字符串拼接模式)
	 * @param prvKeyHex		16进制串私钥
	 * @param cipherHex		16进制串密文
	 * @return 明文16进制串
	 * @throws GmCryptoException
	 */
	public String decryptToHex(String prvKeyHex, String cipherHex) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:decryptToHex]invalid prvKeyHex");
        }
		if(this.invalidCipherObj(cipherHex)) {
			throw new GmCryptoException("[SM2:decryptToHex]invalid cipherHex");
        }
		return gmCryptoNativeUtil.sm2DecryptToHex(this.cipherMode.getValue(), prvKeyHex, cipherHex);
	}
	
	/**
	 * 解密为字节码(字符串拼接模式)
	 * @param prvKeyBytes	字节码私钥
	 * @param cipherBytes	字节码密文
	 * @return 明文字节码
	 * @throws GmCryptoException
	 */
	public byte[] decryptToData(byte[] prvKeyBytes, byte[] cipherBytes) throws GmCryptoException {
		if(this.invalidKey(prvKeyBytes)) {
			throw new GmCryptoException("[SM2:decryptToData]invalid prvKeyBytes");
        }
		if(this.invalidCipherObj(cipherBytes)) {
			throw new GmCryptoException("[SM2:decryptToData]invalid cipherBytes");
        }
		return gmCryptoNativeUtil.sm2DecryptToData(this.cipherMode.getValue(), prvKeyBytes, cipherBytes);
	}
	
	/**
	 * 字符串加密(ASN1-DER编码模式)
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceText 	字符串明文
	 * @return 对象转换模式16进制串密文
	 * @throws GmCryptoException 
	 */
	public String encryptASN1FromText(String pubKeyHex, String sourceText) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:encryptASN1FromText]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceText) == 0) {
			throw new GmCryptoException("[SM2:encryptASN1FromText]invalid sourceText");
        }
		return gmCryptoNativeUtil.sm2EncryptASN1FromText(pubKeyHex, sourceText);
	}

	/**
	 * 16进制串加密(ASN1-DER密文编码模式)
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceHex 	16进制串明文
	 * @return 对象转换模式16进制串密文
	 * @throws GmCryptoException 
	 */
	public String encryptASN1FromHex(String pubKeyHex, String sourceHex) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:encryptASN1FromHex]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceHex) == 0) {
			throw new GmCryptoException("[SM2:encryptASN1FromHex]invalid sourceText");
        }
		return gmCryptoNativeUtil.sm2EncryptASN1FromHex(pubKeyHex, sourceHex);
	}

	/**
	 * 字节码加密(ASN1-DER密文编码模式)
	 * @param pubKeyBytes	字节码公钥
	 * @param sourceBytes	字节码明文
	 * @return 对象转换模式字节码密文
	 * @throws GmCryptoException 
	 */
	public byte[] encryptASN1FromData(byte[] pubKeyBytes, byte[] sourceBytes) throws GmCryptoException {
		if(this.invalidKey(pubKeyBytes)) {
			throw new GmCryptoException("[SM2:encryptASN1FromData]invalid pubKeyBytes");
        }
		if(sourceBytes == null || sourceBytes.length == 0) {
			throw new GmCryptoException("[SM2:encryptASN1FromData]invalid sourceBytes");
        }
		return gmCryptoNativeUtil.sm2EncryptASN1FromData(pubKeyBytes, sourceBytes);
	}
	

	/**
	 * 解密为字符串(ASN1-DER密文编码模式)
	 * @param prvKeyHex		16进制串私钥
	 * @param cipherHex		16进制串密文
	 * @return 明文字符串
	 * @throws GmCryptoException
	 */
	public String decryptASN1ToText(String prvKeyHex, String cipherHex) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:decryptASN1ToText]invalid prvKeyHex");
        }
		if(this.invalidCipherObj(cipherHex)) {
			throw new GmCryptoException("[SM2:decryptASN1ToText]invalid cipherHex");
        }
		return gmCryptoNativeUtil.sm2DecryptASN1ToText(prvKeyHex, cipherHex);
	}
	
	/**
	 * 解密为16进制串(ASN1-DER密文编码模式)
	 * @param prvKeyHex		16进制串私钥
	 * @param cipherHex		16进制串密文
	 * @return 明文字符串
	 * @throws GmCryptoException
	 */
	public String decryptASN1ToHex(String prvKeyHex, String cipherHex)  throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:decryptASN1ToHex]invalid prvKeyHex");
        }
		if(this.invalidCipherObj(cipherHex)) {
			throw new GmCryptoException("[SM2:decryptASN1ToHex]invalid cipherHex");
        }
		return gmCryptoNativeUtil.sm2DecryptASN1ToHex(prvKeyHex, cipherHex);
	}
	
	/**
	 * 解密为字节码(ASN1-DER密文编码模式)
	 * @param prvKeyBytes	字节码私钥
	 * @param cipherBytes	字节码密文
	 * @return 明文字节码
	 * @throws GmCryptoException
	 */
	public byte[] decryptASN1ToData(byte[] prvKeyBytes, byte[] cipherBytes) throws GmCryptoException {
		if(this.invalidKey(prvKeyBytes)) {
			throw new GmCryptoException("[SM2:decryptASN1ToData]invalid prvKeyBytes");
        }
		if(this.invalidCipherObj(cipherBytes)) {
			throw new GmCryptoException("[SM2:decryptASN1ToData]invalid cipherBytes");
        }
		return gmCryptoNativeUtil.sm2DecryptASN1ToData(prvKeyBytes, cipherBytes);
	}
	
	/**
	 * 字符串签名
	 * @param prvKeyHex		16进制串私钥
	 * @param sourceText	字符串明文
	 * @return
	 * @throws GmCryptoException 
	 */
	public String signFromText(String prvKeyHex, String sourceText) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:signFromText]invalid prvKeyHex");
        }
		if(StringUtils.length(sourceText) == 0) {
			throw new GmCryptoException("[SM2:signFromText]invalid sourceText");
        }
		return gmCryptoNativeUtil.sm2SignFromText(prvKeyHex, sourceText);
	}
	
	/**
	 * 16进制串签名
	 * @param prvKeyHex		16进制串私钥
	 * @param sourceHex		16进制串明文
	 * @return
	 * @throws GmCryptoException 
	 */
	public String signFromHex(String prvKeyHex, String sourceHex) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:signFromHex]invalid prvKeyHex");
        }
		if(StringUtils.length(sourceHex) == 0) {
			throw new GmCryptoException("[SM2:signFromHex]invalid sourceText");
        }
		return gmCryptoNativeUtil.sm2SignFromHex(prvKeyHex, sourceHex);
	}
	
	/**
	 * 字节码签名
	 * @param prvKeyBytes	字节码私钥
	 * @param sourceBytes	字节码明文
	 * @return
	 * @throws GmCryptoException 
	 */
	public byte[] signFromData(byte[] prvKeyBytes, byte[] sourceBytes) throws GmCryptoException {
		if(this.invalidKey(prvKeyBytes)) {
			throw new GmCryptoException("[SM2:signFromData]invalid prvKeyBytes");
        }
		if(sourceBytes == null || sourceBytes.length == 0) {
			throw new GmCryptoException("[SM2:signFromData]invalid sourceBytes");
        }
		return gmCryptoNativeUtil.sm2SignFromData(prvKeyBytes, sourceBytes);
	}
	
	/**
	 * 字符串验签
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceText	字符串明文
	 * @param signHex		16进制串签名
	 * @return
	 * @throws GmCryptoException 
	 */
	public boolean verifySignFromText(String pubKeyHex, String sourceText, String signHex) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:verifySignFromText]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceText) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromText]invalid sourceText");
        }
		if(StringUtils.length(signHex) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid signHex");
        }
		return gmCryptoNativeUtil.sm2VerifySignFromText(pubKeyHex, sourceText, signHex) == 1;
	}
	
	/**
	 * 16进制串验签
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceHex		16进制串明文
	 * @param signHex		16进制串签名
	 * @return
	 * @throws GmCryptoException 
	 */
	public boolean verifySignFromHex(String pubKeyHex, String sourceHex, String signHex) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceHex) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid sourceText");
        }
		if(StringUtils.length(signHex) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid signHex");
        }
		return gmCryptoNativeUtil.sm2VerifySignFromHex(pubKeyHex, sourceHex, signHex) == 1;
	}
	
	/**
	 * 字节码验签
	 * @param pubKeyBytes	字节码公钥
	 * @param sourceBytes	字节码明文
	 * @param signBytes		字节码签名
	 * @return
	 * @throws GmCryptoException 
	 */
	public boolean verifySignFromData(byte[] pubKeyBytes, byte[] sourceBytes, byte[] signBytes) throws GmCryptoException {
		if(this.invalidKey(pubKeyBytes)) {
			throw new GmCryptoException("[SM2:verifySignFromData]invalid pubKeyBytes");
        }
		if(sourceBytes == null || sourceBytes.length == 0) {
			throw new GmCryptoException("[SM2:verifySignFromData]invalid sourceBytes");
        }
		if(signBytes == null || signBytes.length == 0) {
			throw new GmCryptoException("[SM2:verifySignFromData]invalid signBytes");
        }
		return gmCryptoNativeUtil.sm2VerifySignFromData(pubKeyBytes, sourceBytes, signBytes) == 1;
	}
	
	/**
	 * 字符串签名(ASN1-DER签名编码模式)
	 * @param prvKeyHex		16进制串私钥
	 * @param sourceText	字符串明文
	 * @return
	 * @throws GmCryptoException 
	 */
	public String signASN1FromText(String prvKeyHex, String sourceText) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:signFromText]invalid prvKeyHex");
        }
		if(StringUtils.length(sourceText) == 0) {
			throw new GmCryptoException("[SM2:signFromText]invalid sourceText");
        }
		return gmCryptoNativeUtil.sm2SignASN1FromText(prvKeyHex, sourceText);
	}
	
	/**
	 * 16进制串签名(ASN1-DER签名编码模式)
	 * @param prvKeyHex		16进制串私钥
	 * @param sourceHex		16进制串明文
	 * @return
	 * @throws GmCryptoException 
	 */
	public String signASN1FromHex(String prvKeyHex, String sourceHex) throws GmCryptoException {
		if(this.invalidKey(prvKeyHex)) {
			throw new GmCryptoException("[SM2:signFromHex]invalid prvKeyHex");
        }
		if(StringUtils.length(sourceHex) == 0) {
			throw new GmCryptoException("[SM2:signFromHex]invalid sourceText");
        }
		return gmCryptoNativeUtil.sm2SignASN1FromHex(prvKeyHex, sourceHex);
	}
	
	/**
	 * 字节码签名(ASN1-DER签名编码模式)
	 * @param prvKeyBytes	字节码私钥
	 * @param sourceBytes	字节码明文
	 * @return
	 * @throws GmCryptoException 
	 */
	public byte[] signASN1FromData(byte[] prvKeyBytes, byte[] sourceBytes) throws GmCryptoException {
		if(this.invalidKey(prvKeyBytes)) {
			throw new GmCryptoException("[SM2:signFromData]invalid prvKeyBytes");
        }
		if(sourceBytes == null || sourceBytes.length == 0) {
			throw new GmCryptoException("[SM2:signFromData]invalid sourceBytes");
        }
		return gmCryptoNativeUtil.sm2SignASN1FromData(prvKeyBytes, sourceBytes);
	}
	
	/**
	 * 字符串验签(ASN1-DER签名编码模式)
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceText	字符串明文
	 * @param signHex		16进制串签名
	 * @return
	 * @throws GmCryptoException 
	 */
	public boolean verifySignASN1FromText(String pubKeyHex, String sourceText, String signHex) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:verifySignFromText]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceText) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromText]invalid sourceText");
        }
		if(StringUtils.length(signHex) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid signHex");
        }
		return gmCryptoNativeUtil.sm2VerifySignASN1FromText(pubKeyHex, sourceText, signHex) == 1;
	}
	
	/**
	 * 16进制串验签(ASN1-DER签名编码模式)
	 * @param pubKeyHex		16进制串公钥
	 * @param sourceHex		16进制串明文
	 * @param signHex		16进制串签名
	 * @return
	 * @throws GmCryptoException 
	 */
	public boolean verifySignASN1FromHex(String pubKeyHex, String sourceHex, String signHex) throws GmCryptoException {
		if(this.invalidKey(pubKeyHex)) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid pubKeyHex");
        }
		if(StringUtils.length(sourceHex) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid sourceText");
        }
		if(StringUtils.length(signHex) == 0) {
			throw new GmCryptoException("[SM2:verifySignFromHex]invalid signHex");
        }
		return gmCryptoNativeUtil.sm2VerifySignASN1FromHex(pubKeyHex, sourceHex, signHex) == 1;
	}
	
	/**
	 * 字节码验签(ASN1-DER签名编码模式)
	 * @param pubKeyBytes	字节码公钥
	 * @param sourceBytes	字节码明文
	 * @param signBytes		字节码签名
	 * @return
	 * @throws GmCryptoException 
	 */
	public boolean verifySignASN1FromData(byte[] pubKeyBytes, byte[] sourceBytes, byte[] signBytes) throws GmCryptoException {
		if(this.invalidKey(pubKeyBytes)) {
			throw new GmCryptoException("[SM2:verifySignFromData]invalid pubKeyBytes");
        }
		if(sourceBytes == null || sourceBytes.length == 0) {
			throw new GmCryptoException("[SM2:verifySignFromData]invalid sourceBytes");
        }
		if(signBytes == null || signBytes.length == 0) {
			throw new GmCryptoException("[SM2:verifySignFromData]invalid signBytes");
        }
		return gmCryptoNativeUtil.sm2VerifySignASN1FromData(pubKeyBytes, sourceBytes, signBytes) == 1;
	}
	
	private final static List<Integer> keyHexLenList = Arrays.asList(new Integer[]{60, 62, 64, 66, 68, 128});
	
	private final static List<Integer> keyByteLenList = Arrays.asList(new Integer[]{30, 31, 32, 33, 34, 64});

	
	/**
	 * 验证是否是无效的Key
	 * @param keyObj
	 * @return
	 */
	private final boolean invalidKey(final Object keyObj) {
		if(keyObj == null) {
			return true;
		} else if(keyObj instanceof String) {
			String keyString = (String)keyObj;
			int keyLength = keyString.length();
			if(keyLength == 130 && keyString.startsWith("04")) {
				return false;
			} else if(keyHexLenList.contains(keyLength)) {
				return false;
			} else {
				return true;
			}
		} else if(keyObj instanceof byte[]) {
			byte[] keyBytes = (byte[])keyObj;
			if(keyBytes.length == 65 && keyBytes[0] == 4) {
				return false;
			} else if(keyByteLenList.contains(keyBytes.length)) {
				return false;
			} else {
				return true;
			}
		} else {
			return true;
		}
	}
	
	/**
	 * 验证是否是无效的密文
	 * @param cipherObj
	 * @return
	 */
	private final boolean invalidCipherObj(final Object cipherObj) {
		if(cipherObj == null) {
			return true;
		} else if(cipherObj instanceof String) {
			String cipherHex = (String)cipherObj;
			int cipherLength = cipherHex.length();
			if(cipherLength < 196 || cipherLength%2 == 1) {
				return true;
			} else {
				return false;
			}
		} else if(cipherObj instanceof byte[]) {
			byte[] cipherBytes = (byte[])cipherObj;
			if(cipherBytes.length < 98) {
				return true;
			} else {
				return false;
			}
		} else {
			return true;
		}
	}
}
