package cn.aotcloud.safe.util;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.DigestUtils;
import org.springframework.util.StreamUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileContentUtils {
	
	private static final Logger LOGGER = LoggerFactory.getLogger(FileContentUtils.class);
    
	private static ConcurrentHashMap<String, String> STATIC_FILE_CACHE = new ConcurrentHashMap<String, String>();
	
	public static String getFileContent(boolean isCached, String filePath) {
		return getFileContent(isCached, filePath, null, null);
	}
	
	public static String getFileContent(boolean isCached, String filePath, String regex, String replacement) {
		if(!isCached) {
			String content = readFileContent(filePath);
			return StringUtils.isNoneBlank(regex, replacement) ? content.replaceAll(regex, replacement) : content;
		} else {
			String key = DigestUtils.md5DigestAsHex(filePath.getBytes(Charset.forName("UTF-8")));
			String val = null;
			if(STATIC_FILE_CACHE.containsKey(key)) {
				val = STATIC_FILE_CACHE.get(key);
			} else {
				val = cacheFileContent(filePath, key, regex, replacement);
			}
			return val;
		}
	}
	
	public static String readFileContent(String filePath) {
		InputStream in = ClassLoader.getSystemClassLoader().getResourceAsStream(filePath);
		try {
			return IOUtils.toString(in, "UTF-8");
		} catch (IOException e) {
			LOGGER.error(String.format("直接读取文件内容异常：%s", e.getMessage()));
			return null;
		} finally {
			if(in != null) {
				try {
					in.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	public static String cacheFileContent(String filePath, String key, String regex, String replacement) {
		InputStream in = ClassLoader.getSystemClassLoader().getResourceAsStream(filePath);
		String content = null;
		try {
			content = StreamUtils.copyToString(in, Charset.forName("UTF-8"));
			content = StringUtils.isNoneBlank(regex, replacement) ? content.replaceAll(regex, replacement) : content;
			STATIC_FILE_CACHE.put(key, content);
		} catch (IOException e) {
			LOGGER.error(String.format("缓存读取文件内容异常：%s", e.getMessage()));
		} finally {
			if(in != null) {
				try {
					in.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		
		return content;
	}
}
