package cn.allbs.websocket.handler;

import cn.allbs.common.constant.AllbsCoreConstants;
import cn.allbs.common.constant.ParamsConstants;
import cn.allbs.common.enums.WebSocketResponseEnum;
import cn.allbs.common.utils.ObjectUtil;
import cn.allbs.websocket.behavior.AuthRequest;
import cn.allbs.websocket.behavior.SysDefaultRequest;
import cn.allbs.websocket.behavior.UserJoinNoticeRequest;
import cn.allbs.websocket.exception.MessageHandlerException;
import cn.allbs.websocket.message.Message;
import cn.allbs.websocket.message.MessageFactory;
import cn.allbs.websocket.util.WebSocketUtil;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.ClassUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import static cn.allbs.websocket.enums.MessageType.AUTH_REQUEST;
import static cn.allbs.websocket.enums.MessageType.SYS_DEFAULT_REQUEST;

/**
 * @author ChenQi
 */
@Slf4j
public class AllbsWebSocketHandler extends TextWebSocketHandler {

    /**
     * OnOpen
     *
     * @param session 会话
     * @throws MessageHandlerException 消息处理器未注册异常
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) throws MessageHandlerException {
        log.debug("[afterConnectionEstablished][session({}) 接入]", session);
        // 解析 accessToken
        String accessToken = (String) session.getAttributes().get(AllbsCoreConstants.ACCESS_TOKEN);
        String userName = (String) session.getAttributes().get(AllbsCoreConstants.USERNAME);
        // 创建 AuthRequest 消息类型
        AuthRequest authRequest = AuthRequest.builder().accessToken(accessToken).userName(userName).build();
        MessageFactory.getInvokeStrategy(AUTH_REQUEST.name()).execute(session, authRequest);
    }

    /**
     * OnMessage 事件
     *
     * @param session     会话
     * @param textMessage 消息
     */
    @Override
    public void handleTextMessage(WebSocketSession session, TextMessage textMessage) {
        log.debug("[handleMessage][session({}) 接收到一条消息({})]", session, textMessage);
        try {
            // 判断是否为JSON格式的消息
            if (JSONUtil.isJson(textMessage.getPayload())) {
                JSONObject jsonObject = JSONUtil.parseObj(textMessage.getPayload());
                if (jsonObject.containsKey(ParamsConstants.WEBSOCKET_TYPE) && ObjectUtil.isNotEmpty(jsonObject.get(ParamsConstants.WEBSOCKET_TYPE))) {
                    Class<?> targetClass = AopProxyUtils.ultimateTargetClass(MessageFactory.getInvokeStrategy(jsonObject.getStr(ParamsConstants.WEBSOCKET_TYPE)));
                    Class<?> fieldClass = ClassUtil.getTypeArgument(targetClass);
                    Class<? extends Message> messageClass = fieldClass.asSubclass(Message.class);
                    Message messageObj = BeanUtil.toBean(jsonObject, messageClass);
                    MessageFactory.getInvokeStrategy(jsonObject.getStr(ParamsConstants.WEBSOCKET_TYPE)).execute(session, messageObj);
                    return;
                }
            }
            SysDefaultRequest sysDefaultRequest = SysDefaultRequest.builder().content(textMessage.getPayload()).build();
            MessageFactory.getInvokeStrategy(SYS_DEFAULT_REQUEST.name()).execute(session, sysDefaultRequest);
        } catch (Throwable throwable) {
            log.info("[onMessage][session({}) message({}) 发生异常]", session, throwable);
        }
    }

    /**
     * OnClose 事件
     *
     * @param session
     * @param status
     * @throws Exception
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) throws Exception {
        log.info("[afterConnectionClosed][session({}) 连接关闭。关闭原因是({})}]", session, status);
        Object userName = session.getAttributes().get(AllbsCoreConstants.USERNAME);
        if (ObjectUtil.isNotEmpty(userName)) {
            UserJoinNoticeRequest request = UserJoinNoticeRequest.builder().userName(userName.toString()).message(StrUtil.format("{}用户离线", userName)).build();
            WebSocketUtil.broadcastWithoutSelf(WebSocketResponseEnum.OFFLINE_USER, request, userName.toString());
        }
        WebSocketUtil.removeSession(session);
    }

    /**
     * 对应 OnError 事件
     *
     * @param session
     * @param exception
     * @throws Exception
     */
    @Override
    public void handleTransportError(WebSocketSession session, Throwable exception) throws Exception {
        log.info("[handleTransportError][session({}) 发生异常]", session, exception);
    }
}
