package cn.allbs.websocket.util;

import cn.allbs.common.constant.AllbsCoreConstants;
import cn.allbs.common.enums.WebSocketResponseEnum;
import cn.allbs.websocket.message.Message;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONObject;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;

import java.io.IOException;

/**
 * @author ChenQi
 */
@UtilityClass
@Slf4j
public class WebSocketUtil {


    /**
     * 添加 Session 。在这个方法中，会添加用户和 Session 之间的映射
     *
     * @param session  Session
     * @param userName 用户
     */
    public static void addSession(WebSocketSession session, String userName) {
        // 新增用户session
        WebSocketSessionManager.add(userName, session);
    }

    /**
     * 移除 Session 。
     *
     * @param session Session
     */
    public static void removeSession(WebSocketSession session) {
        Object user = session.getAttributes().get(AllbsCoreConstants.USERNAME);
        if (ObjectUtil.isNotEmpty(user)) {
            // 用户退出，移除缓存
            WebSocketSessionManager.remove(user.toString());
        }
    }

    /**
     * 广播发送消息给所有在线用户（包括自己）
     *
     * @param webSocketResponseEnum websocket消息枚举
     * @param data                  待发送数据
     * @param <T>                   消息类型
     */
    public static <T extends Message> void broadcast(WebSocketResponseEnum webSocketResponseEnum, T data) {
        broadcast(webSocketResponseEnum.getCode().toString(), webSocketResponseEnum.getMsg(), data);
    }

    /**
     * 广播发送消息给所有在线用户（包括自己）
     *
     * @param code 消息类型
     * @param data 待发送数据
     * @param msg  消息提示
     * @param <T>  消息类型
     */
    public static <T extends Message> void broadcast(String code, String msg, T data) {
        JSONObject json = new JSONObject();
        json.putOpt("code", code);
        json.putOpt("msg", msg);
        json.putOpt("data", data);
        // 创建消息
        TextMessage textMessage = new TextMessage(json.toString());
        // 所以所有用户session逐个发送
        WebSocketSessionManager.getAll().forEach(session -> sendTextMessage(session, textMessage));
    }

    /**
     * 广播发送消息给所有在线用户(除自己)
     *
     * @param webSocketResponseEnum websocket发送消息类型枚举
     * @param data                  数据
     * @param <T>                   消息类型
     */
    public static <T extends Message> void broadcastWithoutSelf(WebSocketResponseEnum webSocketResponseEnum, T data, String userName) {
        broadcastWithoutSelf(webSocketResponseEnum.getCode().toString(), webSocketResponseEnum.getMsg(), data, userName);
    }

    /**
     * 广播发送消息给所有在线用户(除自己)
     *
     * @param code 消息类型
     * @param data 数据
     * @param msg  消息提示
     * @param <T>  消息类型
     */
    public static <T extends Message> void broadcastWithoutSelf(String code, String msg, T data, String userName) {
        JSONObject json = new JSONObject();
        json.putOpt("code", code);
        json.putOpt("msg", msg);
        json.putOpt("data", data);
        // 创建消息
        TextMessage textMessage = new TextMessage(json.toString());
        // 除自身外,所有用户session逐个发送
        WebSocketSessionManager.getAll().stream().filter(a -> !a.equals(WebSocketSessionManager.get(userName))).forEach(session -> sendTextMessage(session, textMessage));
    }

    /**
     * 发送消息给单个用户的 Session
     *
     * @param session               Session
     * @param webSocketResponseEnum websocket消息枚举
     * @param data                  消息体
     * @param <T>                   消息类型
     */
    public static <T extends Message> void send(WebSocketSession session, WebSocketResponseEnum webSocketResponseEnum, T data) {
        send(session, webSocketResponseEnum.getCode().toString(), webSocketResponseEnum.getMsg(), data);
    }

    /**
     * 发送消息给单个用户的 Session
     *
     * @param session Session
     * @param code    消息类型
     * @param data    消息体
     * @param msg     消息提示
     * @param <T>     消息类型
     */
    public static <T extends Message> void send(WebSocketSession session, String code, String msg, T data) {
        JSONObject json = new JSONObject();
        json.putOpt("code", code);
        json.putOpt("msg", msg);
        json.putOpt("data", data);
        // 创建消息
        TextMessage textMessage = new TextMessage(json.toString());
        // 遍历给单个 Session ，进行逐个发送
        sendTextMessage(session, textMessage);
    }

    /**
     * 发送消息给指定用户
     *
     * @param userName 指定用户
     * @param code     消息类型
     * @param data     消息体
     * @param <T>      消息类型
     * @return 发送是否成功你那个
     */
    public static <T extends Message> boolean send(String userName, String code, String msg, T data) {
        // 获得用户对应的 Session
        WebSocketSession session = WebSocketSessionManager.get(userName);
        if (session == null) {
            log.error("{}用户不在线", userName);
            return false;
        }
        // 发送消息
        send(session, code, msg, data);
        return true;
    }

    /**
     * 发送消息给指定用户
     *
     * @param userName              用户名
     * @param webSocketResponseEnum 消息枚举
     * @param data                  消息内容
     * @param <T>                   消息内容
     * @return 是否发送成功
     */
    public static <T extends Message> boolean send(String userName, WebSocketResponseEnum webSocketResponseEnum, T data) {
        return send(userName, webSocketResponseEnum.getCode().toString(), webSocketResponseEnum.getMsg(), data);
    }

    /**
     * 真正发送消息
     *
     * @param session     Session
     * @param textMessage 消息
     */
    private static void sendTextMessage(WebSocketSession session, TextMessage textMessage) {
        if (session == null) {
            log.debug("[sendTextMessage][session 为 null]");
            return;
        }
        try {
            session.sendMessage(textMessage);
        } catch (IOException e) {
            log.debug("[sendTextMessage][session({}) 发送消息{}) 发生异常",
                    session, textMessage, e);
        }
    }
}
