package cn.allbs.utils.validate;

import cn.allbs.utils.annotation.ParamVal;
import cn.allbs.utils.common.enums.RegexOption;
import cn.allbs.utils.validate.exception.ParamCheckException;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReUtil;
import cn.hutool.core.util.StrUtil;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;
import java.util.regex.Pattern;

/**
 * 参数切片校验
 *
 * @author ChenQi
 */
@Aspect
@Component
public class ParamCheckAspect {

    /**
     * 校验的类型
     */
    private static final String LINK_HASH_MAP_TYPE = "java.util.LinkedHashMap";

    /**
     * 定义一个方法，用于声明切入表达式。
     */
    @Pointcut("@annotation(cn.allbs.utils.annotation.ParamVal)")
    public void paramCheckAspect() {
    }

    @Before("paramCheckAspect()")
    public void parameterVerify(JoinPoint point) throws Exception {

        //通过签名 获取方法签名
        MethodSignature signature = (MethodSignature) point.getSignature();
        //通过方法签名获取执行方法
        Method method = signature.getMethod();
        //获取参数列表
        Object[] args = point.getArgs();
    }

    /**
     * 正则的校验方法
     *
     * @param paramVal 校验规则
     * @param fieldObj 校验者
     */
    private void validate(ParamVal paramVal, Object fieldObj) throws Exception {
        //获取paramVal的name
        String name = paramVal.value();
        String message = StrUtil.format("参数{}不能为空！", name);
        //是否时必传 断言判断
        if (paramVal.required()) {
            if (fieldObj == null) {
                throw new ParamCheckException(StrUtil.format("{}为必传参数！", name));
            }
        }
        if (ObjectUtil.isEmpty(fieldObj)) {
            throw new ParamCheckException(message);
        }
        //是否有正则校验
        if (!RegexOption.DEFAULT.getRegex().equals(paramVal.regular().getRegex())) {
            Pattern pattern = Pattern.compile(paramVal.regular().getRegex());
            if (ReUtil.isMatch(pattern, String.valueOf(fieldObj))) {
                throw new ParamCheckException(StrUtil.format("参数{}请求规则不匹配！", name));
            }
        }
    }
}
