package cn.allbs.utils.unipush;

import cn.allbs.utils.common.constants.AllbsCoreConstants;
import cn.allbs.utils.common.enums.MobileType;
import com.gexin.fastjson.JSON;
import com.gexin.rp.sdk.base.IPushResult;
import com.gexin.rp.sdk.base.impl.AppMessage;
import com.gexin.rp.sdk.base.impl.SingleMessage;
import com.gexin.rp.sdk.base.impl.Target;
import com.gexin.rp.sdk.base.payload.APNPayload;
import com.gexin.rp.sdk.http.IGtPush;
import com.gexin.rp.sdk.template.AbstractTemplate;
import com.gexin.rp.sdk.template.LinkTemplate;
import com.gexin.rp.sdk.template.NotificationTemplate;
import com.gexin.rp.sdk.template.TransmissionTemplate;
import com.gexin.rp.sdk.template.style.Style0;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 功能: 个推逻辑
 *
 * @author chenQi
 * @version 1.0
 * @date 2020/9/29 15:23
 */
@Slf4j
@RequiredArgsConstructor
public class UniPushTemplate implements InitializingBean {

    private final String url;

    private final String appId;

    private final String appKey;

    private final String masterSecret;

    private final Long offlineExpireTime;

    private IGtPush push;

    /**
     * 推送
     *
     * @param title   标题
     * @param content 内容
     */
    @SneakyThrows
    public void pushToAllUser(String title, String content) {
        Style0 style = new Style0();
        // STEP2：设置推送标题、推送内容
        style.setTitle(title);
        style.setText(content);
        // 设置推送图标
        style.setLogo("push.png");
        // STEP3：设置响铃、震动等推送效果
        // 设置响铃
        style.setRing(true);
        // 设置震动
        style.setVibrate(true);
        // STEP4：选择通知模板
        NotificationTemplate template = new NotificationTemplate();
        template.setAppId(appId);
        template.setAppkey(appKey);
        template.setStyle(style);
        // STEP5：定义"AppMessage"类型消息对象,设置推送消息有效期等推送参数
        List<String> appIds = new ArrayList<String>();
        appIds.add(appId);
        AppMessage message = new AppMessage();
        message.setData(template);
        message.setAppIdList(appIds);
        message.setOffline(true);
        // 时间单位为毫秒
        message.setOfflineExpireTime(offlineExpireTime * 60 * 1000);
        // STEP6：执行推送
        IPushResult ret = push.pushMessageToApp(message);
        log.info(AllbsCoreConstants.ALLBS_TIP + "app message push to all user, response is " + ret.getResponse().toString());
    }

    /**
     * 推送指定用户
     *
     * @param title    标题
     * @param content  内容
     * @param clientId 用户id
     * @param type     手机类型
     */
    @SneakyThrows
    public void pushToAssignUser(String title, String content, String clientId, Integer type) {
        Style0 style = new Style0();
        // STEP2：设置推送标题、推送内容
        style.setTitle(title);
        style.setText(content);
        // 设置推送图标
        style.setLogo("push.png");
        // STEP3：设置响铃、震动等推送效果
        // 设置响铃
        style.setRing(true);
        // 设置震动
        style.setVibrate(true);
        // STEP4：选择通知模板
        AbstractTemplate template = null;
        if (MobileType.ANDROID.getType().equals(type)) {
            template = getNotificationTemplate(title, content);
        } else if (MobileType.IOS.getType().equals(type)) {
            template = getIOSTemplate(title, content);
        } else {
            template = getLinkTemplate();
        }
        pushToSingle(clientId, template);
    }

    private void pushToSingle(String clientId, AbstractTemplate template) {
        // 定义"AppMessage"类型消息对象，设置消息内容模板、发送的目标App列表、是否支持离线发送、以及离线消息有效期(单位毫秒)
        SingleMessage message = new SingleMessage();
        message.setData(template);
        message.setOffline(true);
        message.setData(template);
        // 可选，1为wifi，0为不限制网络环境。根据手机处于的网络情况，决定是否下发
        message.setPushNetWorkType(0);
        // 毫秒
        message.setOfflineExpireTime(offlineExpireTime * 60 * 1000);
        Target target = new Target();
        target.setAppId(appId);
        target.setClientId(clientId);
        IPushResult ret = push.pushMessageToSingle(message, target);
        log.info(AllbsCoreConstants.ALLBS_TIP + "app message push to signal user, response is " + ret.getResponse().toString());
    }

    private NotificationTemplate getNotificationTemplate(String title, String text) {
        // 定义"点击链接打开通知模板"，并设置标题、内容、链接
        NotificationTemplate template = new NotificationTemplate();
        template.setAppId(appId);
        template.setAppkey(appKey);
        template.setTitle(title);
        template.setText(text);
        template.setLogoUrl(url);
        return template;
    }

    public TransmissionTemplate getIOSTemplate(String title, String content) {
        TransmissionTemplate template = new TransmissionTemplate();
        template.setAppId(appId);
        template.setAppkey(appKey);
        Map<String, Object> contentMap = new HashMap<String, Object>();
        contentMap.put("content", content);
        contentMap.put("title", title);
        String jsonString = JSON.toJSONString(contentMap);
        // 透传消息设置，1为强制启动应用，客户端接收到消息后就会立即启动应用；2为等待应用启动
        template.setTransmissionType(2);
        template.setTransmissionContent(jsonString);
        APNPayload payload = new APNPayload();

        //  在已有数字基础上加1显示，设置为-1时，在已有数字上减1显示，设置为数字时，显示指定数字
        //  payload.setAutoBadge("+1");
        //  payload.setContentAvailable(-1);
        payload.setSound("default");
        payload.setCategory("$由客户端定义");

        // 字典模式使用APNPayload.DictionaryAlertMsg
        payload.setAlertMsg(getDictionaryAlertMsg(title, content));

        template.setAPNInfo(payload);
        return template;
    }

    private LinkTemplate getLinkTemplate() {
        // 定义"点击链接打开通知模板"，并设置标题、内容、链接
        LinkTemplate template = new LinkTemplate();
        template.setAppId(appId);
        template.setAppkey(appKey);
        template.setTitle("告警通知");
        template.setText("");
        template.setUrl("http://getui.com");
        template.setLogoUrl(url);
        return template;
    }

    private static APNPayload.DictionaryAlertMsg getDictionaryAlertMsg(String title, String content) {
        APNPayload.DictionaryAlertMsg alertMsg = new APNPayload.DictionaryAlertMsg();
        alertMsg.setBody(content);
        alertMsg.setActionLocKey("ActionLockey");
        alertMsg.setLocKey("LocKey");
        alertMsg.addLocArg("loc-args");
        alertMsg.setLaunchImage("launch-image");
        // iOS8.2以上版本支持
        alertMsg.setTitle(title);
        alertMsg.setTitleLocKey("TitleLocKey");
        alertMsg.addTitleLocArg("TitleLocArg");
        return alertMsg;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.hasText(url, "url 为空");
        Assert.hasText(appKey, "appKey为空");
        Assert.hasText(masterSecret, "masterSecret为空");
        this.push = new IGtPush(url, appKey, masterSecret);
    }
}
