package cn.allbs.utils.common.handle;

import cn.allbs.utils.common.constants.AllbsCoreConstants;
import cn.allbs.utils.common.enums.ErrorCodeEnum;
import cn.allbs.utils.common.util.R;
import cn.allbs.utils.validate.exception.ParamCheckException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.server.ServerErrorException;
import org.springframework.web.servlet.NoHandlerFoundException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * 异常统一返回
 *
 * @author chenqi
 * @version 1.0
 * @date 2020/8/7 9:42
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandlerResolver {

    /**
     * Exception
     *
     * @return R
     */
    @ExceptionHandler(value = Exception.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R unKnowExceptionHandler() {
        return R.exception(ErrorCodeEnum.EXCEPTION_RESPONSE);
    }

    /**
     * RuntimeException
     *
     * @param e RuntimeException
     * @return R
     */
    @ExceptionHandler(value = RuntimeException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R runtimeExceptionHandler(RuntimeException e) {
        return R.exception(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getLocalizedMessage());
    }

    /**
     * 空指针异常
     */
    @ExceptionHandler(NullPointerException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R nullPointerExceptionHandler(Exception e) {
        e.printStackTrace();
        return R.exception(ErrorCodeEnum.NULL_POINTER_EXCEPTION_RESPONSE);
    }

    /**
     * 类型转换异常
     */
    @ExceptionHandler(ClassCastException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R classCastExceptionHandler() {
        return R.exception(ErrorCodeEnum.CAST_EXCEPTION_RESPONSE);
    }

    /**
     * IO异常
     */
    @ExceptionHandler(IOException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R iOExceptionHandler() {
        return R.exception(ErrorCodeEnum.IO_EXCEPTION_RESPONSE);
    }

    /**
     * 未知方法异常
     */
    @ExceptionHandler(NoSuchMethodException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R noSuchMethodExceptionHandler() {
        return R.exception(ErrorCodeEnum.NO_SUCH_METHOD_EXCEPTION_RESPONSE);
    }

    /**
     * 数组越界异常
     */
    @ExceptionHandler(IndexOutOfBoundsException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R indexOutOfBoundsExceptionHandler() {
        return R.exception(ErrorCodeEnum.INDEX_OUT_OF_BOUND_EXCEPTION_RESPONSE);
    }

    /**
     * 400错误
     */
    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R requestNotReadable() {
        return R.exception(ErrorCodeEnum.HTTP_MESSAGE_NOT_READABLE_EXCEPTION_RESPONSE);
    }

    /**
     * 400错误 类型不匹配
     */
    @ExceptionHandler({TypeMismatchException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R requestTypeMismatch() {
        return R.exception(ErrorCodeEnum.TYPE_MISMATCH_EXCEPTION_RESPONSE);
    }

    /**
     * 400错误 缺少参数
     */
    @ExceptionHandler({MissingServletRequestParameterException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R requestMissingServletRequest() {
        return R.exception(ErrorCodeEnum.MISSING_SERVLET_REQUEST_PARAMETER_EXCEPTION_RESPONSE);
    }

    /**
     * 参数错误
     *
     * @return
     */
    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R methodArgumentNotValidExceptionHandler(MethodArgumentNotValidException e) {
        List<String> list = new ArrayList<>();
        // 从异常对象中拿到ObjectError对象
        if (!e.getBindingResult().getAllErrors().isEmpty()) {
            for (ObjectError error : e.getBindingResult().getAllErrors()) {
                list.add(error.getDefaultMessage().toString());
            }
        }
        log.error(AllbsCoreConstants.ALLBS_TIP + "方法参数错误异常" + list);
        return R.exception(ErrorCodeEnum.METHOD_ARGUMENT_NOT_VALID_EXCEPTION_RESPONSE.getCode(), list.toString());
    }

    /**
     * 处理自定义的异常
     *
     * @param ex
     * @return
     */
    @ExceptionHandler(ParamCheckException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R resolveMyException(ParamCheckException ex) {
        return R.exception(ErrorCodeEnum.METHOD_ARGUMENT_NOT_VALID_EXCEPTION_RESPONSE.getCode(), ex.getMessage());
    }

    /**
     * 405错误
     */
    @ExceptionHandler({HttpRequestMethodNotSupportedException.class})
    @ResponseStatus(HttpStatus.METHOD_NOT_ALLOWED)
    public R request405() {
        return R.exception(ErrorCodeEnum.HTTP_REQUEST_METHOD_NOT_SUPPORTED_EXCEPTION_RESPONSE);
    }

    /**
     * 406错误
     */

    @ExceptionHandler({HttpMediaTypeNotAcceptableException.class})
    @ResponseStatus(HttpStatus.NOT_ACCEPTABLE)
    public R request406() {
        return R.exception(ErrorCodeEnum.HTTP_MEDIA_TYPE_NOT_ACCEPTABLE_EXCEPTION_RESPONSE);
    }

    /**
     * 500错误
     */
    @ExceptionHandler({ConversionNotSupportedException.class, HttpMessageNotWritableException.class})
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R server500() {
        return R.exception(ErrorCodeEnum.CONVERSION_NOT_ACCEPTABLE_EXCEPTION_RESPONSE);
    }

    /**
     * 媒体格式不支持
     *
     * @return R
     */
    @ExceptionHandler(value = HttpMediaTypeNotSupportedException.class)
    @ResponseStatus(HttpStatus.UNSUPPORTED_MEDIA_TYPE)
    public R httpMediaTypeNotSupportedExceptionHandler() {
        return R.exception(ErrorCodeEnum.HTTP_MEDIA_TYPE_NOT_SUPPORTED_EXCEPTION_RESPONSE);
    }


    /**
     * 404
     */
    @ExceptionHandler(value = NoHandlerFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public R notFoundException() {
        return R.exception(ErrorCodeEnum.NO_HANDLER_FOUND_EXCEPTION_RESPONSE);
    }

    /**
     * ServerErrorException
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = ServerErrorException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R serverErrorExceptionHandler(ServerErrorException e) {
        return R.exception(HttpStatus.INTERNAL_SERVER_ERROR.value(), e.getLocalizedMessage());
    }
}
