package cn.allbs.utils.excel.handle;

import cn.allbs.utils.annotation.ExportExcel;
import cn.allbs.utils.excel.aop.DynamicNameAspect;
import cn.allbs.utils.excel.convert.LocalDateStringConverter;
import cn.allbs.utils.excel.convert.LocalDateTimeStringConverter;
import cn.allbs.utils.excel.convert.TimestampStringConverter;
import cn.allbs.utils.excel.exception.ExcelException;
import cn.allbs.utils.excel.head.HeadGenerator;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.handler.WriteHandler;
import com.alibaba.excel.write.metadata.WriteSheet;
import lombok.SneakyThrows;
import org.springframework.beans.BeanUtils;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.HttpHeaders;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Objects;

/**
 * 功能:
 *
 * @author ChenQi
 * @version 1.0
 * @date 2021/3/29
 */
public abstract class AbstractSheetWriteHandler implements SheetWriteHandler {

    @Override
    public void check(ExportExcel exportExcel) {
        if (!StringUtils.hasText(exportExcel.name())) {
            throw new ExcelException("@ExportExcel name 配置不合法");
        }

        if (exportExcel.sheet().length == 0) {
            throw new ExcelException("@ExportExcel sheet 配置不合法");
        }
    }

    @Override
    @SneakyThrows
    public void export(Object o, HttpServletResponse response, ExportExcel exportExcel) {
        check(exportExcel);
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        String name = (String) Objects.requireNonNull(requestAttributes)
                .getAttribute(DynamicNameAspect.EXCEL_NAME_KEY, RequestAttributes.SCOPE_REQUEST);
        String fileName = String.format("%s%s", URLEncoder.encode(name, "UTF-8"), exportExcel.suffix().getValue());
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        response.setHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment;filename=" + fileName);
        write(o, response, exportExcel);
    }

    /**
     * 通用的获取ExcelWriter方法
     *
     * @param response      HttpServletResponse
     * @param exportExcel ExportExcel注解
     * @param templatePath  模板地址
     * @return ExcelWriter
     */
    @SneakyThrows
    public ExcelWriter getExcelWriter(HttpServletResponse response, ExportExcel exportExcel, String templatePath) {
        ExcelWriterBuilder writerBuilder = EasyExcel.write(response.getOutputStream())
                .registerConverter(LocalDateStringConverter.INSTANCE)
                .registerConverter(LocalDateTimeStringConverter.INSTANCE)
                .registerConverter(TimestampStringConverter.INSTANCE)
                .autoCloseStream(true)
                .excelType(exportExcel.suffix())
                .inMemory(exportExcel.inMemory());

        if (StringUtils.hasText(exportExcel.password())) {
            writerBuilder.password(exportExcel.password());
        }

        if (exportExcel.include().length != 0) {
            writerBuilder.includeColumnFiledNames(Arrays.asList(exportExcel.include()));
        }

        if (exportExcel.exclude().length != 0) {
            writerBuilder.excludeColumnFiledNames(Arrays.asList(exportExcel.include()));
        }

        if (exportExcel.writeHandler().length != 0) {
            for (Class<? extends WriteHandler> clazz : exportExcel.writeHandler()) {
                writerBuilder.registerWriteHandler(BeanUtils.instantiateClass(clazz));
            }
        }

        // 自定义注入的转换器
        registerCustomConverter(writerBuilder);

        if (exportExcel.converter().length != 0) {
            for (Class<? extends Converter> clazz : exportExcel.converter()) {
                writerBuilder.registerConverter(BeanUtils.instantiateClass(clazz));
            }
        }

        if (StringUtils.hasText(exportExcel.template())) {
            ClassPathResource classPathResource = new ClassPathResource(templatePath
                    + File.separator + exportExcel.template());
            InputStream inputStream = classPathResource.getInputStream();
            writerBuilder.withTemplate(inputStream);
        }

        return writerBuilder.build();
    }

    /**
     * 自定义注入转换器
     * 如果有需要，子类自己重写
     *
     * @param builder ExcelWriterBuilder
     */
    public void registerCustomConverter(ExcelWriterBuilder builder) {
        // do nothing
    }


    /**
     * 获取 WriteSheet 对象
     *
     * @param sheetNo           页签号
     * @param sheetName         页签名称
     * @param dataClass         数据类型
     * @param template          模板
     * @param headEnhancerClass 自定义头处理器
     * @return WriteSheet
     */
    public WriteSheet sheet(Integer sheetNo, String sheetName, Class<?> dataClass, String template,
                            Class<? extends HeadGenerator> headEnhancerClass) {
        // 头信息增强
        HeadGenerator headGenerator = null;
        if (!headEnhancerClass.isInterface()) {
            headGenerator = BeanUtils.instantiateClass(headEnhancerClass);
        }

        // 是否模板写入
        ExcelWriterSheetBuilder excelWriterSheetBuilder = StringUtils.hasText(template) ? EasyExcel.writerSheet(sheetNo)
                : EasyExcel.writerSheet(sheetNo, sheetName);
        // 自定义头信息
        if (headGenerator != null) {
            excelWriterSheetBuilder.head(headGenerator.head(dataClass));
        } else if (dataClass != null) {
            excelWriterSheetBuilder.head(dataClass);
        }

        return excelWriterSheetBuilder.build();
    }


}
