package cn.allbs.utils.cache.support;

import cn.allbs.utils.cache.listener.CacheMessage;
import cn.allbs.utils.cache.properties.CacheConfigProperties;
import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.github.benmanes.caffeine.cache.Cache;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.support.AbstractValueAdaptingCache;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 功能: 多级读取、过期策略实现 所有更新操作都基于 redis pub/sub 消息机制更新
 *
 * @author ChenQi
 * @version 1.0
 * @date 2021/3/20 下午4:13
 */
@Slf4j
public class RedisCaffeineCache extends AbstractValueAdaptingCache {

    @Getter
    private final String name;
    @Getter
    private final Cache<Object, Object> caffeineCache;

    private final String cachePrefix;

    private final long defaultExpiration;

    private final Map<String, Long> expires;

    private final String topic;

    private final Map<String, ReentrantLock> keyLockMap = new ConcurrentHashMap<>();

    private final boolean usedCaffeineCache;

    private final boolean usedRedisCache;

    private final RedisTemplate<Object, Object> redisTemplate;

    public RedisCaffeineCache(String name, RedisTemplate<Object, Object> redisTemplate,
                              Cache<Object, Object> caffeineCache, CacheConfigProperties cacheConfigProperties) {
        super(cacheConfigProperties.isCacheNullValues());
        this.name = name;
        this.caffeineCache = caffeineCache;
        this.cachePrefix = cacheConfigProperties.getCachePrefix();
        this.defaultExpiration = cacheConfigProperties.getRedis().getDefaultExpiration();
        this.expires = cacheConfigProperties.getRedis().getExpires();
        this.topic = cacheConfigProperties.getRedis().getTopic();
        this.usedCaffeineCache = cacheConfigProperties.getCaffeine().isUsedCaffeineCache();
        this.usedRedisCache = cacheConfigProperties.getRedis().isUsedRedisCache();
        this.redisTemplate = redisTemplate;
    }

    @Override
    public Object getNativeCache() {
        return this;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T get(Object key, Callable<T> valueLoader) {
        Object value = lookup(key);
        if (value != null) {
            return (T) value;
        }
        ReentrantLock lock = keyLockMap.computeIfAbsent(key.toString(), s -> {
            log.trace("create lock for key : {}", s);
            return new ReentrantLock();
        });
        lock.lock();
        try {
            value = lookup(key);
            if (value != null) {
                return (T) value;
            }
            value = valueLoader.call();
            Object storeValue = toStoreValue(value);
            put(key, storeValue);
            return (T) value;
        } catch (Exception e) {
            throw new ValueRetrievalException(key, valueLoader, e.getCause());
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void put(Object key, Object value) {
        if (!super.isAllowNullValues() && value == null) {
            this.evict(key);
            return;
        }
        if (usedRedisCache) {
            long expire = getExpire();
            if (expire > 0) {
                redisTemplate.opsForValue().set(getKey(key), toStoreValue(value), expire, TimeUnit.MILLISECONDS);
            } else {
                redisTemplate.opsForValue().set(getKey(key), toStoreValue(value));
            }
            push(new CacheMessage(this.name, key));
        }
        if (usedCaffeineCache) {
            caffeineCache.put(key, value);
        }
    }

    @Override
    public ValueWrapper putIfAbsent(Object key, Object value) {
        Object cacheKey = getKey(key);
        Object prevValue = null;
        // 考虑使用分布式锁，或者将redis的setIfAbsent改为原子性操作
        synchronized (key) {
            if (usedRedisCache) {
                prevValue = redisTemplate.opsForValue().get(getKey(key));
                if (prevValue == null) {
                    long expire = getExpire();
                    if (expire > 0) {
                        redisTemplate.opsForValue().set(cacheKey, toStoreValue(value), expire, TimeUnit.MILLISECONDS);
                    } else {
                        redisTemplate.opsForValue().set(cacheKey, toStoreValue(value));
                    }
                    push(new CacheMessage(this.name, key));
                }
            }
            if (usedCaffeineCache) {
                prevValue = caffeineCache.getIfPresent(key);
                if (prevValue == null) {
                    caffeineCache.put(key, toStoreValue(value));
                }
            }

        }
        return toValueWrapper(prevValue);
    }

    @Override
    public void evict(Object key) {
        // 先清除redis中缓存数据，然后清除caffeine中的缓存，避免短时间内如果先清除caffeine缓存后其他请求会再从redis里加载到caffeine中
        if (usedRedisCache) {
            redisTemplate.delete(getKey(key));
            push(new CacheMessage(this.name, key));
        }
        if (usedCaffeineCache) {
            caffeineCache.invalidate(key);
        }
    }

    @Override
    public void clear() {
        if (usedRedisCache) {
            // 先清除redis中缓存数据，然后清除caffeine中的缓存，避免短时间内如果先清除caffeine缓存后其他请求会再从redis里加载到caffeine中
            Set<Object> keys = redisTemplate.keys(this.name.concat(":*"));
            if (!CollUtil.isEmpty(keys)) {
                redisTemplate.delete(keys);
            }
            push(new CacheMessage(this.name, null));
        }
        if (usedCaffeineCache) {
            caffeineCache.invalidateAll();
        }
    }

    @Override
    protected Object lookup(Object key) {
        Object cacheKey = getKey(key);
        Object value = null;
        if (usedCaffeineCache) {
            value = caffeineCache.getIfPresent(key);
            if (value != null) {
                log.debug("get cache from caffeine, the key is : {}", cacheKey);
                return value;
            }
        }
        if (usedRedisCache) {
            // 避免自动一个 RedisTemplate 覆盖失效
            redisTemplate.setKeySerializer(new StringRedisSerializer());
            value = redisTemplate.opsForValue().get(cacheKey);
            if (value != null) {
                log.debug("get cache from redis and put in caffeine, the key is : {}", cacheKey);
                caffeineCache.put(key, value);
            }
        }
        return value;
    }

    private Object getKey(Object key) {
        return this.name.concat(":").concat(
                StrUtil.isEmpty(cachePrefix) ? key.toString() : cachePrefix.concat(":").concat(key.toString()));
    }

    private long getExpire() {
        Long cacheNameExpire = expires.get(this.name);
        return cacheNameExpire == null ? defaultExpiration : cacheNameExpire;
    }

    /**
     * 缓存变更时通知其他节点清理本地缓存
     *
     * @param message
     */
    private void push(CacheMessage message) {
        redisTemplate.convertAndSend(topic, message);
    }

    /**
     * 清理本地缓存
     *
     * @param key
     */
    public void clearLocal(Object key) {
        log.debug("clear local cache, the key is : {}", key);
        if (key == null) {
            caffeineCache.invalidateAll();
        } else {
            caffeineCache.invalidate(key);
        }
    }

}