package cn.allbs.utils.cache;

import cn.allbs.utils.cache.listener.CacheMessageListener;
import cn.allbs.utils.cache.properties.CacheConfigProperties;
import cn.allbs.utils.cache.support.RedisCaffeineCacheManager;
import com.fasterxml.jackson.annotation.JsonTypeInfo.As;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectMapper.DefaultTyping;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.listener.ChannelTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

/**
 * 功能:
 *
 * @author ChenQi
 * @version 1.0
 * @date 2021/3/20 下午4:21
 */
@Configuration(proxyBeanMethods = false)
@AutoConfigureAfter(RedisAutoConfiguration.class)
@EnableConfigurationProperties(CacheConfigProperties.class)
@Slf4j
public class MultilevelCacheAutoConfiguration {

    @Bean
    @ConditionalOnBean(RedisTemplate.class)
    public RedisCaffeineCacheManager cacheManager(
            CacheConfigProperties cacheConfigProperties,
            RedisTemplate<Object, Object> redisTemplate) {
        return new RedisCaffeineCacheManager(cacheConfigProperties, redisTemplate);
    }

    @Bean
    @ConditionalOnMissingBean(RedisSerializer.class)
    public RedisSerializer<Object> redisSerializer(ObjectProvider<ObjectMapper> objectProvider) {
        // jackson findAndRegisterModules，use copy
        ObjectMapper objectMapper = objectProvider.getIfAvailable(ObjectMapper::new).copy();
        // findAndRegisterModules
        objectMapper.findAndRegisterModules();
        // class type info to json
        GenericJackson2JsonRedisSerializer.registerNullValueSerializer(objectMapper, null);
        objectMapper.activateDefaultTyping(objectMapper.getPolymorphicTypeValidator(), DefaultTyping.NON_FINAL, As.PROPERTY);
        return new GenericJackson2JsonRedisSerializer(objectMapper);
    }

    @Bean
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory, RedisSerializer<Object> redisSerializer) {
        RedisTemplate<Object, Object> redisTemplate = new RedisTemplate<>();
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setHashKeySerializer(new StringRedisSerializer());
        redisTemplate.setValueSerializer(redisSerializer);
        redisTemplate.setHashValueSerializer(redisSerializer);
        redisTemplate.setConnectionFactory(redisConnectionFactory);
        return redisTemplate;
    }

    /**
     * 两级缓存topic监听
     *
     * @param cacheConfigProperties
     * @param redisTemplate
     * @param redisCaffeineCacheManager
     * @return
     */
    @Bean
    public RedisMessageListenerContainer redisMessageListenerContainer(
            CacheConfigProperties cacheConfigProperties,
            RedisTemplate<Object, Object> redisTemplate,
            RedisCaffeineCacheManager redisCaffeineCacheManager) {
        // 二级缓存监听并操作caffeine
        RedisMessageListenerContainer redisMessageListenerContainer = new RedisMessageListenerContainer();
        redisMessageListenerContainer.setConnectionFactory(redisTemplate.getConnectionFactory());
        CacheMessageListener cacheMessageListener = new CacheMessageListener(redisTemplate,
                redisCaffeineCacheManager);
        redisMessageListenerContainer.addMessageListener(cacheMessageListener,
                new ChannelTopic(cacheConfigProperties.getRedis().getTopic()));
        return redisMessageListenerContainer;
    }
}
