package cn.allbs.core.config;

import cn.allbs.common.constant.StringPool;
import cn.allbs.common.enums.PropertiesEnum;
import cn.allbs.common.log.LogPrintStream;
import cn.allbs.common.utils.SystemUtil;
import cn.hutool.core.util.StrUtil;
import io.netty.resolver.HostsFileParser;
import org.springframework.boot.web.context.WebServerApplicationContext;
import org.springframework.boot.web.context.WebServerInitializedEvent;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;
import org.springframework.scheduling.annotation.Async;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.context.support.StandardServletEnvironment;

import java.net.Inet4Address;
import java.util.Map;

import static cn.allbs.common.constant.PropertiesConstant.SPRING_APPLICATION_NAME;


/**
 * @author ChenQi
 */
@Configuration(proxyBeanMethods = false)
public class StartedEventListener {

    @Async
    @Order(Ordered.LOWEST_PRECEDENCE - 1)
    @EventListener(WebServerInitializedEvent.class)
    public void afterStart(WebServerInitializedEvent event) {
        WebServerApplicationContext context = event.getApplicationContext();
        Environment environment = context.getEnvironment();
        String appName = environment.getRequiredProperty(SPRING_APPLICATION_NAME);
        int localPort = event.getWebServer().getPort();
        String profile = StringUtils.arrayToCommaDelimitedString(environment.getActiveProfiles());
        System.err.println("-----------------------------系统信息-------------------------------");
        System.err.printf("当前服务[%s]启动完成,使用的端口:[%d],环境变量:[%s] %n", appName, localPort, profile);
        // 如果有 swagger，打印开发阶段的 swagger ui 地址
        // 判断是否是在prod环境下
        if (!"prod".equals(profile)) {
            if (ClassUtils.isPresent("springfox.documentation.spring.web.plugins.Docket", null)) {
                System.err.printf("当前服务swagger文档访问路径为[http://localhost:%s/doc.html]%n", localPort);
            } else {
                System.err.printf("当前服务uri基础路径为[http://localhost:%s%n]", localPort);
            }
        }

        // 获取系统变量
        Map<String, Object> environmentMap = ((StandardServletEnvironment) environment).getSystemEnvironment();

        // 获取系统属性
        Map<String, Object> systemProperties = ((StandardServletEnvironment) environment).getSystemProperties();

        // 获取本地host文件
        Map<String, Inet4Address> stringInet4AddressMap = HostsFileParser.parseSilently().inet4Entries();

        // 打印所需要的信息
        for (PropertiesEnum properties : PropertiesEnum.values()) {
            switch (properties.getType()) {
                case SYSTEM:
                    if (systemProperties.containsKey(properties.getProperties())) {
                        System.err.printf(properties.getDesc(), environment.getRequiredProperty(properties.getProperties()));
                    }
                    break;
                case IP:
                    if (environment.containsProperty(properties.getProperties())) {
                        String ip = environment.getRequiredProperty(properties.getProperties());
                        if (stringInet4AddressMap.containsKey(ip)) {
                            ip = stringInet4AddressMap.get(ip).getHostAddress();
                        }
                        System.err.printf(properties.getDesc(), ip);
                    }
                    break;
                case URL:
                    if (environment.containsProperty(properties.getProperties())) {
                        String url = environment.getRequiredProperty(properties.getProperties());
                        String server = StrUtil.subBetween(url, StringPool.PRE_SLASH, StringPool.COLON);
                        if (stringInet4AddressMap.containsKey(server)) {
                            String hostServer = stringInet4AddressMap.get(server).getHostAddress();
                            url = url.replace(server, hostServer);
                        }
                        System.err.printf(properties.getDesc(), url);
                    }
                    break;
                case TRUE_OR_FALSE_TRANSLATE:
                    if (environment.containsProperty(properties.getProperties())) {
                        System.err.printf(properties.getDesc(), StringPool.TRUE.equals(environment.getRequiredProperty(properties.getProperties())) ? "是" : "否");
                    }
                    break;
                default:
                    if (environment.containsProperty(properties.getProperties())) {
                        System.err.printf(properties.getDesc(), environment.getRequiredProperty(properties.getProperties()));
                    }
            }
        }

        System.err.println("-----------------------------系统信息-------------------------------");
        // linux 上将全部的 System.err 和 System.out 替换为log
        if (SystemUtil.isLinux()) {
            System.setOut(LogPrintStream.log(false));
        }
    }

}
