package cn.allbs.core.handle;

import cn.allbs.common.constant.AllbsCoreConstants;
import cn.allbs.common.enums.ErrorCodeEnum;
import cn.allbs.common.exception.ParamCheckException;
import cn.allbs.common.utils.R;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.server.ServerErrorException;
import org.springframework.web.servlet.NoHandlerFoundException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * 异常统一返回
 *
 * @author chenqi
 * @version 1.0
 * @since 2020/8/7 9:42
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandlerResolver {

    /**
     * Exception
     *
     * @return R
     */
    @ExceptionHandler(value = Exception.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> unKnowExceptionHandler(Exception e) {
        String timeNow = "EX-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * RuntimeException
     *
     * @param e RuntimeException
     * @return R
     */
    @ExceptionHandler(value = RuntimeException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> runtimeExceptionHandler(RuntimeException e) {
        String timeNow = "RE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.INTERNAL_SERVER_ERROR, timeNow);
    }

    /**
     * 空指针异常
     */
    @ExceptionHandler(NullPointerException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> nullPointerExceptionHandler(NullPointerException e) {
        String timeNow = "NPE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.NULL_POINTER_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 类型转换异常
     */
    @ExceptionHandler(ClassCastException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> classCastExceptionHandler(ClassCastException e) {
        String timeNow = "CCE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.CAST_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * IO异常
     */
    @ExceptionHandler(IOException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> iOExceptionHandler(IOException e) {
        String timeNow = "IOE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.IO_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 未知方法异常
     */
    @ExceptionHandler(NoSuchMethodException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> noSuchMethodExceptionHandler(NoSuchMethodException e) {
        String timeNow = "NSUE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.NO_SUCH_METHOD_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 数组越界异常
     */
    @ExceptionHandler(IndexOutOfBoundsException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> indexOutOfBoundsExceptionHandler(IndexOutOfBoundsException e) {
        String timeNow = "IOOBE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.INDEX_OUT_OF_BOUND_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 400错误
     */
    @ExceptionHandler(value = HttpMessageNotReadableException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R<ErrorCodeEnum> requestNotReadable(HttpMessageNotReadableException e) {
        String timeNow = "HMNRE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.HTTP_MESSAGE_NOT_READABLE_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 400错误 类型不匹配
     */
    @ExceptionHandler({TypeMismatchException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R<ErrorCodeEnum> requestTypeMismatch(TypeMismatchException e) {
        String timeNow = "TME-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.TYPE_MISMATCH_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 400错误 缺少参数
     */
    @ExceptionHandler({MissingServletRequestParameterException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R<ErrorCodeEnum> requestMissingServletRequest(MissingServletRequestParameterException e) {
        String timeNow = "MSRPE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.MISSING_SERVLET_REQUEST_PARAMETER_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 参数错误
     *
     * @return R
     */
    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R<ErrorCodeEnum> methodArgumentNotValidExceptionHandler(MethodArgumentNotValidException e) {
        String timeNow = "MANVE-" + System.currentTimeMillis();
        List<String> list = new ArrayList<>();
        // 从异常对象中拿到ObjectError对象
        if (!e.getBindingResult().getAllErrors().isEmpty()) {
            for (ObjectError error : e.getBindingResult().getAllErrors()) {
                list.add(error.getDefaultMessage());
            }
        }
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + list, e.getCause());
        return R.exception(ErrorCodeEnum.METHOD_ARGUMENT_NOT_VALID_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 处理自定义的异常
     *
     * @param ex
     * @return
     */
    @ExceptionHandler(ParamCheckException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public R<ErrorCodeEnum> resolveMyException(ParamCheckException ex) {
        String timeNow = "PCE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + ex.getLocalizedMessage(), ex.getCause());
        return R.exception(ErrorCodeEnum.METHOD_ARGUMENT_NOT_VALID_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 405错误
     */
    @ExceptionHandler({HttpRequestMethodNotSupportedException.class})
    @ResponseStatus(HttpStatus.METHOD_NOT_ALLOWED)
    public R<ErrorCodeEnum> request405(HttpRequestMethodNotSupportedException e) {
        String timeNow = "HRMNSE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.HTTP_REQUEST_METHOD_NOT_SUPPORTED_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 406错误
     */

    @ExceptionHandler({HttpMediaTypeNotAcceptableException.class})
    @ResponseStatus(HttpStatus.NOT_ACCEPTABLE)
    public R<ErrorCodeEnum> request406(HttpMediaTypeNotAcceptableException e) {
        String timeNow = "HMTNAE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.HTTP_MEDIA_TYPE_NOT_ACCEPTABLE_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 500错误
     */
    @ExceptionHandler(ConversionNotSupportedException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> server500(ConversionNotSupportedException e) {
        String timeNow = "CNSE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.CONVERSION_NOT_ACCEPTABLE_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 500错误
     *
     * @param e
     * @return
     */
    @ExceptionHandler(HttpMessageNotWritableException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> server500(HttpMessageNotWritableException e) {
        String timeNow = "HMNWE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.CONVERSION_NOT_ACCEPTABLE_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * 媒体格式不支持
     *
     * @return R
     */
    @ExceptionHandler(value = HttpMediaTypeNotSupportedException.class)
    @ResponseStatus(HttpStatus.UNSUPPORTED_MEDIA_TYPE)
    public R<ErrorCodeEnum> httpMediaTypeNotSupportedExceptionHandler(HttpMediaTypeNotSupportedException e) {
        String timeNow = "HMTNSE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.HTTP_MEDIA_TYPE_NOT_SUPPORTED_EXCEPTION_RESPONSE, timeNow);
    }


    /**
     * 404
     */
    @ExceptionHandler(value = NoHandlerFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public R<ErrorCodeEnum> notFoundException(NoHandlerFoundException e) {
        String timeNow = "NHFE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.NO_HANDLER_FOUND_EXCEPTION_RESPONSE, timeNow);
    }

    /**
     * ServerErrorException
     *
     * @param e
     * @return
     */
    @ExceptionHandler(value = ServerErrorException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public R<ErrorCodeEnum> serverErrorExceptionHandler(ServerErrorException e) {
        String timeNow = "SEE-" + System.currentTimeMillis();
        log.error(AllbsCoreConstants.ALLBS_TIP + "异常拦截[" + timeNow + "]:" + e.getLocalizedMessage(), e.getCause());
        return R.exception(ErrorCodeEnum.INTERNAL_SERVER_ERROR, timeNow);
    }
}
