package cn.allbs.cache.sequence;

import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.support.atomic.RedisAtomicLong;

import java.time.Instant;
import java.util.concurrent.TimeUnit;

/**
 * @author ChenQi
 */
public class SequenceManage {

    private final RedisTemplate<Object, Object> redisTemplate;

    public SequenceManage(RedisTemplate<Object, Object> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    /**
     * 获取链接工厂
     */
    public RedisConnectionFactory getConnectionFactory() {
        return redisTemplate.getConnectionFactory();
    }

    /**
     * 自增数
     *
     * @param key redis储存发号器的时间
     * @return 自增序号
     */
    public long increment(String key) {
        RedisAtomicLong redisAtomicLong = new RedisAtomicLong(key, getConnectionFactory());
        return redisAtomicLong.incrementAndGet();
    }

    /**
     * 自增数（带过期时间）
     *
     * @param key      redis 储存发号器的key
     * @param time     指定过期时间
     * @param timeUnit 过期时间单位
     * @return 自增数
     */
    public long increment(String key, long time, TimeUnit timeUnit) {
        RedisAtomicLong redisAtomicLong = new RedisAtomicLong(key, getConnectionFactory());
        redisAtomicLong.expire(time, timeUnit);
        return redisAtomicLong.incrementAndGet();
    }

    /**
     * 自增数（带过期时间）
     *
     * @param key      redis 储存发号器的key
     * @param expireAt 过期所在的时间
     * @return 自增数（带过期时间）
     */
    public long increment(String key, Instant expireAt) {
        RedisAtomicLong redisAtomicLong = new RedisAtomicLong(key, getConnectionFactory());
        redisAtomicLong.expireAt(expireAt);
        return redisAtomicLong.incrementAndGet();
    }

    /**
     * 自增数（带过期时间和步长）
     *
     * @param key       redis 储存发号器的key
     * @param increment 步长
     * @param time      过期时间
     * @param timeUnit  过期时间单位
     * @return 自增数（带过期时间和步长）
     */
    public long increment(String key, int increment, long time, TimeUnit timeUnit) {
        RedisAtomicLong redisAtomicLong = new RedisAtomicLong(key, getConnectionFactory());
        redisAtomicLong.expire(time, timeUnit);
        return redisAtomicLong.addAndGet(increment);
    }

    /**
     * 自增数不带过期时间和步长
     *
     * @param key       redis 储存自增序列的key
     * @param increment 步长
     * @return 自增数不带过期时间和步长
     */
    public long increment(String key, int increment) {
        RedisAtomicLong redisAtomicLong = new RedisAtomicLong(key, getConnectionFactory());
        return redisAtomicLong.addAndGet(increment);
    }

    /**
     * 初始值初始化
     *
     * @param key          redis的key
     * @param initialValue 初始化的值
     */
    public void initIncrement(String key, int initialValue) {
        RedisAtomicLong redisAtomicLong = new RedisAtomicLong(key, getConnectionFactory(), initialValue);
    }
}
